/* KinokoController.cc */
/* Created by Enomoto Sanshiro on 7 December 2000. */
/* Last updated by Enomoto Sanshiro on 9 August 2007. */


#include <string>
#include <iostream>
#include <sstream>
#include "KinokoPlatform.hh"
#include "KinokoControlInputWatcher.hh"
#include "KinokoController.hh"

using namespace std;


TKinokoController::TKinokoController(istream& InputStream, ostream& OutputStream, TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry)
: _InputStream(InputStream), _OutputStream(OutputStream)
{
    _EventEmitter = EventEmitter;
    _Registry = Registry;

    _InputWatcher = new TKinokoControlInputWatcher(_InputStream, ';');
    _InputWatcher->Start();
}

TKinokoController::~TKinokoController()
{
    delete _InputWatcher;
}

int TKinokoController::ProcessInput(void)
{
    if (_InputStream.eof() || (! _InputWatcher->IsDataAvailable())) {
	return 0;
    }

    const string& Line = _InputWatcher->Line();
    ProcessCommand(Line);

    _InputWatcher->Flush();
    
    return 1;
}

void TKinokoController::ProcessCommand(const string& Line)
{
    istringstream InputStream(Line);

    string Command, Qualifier, ArgumentString;
    vector<string> ArgumentNameList, ArgumentValueList;

    getline(InputStream >> ws >> Command >> ws >> Qualifier >> ws, ArgumentString);

    while (! ArgumentString.empty()) {
	int NameLength = ArgumentString.find_first_of('=');
	int ArgumentLength = ArgumentString.find_first_of('&');
	int ValueLength = ArgumentLength - NameLength - 1;
	if (NameLength > 0) {
	    string Name = ArgumentString.substr(0, NameLength);
	    string Value = ArgumentString.substr(NameLength + 1, ValueLength);
	    ArgumentNameList.push_back(Name);
	    ArgumentValueList.push_back(DecodeEscape(Value));
	}
	if (ArgumentLength > 0) {
	    ArgumentString.erase(0, ArgumentLength + 1);
	}
	else {
	    ArgumentString.erase(0, ArgumentLength);
	}
    }

    if (Command == ".invoke") {
	if (! Qualifier.empty()) {
	    _EventEmitter->EmitEvent(Qualifier, ArgumentValueList);
	}
    }
    else if (Command == ".update") {
	for (unsigned i = 0; i < ArgumentNameList.size(); i++) {
	    _Registry->SetValue(
		Qualifier + "/" + ArgumentNameList[i], 
		ArgumentValueList[i]
	    );
	}
    }
    else {
	//... error: unknown method
    }
}

void TKinokoController::Open(const string& FileName)
{
    _OutputStream << ".openControlPanel " << FileName << ";" << endl;
    _OutputStream << ".start;" << endl;
}

void TKinokoController::ChangeState(const string& StateName)
{
    _OutputStream << ".changeState " << StateName << ";" << endl;
}

void TKinokoController::SetWidgetValue(const string& WidgetName, const string& Value)
{
    _OutputStream << ".setWidgetValue ";
    _OutputStream << WidgetName << " " << Value;
    _OutputStream<< ";" << endl;
}

void TKinokoController::SetWidgetAttribute(const string& WidgetName, const string& AttributeName, const string& Value)
{
    _OutputStream << ".setWidgetAttribute ";
    _OutputStream << WidgetName << " " << AttributeName << " " << Value;
    _OutputStream<< ";" << endl;
}

void TKinokoController::ExecuteAction(const string& ActionName, const vector<string>& ArgumentList)
{
    _OutputStream << ".executeAction " << ActionName << " ";
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	_OutputStream << ArgumentList[i] << ", ";
    }
    _OutputStream << ";" << endl;
}

void TKinokoController::OpenPopup(const string& Type, const string& ActionList, const string& Message)
{
    _OutputStream << ".openPopup ";
    _OutputStream << Type << ", " << ActionList << ", " << Message;
    _OutputStream << ";" << endl;
}

string TKinokoController::OpenQueryPopup(const string& Type, const string& ActionList, const string& Message)
{
    _OutputStream << ".openQueryPopup ";
    _OutputStream << Type << ", " << ActionList << ", " << Message;
    _OutputStream << ";" << endl;

    string Reply;
    while (1) {
	if (_InputWatcher->WaitData()) {
	    Reply = _InputWatcher->Line();
	    _InputWatcher->Flush();

	    if ((! Reply.empty()) && (Reply[0] == '.')) {
		ProcessCommand(Reply);
	    }
	    else {
		break;
	    }
	}
    }

    return Reply;
}

void TKinokoController::Save(const string& FileName)
{
    _OutputStream << ".saveWidgetValues " << FileName << ";" << endl;
}

void TKinokoController::Load(const string& FileName)
{
    _OutputStream << ".loadWidgetValues " << FileName << ";" << endl;
}

void TKinokoController::LoadScript(const string& FileName)
{
    _OutputStream << ".loadControlScript " << FileName << ";" << endl;
}

void TKinokoController::Quit(void)
{
    _InputWatcher->Stop();

    _OutputStream << ".stop;" << endl;
    _OutputStream << ".quit;" << endl;

    _InputWatcher->Join();
}

string TKinokoController::DecodeEscape(const string& Value)
{
    string Result;
    bool IsEscaped = false;
    for (unsigned i = 0; i < Value.size(); i++) {
	if (! IsEscaped && (Value[i] == '%')) {
	    IsEscaped = true;
	    continue;
	}
	if (! IsEscaped) {
	    Result += Value[i];
	    continue;
	}

	switch (Value[i]) {
          case 'p':
	    Result += '%';
	    break;
          case 'a':
	    Result += '&';
	    break;
          case 'e':
	    Result += '=';
	    break;
          case 's':
	    Result += ';';
	    break;
          default:
	    Result += Value[i];
	}

	IsEscaped = false;
    }

    return Result;
}
