/* KinokoDataSection.cc */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 12 July 2007. */


#include <iostream>
#include <string>
#include "ParaTokenizer.hh"
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"

#include "KinokoNestedDataSection.hh"
#include "KinokoTaggedDataSection.hh"
#include "KinokoIndexedDataSection.hh"
#include "KinokoBlockDataSection.hh"

using namespace std;


int TKinokoDataSection::AutomaticSectionIdAssignmentBase(void)
{
    return 0x4000; // 16384, half of signed 16bit
}

int TKinokoDataSection::GenerateNextSectionId(void)
{
    static int SectionCounts = 0;
    SectionCounts++;
    
    return AutomaticSectionIdAssignmentBase() + SectionCounts;
}

TKinokoDataSection::TKinokoDataSection(TKinokoDataSource* DataSource, const string& SectionName)
{
    _DataSource = DataSource;
    _SectionName = SectionName;

    _SectionId = SectionId_Unknown;
    _Parent = 0;

    _DataSource->DataSectionCleaner().RegisterDataSection(this);
}

TKinokoDataSection::TKinokoDataSection(TKinokoDataSource* DataSource, const string& SectionName, int SectionId)
{
    _DataSource = DataSource;
    _SectionName = SectionName;

    _SectionId = SectionId;
    _Parent = 0;

    _DataSource->DataSectionCleaner().RegisterDataSection(this);
}

TKinokoDataSection::~TKinokoDataSection()
{
}

void TKinokoDataSection::SetSectionId(int SectionId)
{
    _SectionId = SectionId;
}

void TKinokoDataSection::SetParent(TKinokoDataSection* Parent)
{
    _Parent = Parent;
}

string TKinokoDataSection::SectionNameOf(const string& SectionPath)
{
    int Length = SectionPath.find_first_of(':');
    string SectionName = SectionPath.substr(0, Length);

    return SectionName;
}

string TKinokoDataSection::SubSectionPathOf(const string& SectionPath)
{
    string SubSectionPath = SectionPath;
    int Length = SubSectionPath.find_first_of(':');
    SubSectionPath.erase(0, Length + 1);

    return SubSectionPath;
}



TKinokoDataSectionFactory* TKinokoDataSectionFactory::_Instance = 0;

TKinokoDataSectionFactory::TKinokoDataSectionFactory(void)
{
}

TKinokoDataSectionFactory::~TKinokoDataSectionFactory()
{
}

TKinokoDataSectionFactory* TKinokoDataSectionFactory::GetInstance(void)
{
    if (_Instance == 0) {
	_Instance = new TKinokoDataSectionFactory();
    }

    return _Instance;
}

TKinokoDataSection* TKinokoDataSectionFactory::ReadFrom(TParaTokenizer& Tokenizer, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    string DataSectionName;
    long DataSectionId;
    string DataSectionType;
    try {
	Tokenizer.Next().MustBe("section");
	DataSectionName = Tokenizer.Next().RemoveQuotation().AsString();
	Tokenizer.Next().MustBe("<");
	DataSectionId = Tokenizer.Next().AsLong();
	Tokenizer.Next().MustBe(">");
	Tokenizer.Next().MustBe(":");
	DataSectionType = Tokenizer.Next().AsString();
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    "TKinokoDataSectionFactory::ReadFrom()",
	    "script exception: " + e.Message()
	);
    }
	
    TKinokoDataSection* DataSection = 0;
    if (DataSectionType == "nested") {
	DataSection = new TKinokoNestedDataSection(
	    DataSource, DataSectionName, DataSectionId
	);
    }
    else if (DataSectionType == "tagged") {
	DataSection = new TKinokoTaggedDataSection(
	    DataSource, DataSectionName, DataSectionId
	);
    }
    else if (DataSectionType == "indexed") {
	DataSection = new TKinokoIndexedDataSection(
	    DataSource, DataSectionName, DataSectionId
	);
    }
    else if (DataSectionType == "block") {
	DataSection = new TKinokoBlockDataSection(
	    DataSource, DataSectionName, DataSectionId
	);
    }
    else {
	throw TKinokoException(
	    "TKinokoDataSectionFactory::ReadFrom()",
	    "unknown data section type: " + DataSectionType
	);
    }
    
    try {
	DataSection->ReadFrom(Tokenizer);
    }
    catch (TKinokoException &e) {
	try {
	    delete DataSection;
	}
	catch (TKinokoException &e) {
	    ;
	}

	throw TKinokoException(
	    "TKinokoDataSectionFactory::ReadFrom()",
	    e.Message()
	);
    }

    return DataSection;
}



TKinokoDataSectionFormatter::TKinokoDataSectionFormatter(TKinokoDataSection* DataSection) throw(TKinokoException)
{
    _DataSourceId = DataSection->DataSource()->DataSourceId();
    _SectionId = DataSection->SectionId();
    
    if (_SectionId == 0) {
	throw TKinokoException(
	    "TKinokoDataSectionFormatter::TKinokoDataSectionFormatter()",
	    "invalid section ID (internal): " + DataSection->FullSectionName()
	);
    }
}

TKinokoDataSectionFormatter::~TKinokoDataSectionFormatter()
{
}



TKinokoDataSectionScanner::TKinokoDataSectionScanner(void)
{
}

TKinokoDataSectionScanner::~TKinokoDataSectionScanner()
{
}



TKinokoDataDescriptorAttributeList::TKinokoDataDescriptorAttributeList(void)
{
}

TKinokoDataDescriptorAttributeList::~TKinokoDataDescriptorAttributeList()
{
}

void TKinokoDataDescriptorAttributeList::AddAttribute(const string& Name, const string& Value)
{
    _AttributeList.push_back(make_pair(Name, Value));
}

bool TKinokoDataDescriptorAttributeList::GetAttributeOf(const std::string& Name, std::string& Value)
{
    bool Result = false;
    for (unsigned i = 0; i < _AttributeList.size(); i++) {
	if (_AttributeList[i].first == Name) {
	    Value = _AttributeList[i].second;
	    Result = true;
	    break;
	}
    }

    return Result;
}

void TKinokoDataDescriptorAttributeList::ReadAttribute(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    string AttributeName;
    string AttributeValue;
    try {
	Tokenizer.Next().MustBe("attribute");
	AttributeName = Tokenizer.Next().AsString();
	Tokenizer.Next().MustBe("=");
	AttributeValue = Tokenizer.Next().RemoveQuotation().AsString();
	Tokenizer.Next().MustBe(";");
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    "TKinokoDataSource::ParseAttribute()",
	    "script exception: " + e.Message()
	);
    }

    AddAttribute(AttributeName, AttributeValue);
}

void TKinokoDataDescriptorAttributeList::WriteAttributeList(ostream& os, const string& Indent)
{
    for (unsigned i = 0; i < _AttributeList.size(); i++) {
	os << Indent << "attribute " << _AttributeList[i].first;
	os << " = \"" <<  _AttributeList[i].second << "\";";
	os << endl;
    }    

    if (_AttributeList.size() > 0) {
	os << endl;
    }
}
