/* KinokoDataSection.hh */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 12 July 2007. */


#ifndef __KinokoDataSection_hh__
#define __KinokoDataSection_hh__


#include <iostream>
#include <string>
#include <vector>
#include "RoomConfig.hh"
#include "ParaTokenizer.hh"
#include "KinokoDefs.hh"
#include "KinokoDataStreamFormatter.hh"

class TKinokoDataSource;


class TKinokoDataDescriptorAttributeList {
  public:
    TKinokoDataDescriptorAttributeList(void);
    virtual ~TKinokoDataDescriptorAttributeList();
    virtual void AddAttribute(const std::string& Name, const std::string& Value);
    virtual bool GetAttributeOf(const std::string& Name, std::string& Value);
  protected:
    void ReadAttribute(TParaTokenizer& Tokenizer) throw(TKinokoException);
    void WriteAttributeList(std::ostream& os, const std::string& Indent);  
  protected:
    std::vector<std::pair<std::string, std::string> > _AttributeList;
};


class TKinokoDataSection: public TKinokoDataDescriptorAttributeList {
  public:
    TKinokoDataSection(TKinokoDataSource* DataSource, const std::string& SectionName);
    TKinokoDataSection(TKinokoDataSource* DataSource, const std::string& SectionName, int SectionId);
    virtual ~TKinokoDataSection();
    virtual void SetSectionId(int SectionId);
    virtual void SetParent(TKinokoDataSection* Parent);
    static int AutomaticSectionIdAssignmentBase(void);
    static int GenerateNextSectionId(void);
  public:
    virtual int SectionType(void) const = 0;
    inline int SectionId(void) const;
    inline const std::string& SectionName(void) const;
    inline std::string FullSectionName(void) const;
    inline TKinokoDataSource* DataSource(void) const;
    inline TKinokoDataSection* Parent(void) const;
    virtual void ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException) = 0;
    virtual void WriteTo(std::ostream& os, const std::string& Indent) = 0;
  public:
    static std::string SectionNameOf(const std::string& SectionPath);
    static std::string SubSectionPathOf(const std::string& SectionPath);
  protected:
    int _SectionId;
    std::string _SectionName;
    TKinokoDataSource* _DataSource;
    TKinokoDataSection* _Parent;
  private:
    static int _NextSectionId;
  public:
    enum TSectionType {
	SectionType_Block,
	SectionType_Indexed,
	SectionType_Tagged,
	SectionType_Nested,
	_NumberOfSectionTypes
    };
    enum TSpecialSectionId {
        SectionId_Unknown = -1,
	_NumberOfSpecialSectionIds
    };
    enum TElementType {
	ElementType_Unknown,
	ElementType_Integer,
	ElementType_Float,
	ElementType_Time,
	_NumberOfElementTypes
    };
};


class TKinokoDataSectionFactory {
  protected:
    TKinokoDataSectionFactory(void);
  public:
    virtual ~TKinokoDataSectionFactory();
    static TKinokoDataSectionFactory* GetInstance(void);
    virtual TKinokoDataSection* ReadFrom(TParaTokenizer& Tokenizer, TKinokoDataSource* DataSource) throw(TKinokoException);
  protected:
    static TKinokoDataSectionFactory* _Instance;
};


class TKinokoDataSectionFormatter {
  public:
    TKinokoDataSectionFormatter(TKinokoDataSection* DataSection) throw(TKinokoException);
    virtual ~TKinokoDataSectionFormatter();
    static inline int HeaderSize(void);
    static inline int PacketSizeFor(int DataSize);
    static inline void* DataAreaOf(void* Buffer);
    inline void WriteHeaderTo(void* Buffer, int DataSize) const;
  protected:
    static inline void PutElement(void* Buffer, int BitOffset, int BitWidth, int Value);
  protected:
    long _DataSourceId;
    int _SectionId;
  public:
    enum {
	Offset_DataSourceId = TKinokoDataStreamFormatter::Offset_DataSourceId,
	Offset_PacketType = TKinokoDataStreamFormatter::Offset_PacketType,
	Offset_DataAreaSize = TKinokoDataStreamFormatter::Offset_Contents,
	Offset_DataArea
    };
    enum {
	_PacketType = TKinokoDataStreamFormatter::PacketType_DataSection
    };
};


class TKinokoDataSectionScanner {
  public:
    TKinokoDataSectionScanner(void);
    virtual ~TKinokoDataSectionScanner();
    static inline int SectionIdOf(void* Buffer);
    static inline int PacketSizeOf(void* Buffer);
    static inline int DataSizeOf(void* Buffer);
    static inline void* DataAreaOf(void* Buffer);
    static inline int DataWordOf(void* Buffer, int WordIndex);
  protected:
    static inline void GetElement(void* Buffer, int BitOffset, int BitWidth, int& Value);
  public:
    enum {
	Offset_PacketType = TKinokoDataSectionFormatter::Offset_PacketType,
	Offset_DataAreaSize = TKinokoDataSectionFormatter::Offset_DataAreaSize,
	Offset_DataArea = TKinokoDataSectionFormatter::Offset_DataArea
    };
    enum {
	_SectionIdMask = ~TKinokoDataStreamFormatter::PacketTypeMask
    };
};



inline int TKinokoDataSection::SectionId(void) const
{
    return _SectionId;
}

inline const std::string& TKinokoDataSection::SectionName(void) const
{
    return _SectionName;
}

inline std::string TKinokoDataSection::FullSectionName(void) const
{
    if (_Parent == 0) {
	return _SectionName;
    }
    else {
	return _Parent->FullSectionName() + ":" + _SectionName;
    }
}

inline TKinokoDataSource* TKinokoDataSection::DataSource(void) const
{
    return _DataSource;
}

inline TKinokoDataSection* TKinokoDataSection::Parent(void) const
{
    return _Parent;
}



inline int TKinokoDataSectionFormatter::HeaderSize(void)
{
    return sizeof(U32bit) * Offset_DataArea;
}

inline int TKinokoDataSectionFormatter::PacketSizeFor(int DataSize)
{
    return sizeof(U32bit) * Offset_DataArea + ((DataSize - 1) / 4 + 1) * 4;
}

inline void* TKinokoDataSectionFormatter::DataAreaOf(void* Buffer)
{
    return (U32bit*) Buffer + Offset_DataArea;
}

inline void TKinokoDataSectionFormatter::WriteHeaderTo(void* Buffer, int DataSize) const
{
    ((U32bit*) Buffer)[Offset_DataSourceId] = _DataSourceId;
    ((U32bit*) Buffer)[Offset_PacketType] = _PacketType | _SectionId;
    ((U32bit*) Buffer)[Offset_DataAreaSize] = DataSize;
}

inline void TKinokoDataSectionFormatter::PutElement(void* Buffer, int BitOffset, int BitWidth, int Value)
{
    U32bit* DataBlock = (U32bit*) Buffer + Offset_DataArea;

    unsigned WordIndex = BitOffset >> 5;    // divide by 32-bit
    unsigned BitIndex = BitOffset & 0x001f; // mod by 32-bit
    int CarryOverWidth = BitIndex + BitWidth - 32;

    if (BitWidth <= 0) {
	;
    }
    else if ((BitWidth == 32) && (CarryOverWidth == 0)) {
	DataBlock[WordIndex] = Value;
    }
    else if (CarryOverWidth <= 0) {
	U32bit Mask = ((0x00000001u << BitWidth) - 1) << BitIndex;
	DataBlock[WordIndex] &= ~Mask;
	DataBlock[WordIndex] |= (Value << BitIndex) & Mask;
    }
    else {
	U32bit Mask0 = ~((0x00000001u << BitIndex) - 1);
	DataBlock[WordIndex] &= ~Mask0;
	DataBlock[WordIndex] |= (Value << BitIndex) & Mask0;

	U32bit Mask1 = ((0x00000001u << CarryOverWidth) - 1);
	DataBlock[WordIndex+1] &= ~Mask1;
	DataBlock[WordIndex+1] |= (Value >> (32-BitIndex)) & Mask1;
    }
}



inline int TKinokoDataSectionScanner::SectionIdOf(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_PacketType] & _SectionIdMask;
}

inline int TKinokoDataSectionScanner::PacketSizeOf(void* Buffer)
{
    return (sizeof(U32bit) * Offset_DataArea) + DataSizeOf(Buffer);
}

inline int TKinokoDataSectionScanner::DataSizeOf(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_DataAreaSize];
}

inline void* TKinokoDataSectionScanner::DataAreaOf(void* Buffer)
{
    return (U32bit*) Buffer + Offset_DataArea;
}

inline int TKinokoDataSectionScanner::DataWordOf(void* Buffer, int WordIndex)
{
    return ((U32bit*) Buffer)[Offset_DataArea + WordIndex];
}

inline void TKinokoDataSectionScanner::GetElement(void* Buffer, int BitOffset, int BitWidth, int& Value)
{
    U32bit* DataBlock = (U32bit*) Buffer + Offset_DataArea;

    unsigned WordIndex = BitOffset >> 5;    // divide by 32-bit
    unsigned BitIndex = BitOffset & 0x001f; // mod by 32-bit
    int CarryOverWidth = BitIndex + BitWidth - 32;

    Value = DataBlock[WordIndex] >> BitIndex;
    if (CarryOverWidth > 0) {
	Value |= DataBlock[WordIndex + 1] << (32 - BitIndex);
    }

    if (BitWidth < 32) {
	Value &= (0x00000001u << BitWidth) - 1;
    }
    else {
	Value &= 0xffffffff;
    }
}


#endif
