/* KinokoDataSource.cc */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 25 January 2002. */


#include <iostream>
#include <string>
#include <vector>
#include <map>
#include "ParaTokenizer.hh"
#include "KinokoDataSection.hh"
#include "KinokoDataSectionContainer.hh"
#include "KinokoDataSource.hh"

using namespace std;



int TKinokoDataSource::AutomaticDataSourceIdAssignmentBase(void)
{
    return 0x4000; // 16384, half of signed 16bit
}

TKinokoDataSource::TKinokoDataSource(const string& DataSourceName, long DataSourceId)
{
    _DataSourceName = DataSourceName;
    _DataSourceId = DataSourceId;
}

TKinokoDataSource::~TKinokoDataSource()
{
}

void TKinokoDataSource::SetName(const string& DataSourceName, long DataSourceId)
{
    _DataSourceName = DataSourceName;
    _DataSourceId = DataSourceId;
}

void TKinokoDataSource::ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    try {
	TParaToken Token;
	Tokenizer.Next().MustBe("{");
	
	while ((Token = Tokenizer.LookAhead()).IsNot("}")) {
	    if (Token.Is("attribute")) {
		ReadAttribute(Tokenizer);
	    }
	    else if (Token.Is("section")) {
		ReadSection(Tokenizer);
	    }
	    else {
		Tokenizer.Unget(Token);
		SkipUnknownEntry(Tokenizer);
	    }
	}
	
	Tokenizer.Next().MustBe("}");
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    "TKinokoDataSource::ReadFrom()",
	    "script exception: " + e.Message()
	);
    }
}

void TKinokoDataSource::SkipUnknownEntry(TParaTokenizer& Tokenizer)
{
    int NestLevel = 0;

#if 1
    cerr << "WARNING: unknown entry skipped: ";
    cerr << Tokenizer.LookAhead().AsString() << endl;
#endif

    while (1) {
	TParaToken Token = Tokenizer.Next();
	if (Token.Is("{")) {
	    NestLevel++;
	}
	else if (Token.Is("}")) {
	    NestLevel--;
	}
	if (NestLevel == 0) {
	    if (Token.Is("}") || Token.Is(";")) {
		break;
	    }
	}
    }
}

void TKinokoDataSource::ReadSection(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    TKinokoDataSection* DataSection;
    TKinokoDataSource* DataSource;
    TKinokoDataSectionFactory* SectionFactory;

    DataSource = this;
    SectionFactory = TKinokoDataSectionFactory::GetInstance();
    DataSection = SectionFactory->ReadFrom(Tokenizer, DataSource);
    
    AddDataSection(DataSection);
}

void TKinokoDataSource::WriteTo(ostream& os)
{
    os << "datasource \"" << _DataSourceName << "\"";
    os << "<" << _DataSourceId << ">" << endl;
    os << "{" << endl;

    WriteAttributeList(os, "    ");

    string Indent;
    for (unsigned j = 0; j < _DataSectionList.size(); j++) {
	if (j != 0) {
	    os << endl;
	}
	_DataSectionList[j]->WriteTo(os, Indent = "    ");
    }

    os << "}" << endl;
}

TKinokoDataSectionCleaner& TKinokoDataSource::DataSectionCleaner(void)
{
    return _DataSectionCleaner;
}



TKinokoDataSectionCleaner::TKinokoDataSectionCleaner(void)
{
}

TKinokoDataSectionCleaner::~TKinokoDataSectionCleaner()
{
    set<TKinokoDataSection*>::iterator i;
    for (i = _DataSectionList.begin(); i != _DataSectionList.end(); i++) {
	delete *i;
    }
}

void TKinokoDataSectionCleaner::RegisterDataSection(TKinokoDataSection* DataSection)
{
    _DataSectionList.insert(DataSection);
}
