/* KinokoTaggedDataSection.cc */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 12 July 2007. */


#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <map>
#include <cstdio>
#include "ParaTokenizer.hh"
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"
#include "KinokoTaggedDataSection.hh"

using namespace std;


TKinokoTaggedDataSection::TKinokoTaggedDataSection(TKinokoDataSource* DataSource, const string& SectionName)
: TKinokoDataSection(DataSource, SectionName)
{
    _Formatter = 0;
    _Scanner = 0;
    
    _NumberOfFields = 0;
}

TKinokoTaggedDataSection::TKinokoTaggedDataSection(TKinokoDataSource* DataSource, const string& SectionName, int SectionId)
: TKinokoDataSection(DataSource, SectionName, SectionId)
{
    _Formatter = 0;
    _Scanner = 0;

    _NumberOfFields = 0;
}

TKinokoTaggedDataSection::~TKinokoTaggedDataSection()
{
    delete _Formatter;
    delete _Scanner;
}

int TKinokoTaggedDataSection::SectionType(void) const
{
    return TKinokoDataSection::SectionType_Tagged;
}

int TKinokoTaggedDataSection::AddField(const string& FieldName, int ValueWidth, const string& PrintFormat)
{
    int FieldIndex = _NumberOfFields;
    _NumberOfFields++;

    _FieldNameList.push_back(FieldName);
    _FieldNameTable[FieldName] = FieldIndex;

    _FieldValueWidthList.push_back(ValueWidth);
    _FieldValueTypeList.push_back(ElementType_Integer);

    _FieldValueNameTableList.push_back(TValueNameTable());
    _PrintFormatList.push_back(PrintFormat);

    return FieldIndex;
}

int TKinokoTaggedDataSection::AddField(const string& FieldName, int ValueWidth, const TValueNameTable& ValueNameTable)
{
    int FieldIndex = _NumberOfFields;
    _NumberOfFields++;

    _FieldNameList.push_back(FieldName);
    _FieldNameTable[FieldName] = FieldIndex;

    _FieldValueWidthList.push_back(ValueWidth);
    _FieldValueTypeList.push_back(ElementType_Integer);

    _FieldValueNameTableList.push_back(ValueNameTable);
    _PrintFormatList.push_back("");

    return FieldIndex;
}

int TKinokoTaggedDataSection::AddField(const string& FieldName, int ValueWidth, int ValueType, const string& PrintFormat)
{
    int FieldIndex = _NumberOfFields;
    _NumberOfFields++;

    _FieldNameList.push_back(FieldName);
    _FieldNameTable[FieldName] = FieldIndex;

    _FieldValueWidthList.push_back(ValueWidth);
    _FieldValueTypeList.push_back(ValueType);

    _FieldValueNameTableList.push_back(TValueNameTable());
    _PrintFormatList.push_back(PrintFormat);

    return FieldIndex;
}

bool TKinokoTaggedDataSection::HasField(const string& FieldName)
{
    return (_FieldNameTable.count(FieldName) > 0);
}

int TKinokoTaggedDataSection::FieldIndexOf(const string& FieldName) throw(TKinokoException)
{
    if (_FieldNameTable.count(FieldName) == 0) {
	throw TKinokoException(
	    "TKinokoTaggedDataSection::FieldIndexOf()",
	    "unknown field name: " + _SectionName + "::" + FieldName
	);
    }
    
    return _FieldNameTable[FieldName];    
}

string TKinokoTaggedDataSection::FieldNameOf(int FieldIndex) throw(TKinokoException)
{
    if ((FieldIndex < 0) || (FieldIndex >= _NumberOfFields)) {
	throw TKinokoException(
	    "TKinokoTaggedDataSection::FieldNameOf()",
	    "bad field index"
	);
    }

    return _FieldNameList[FieldIndex];
}

int TKinokoTaggedDataSection::NumberOfFields(void)
{
    return _NumberOfFields;
}

void TKinokoTaggedDataSection::ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    try {
	TParaToken Token;
	Tokenizer.Next().MustBe("{");
	
	while ((Token = Tokenizer.Next()).IsNot("}")) {
	    if (Token.Is("attribute")) {
		Tokenizer.Unget(Token);
		ReadAttribute(Tokenizer);
	    }
	    else if (Token.Is("field")) {
		string FieldName, PrintFormat;
		int ValueWidth = 0;
		int ValueType = ElementType_Integer;
		TValueNameTable ValueNameTable;
		
		Token = Tokenizer.Next();
		FieldName = Token.RemoveQuotation().AsString();
		if (Tokenizer.LookAhead().Is(":")) {
		    Tokenizer.Next().MustBe(":");
		    Tokenizer.Next().MustBe("int");
		    Tokenizer.Next().MustBe("-");
		    ValueWidth = Tokenizer.Next().AsLong();
		    Tokenizer.Next().MustBe("bit");

		    //... for backward compatibility (KDF2) ...//
		    ValueWidth = 32;
		}
		
		if ((Token = Tokenizer.Next()).Is("{")) {
		    long Value; 
		    string ValueName;
		    while ((Token = Tokenizer.Next()).IsNot("}")) {
			if (Token.Is("data")) {
			    Tokenizer.Next().MustBe(":");
			    Token = Tokenizer.Next();
			    if (Token.Is("int")) {
				ValueType = ElementType_Integer;
			    }
			    else if (Token.Is("float")) {
				ValueType = ElementType_Float;
			    }
			    else {
				ValueType = ElementType_Unknown;
			    }
			    Tokenizer.Next().MustBe("-");
			    ValueWidth = Tokenizer.Next().AsLong();
			    Tokenizer.Next().MustBe("bit");
			    Tokenizer.Next().MustBe(";");
			}
			else if (Token.Is("alias")) {
			    Value = Tokenizer.Next().AsLong();
			    Tokenizer.Next().MustBe("=>");
			    Token = Tokenizer.Next();
			    ValueName = Token.RemoveQuotation().AsString();
			    Tokenizer.Next().MustBe(";");
			    ValueNameTable[Value] = ValueName;
			}
			else if (Token.Is("print_format")) {
			    Tokenizer.Next().MustBe(":");
			    Token = Tokenizer.Next();
			    PrintFormat = Token.RemoveQuotation().AsString();
			    Tokenizer.Next().MustBe(";");
			}
			else {
			    Tokenizer.Unget(Token);
			    TKinokoDataSource::SkipUnknownEntry(Tokenizer);
			}
		    }
		}
		else {
		    Token.MustBe(";");
		}

		if (ValueWidth == 0) {
		    throw TKinokoException(
			"TKinokoTaggedDataSection::ReadFrom()",
			"data width not specified"
		    );
		}
		if (! ValueNameTable.empty()) {
		    AddField(FieldName, ValueWidth, ValueNameTable);
		}
		else {
		    AddField(FieldName, ValueWidth, ValueType, PrintFormat);
		}
	    }
	    else {
		Tokenizer.Unget(Token);
		TKinokoDataSource::SkipUnknownEntry(Tokenizer);
	    }
	}
	Token.MustBe("}");
    }
    catch (TScriptException &e) {
	cerr << e.Message() << endl;
	throw TKinokoException(
	    "TKinokoTaggedDataSection::ReadFrom()",
	    "script exception: " + e.Message()
	);
    }
}

void TKinokoTaggedDataSection::WriteTo(ostream& os, const string& Indent)
{
    os << Indent << "section \"" << _SectionName << "\"";
    os << "<" << _SectionId << ">";
    os << ": tagged";

    os << " {" << endl;

#if USE_KDF2
    ;
#else
    WriteAttributeList(os, Indent + "    ");
#endif

    for (unsigned i = 0; i < _FieldNameList.size(); i++) {
	os << Indent << "    " << "field \"" << _FieldNameList[i] << "\"";
#if USE_KDF2
	os << ": int-32bit;" << endl;
#else
	os << " {" << endl;
	os << Indent << "    ""    ""data: ";
	if (_FieldValueTypeList[i] == ElementType_Float) {
	    os << "float-32bit;" << endl;
	}
	else {
	    os << "int-" << _FieldValueWidthList[i] << "bit;" << endl;
	}

	if (! _FieldValueNameTableList[i].empty()) {
	    TValueNameTable::iterator ValueNameIterator;
	    for (
		ValueNameIterator = _FieldValueNameTableList[i].begin();
		ValueNameIterator != _FieldValueNameTableList[i].end();
		ValueNameIterator++
	    ){
		os << Indent << "    ""    ""alias ";
		os << (*ValueNameIterator).first << " => ";
                os << "\"" << (*ValueNameIterator).second << "\";" << endl;
	    }
	}
	else if (! _PrintFormatList[i].empty()) {
	    os << Indent << "    ""    ""print_format: ";
	    os << "\"" << _PrintFormatList[i] << "\";" << endl;
	}
	os << Indent << "    ""}" << endl;
#endif
    }

    os << Indent << "}" << endl;
}



TKinokoTaggedDataSectionFormatter::TKinokoTaggedDataSectionFormatter(TKinokoTaggedDataSection* DataSection)
: TKinokoDataSectionFormatter(DataSection)
{
    const std::vector<int>& FieldValueWidthList = (
	DataSection->FieldValueWidthList()
    );

    _FieldBitOffsetList = new int[FieldValueWidthList.size()];
    _FieldBitWidthList = new int[FieldValueWidthList.size()];

    unsigned FieldBitWidth, FieldBitOffset = 0;
    for (unsigned i = 0; i < FieldValueWidthList.size(); i++) {
#if USE_KDF2
	FieldBitWidth = 32;
#else
	FieldBitWidth = FieldValueWidthList[i];
#endif
	_FieldBitOffsetList[i] = FieldBitOffset;
	_FieldBitWidthList[i] = FieldBitWidth;
	FieldBitOffset += FieldBitWidth;
    }

    _DataSize = FieldBitOffset / 8;
    if (FieldBitOffset & 0x07) {
	_DataSize += 1;
    }
}

TKinokoTaggedDataSectionFormatter::~TKinokoTaggedDataSectionFormatter()
{
    delete[] _FieldBitOffsetList;
    delete[] _FieldBitWidthList;
}

void TKinokoTaggedDataSectionFormatter::MapVariable(int FieldIndex, int* Variable)
{
    _MappedIntList.push_back(make_pair(FieldIndex, Variable));
}

void TKinokoTaggedDataSectionFormatter::MapFloatVariable(int FieldIndex, float* Variable)
{
    _MappedFloatList.push_back(make_pair(FieldIndex, Variable));
}

void TKinokoTaggedDataSectionFormatter::Update(void* Buffer) const
{
    for (unsigned i = 0; i < _MappedIntList.size(); i++) {
	WriteTo(
	    Buffer, _MappedIntList[i].first, *_MappedIntList[i].second
	);
    }
    for (unsigned i = 0; i < _MappedFloatList.size(); i++) {
	WriteFloatTo(
	    Buffer, _MappedFloatList[i].first, *_MappedFloatList[i].second
	);
    }
}



TKinokoTaggedDataSectionScanner::TKinokoTaggedDataSectionScanner(TKinokoTaggedDataSection* DataSection)
{
    _DataSection = DataSection;

    const std::vector<int>& FieldValueWidthList = (
	DataSection->FieldValueWidthList()
    );
    const std::vector<int>& FieldValueTypeList = (
	DataSection->FieldValueTypeList()
    );

    _FieldBitOffsetList = new int[FieldValueWidthList.size()];
    _FieldBitWidthList = new int[FieldValueWidthList.size()];
    _FieldValueTypeList = new int[FieldValueWidthList.size()];

    unsigned FieldBitWidth, FieldBitOffset = 0;
    for (unsigned i = 0; i < FieldValueWidthList.size(); i++) {
	FieldBitWidth = FieldValueWidthList[i];
	_FieldBitOffsetList[i] = FieldBitOffset;
	_FieldBitWidthList[i] = FieldBitWidth;
	FieldBitOffset += FieldBitWidth;

	_FieldValueTypeList[i] = FieldValueTypeList[i];
    }
}

TKinokoTaggedDataSectionScanner::~TKinokoTaggedDataSectionScanner()
{
    delete[] _FieldBitOffsetList;
    delete[] _FieldBitWidthList;
    delete[] _FieldValueTypeList;
}

void TKinokoTaggedDataSectionScanner::ReadStringFrom(void* Buffer, int FieldIndex, std::string& Value) const
{
    int IntValue;

    GetElement(
	Buffer, 
	_FieldBitOffsetList[FieldIndex], _FieldBitWidthList[FieldIndex], 
	IntValue
    );

    if (_DataSection->HasFieldValueNameTableAt(FieldIndex)) {
	Value = _DataSection->FieldValueNameTableOf(FieldIndex)[IntValue];
	if (! Value.empty()) {
	    return;
	}
    }

    bool IsFloat = (_FieldValueTypeList[FieldIndex] == TKinokoDataSection::ElementType_Float);

    const string& PrintFormat = _DataSection->PrintFormatOf(FieldIndex);
    if (! PrintFormat.empty()) {
	char Output[64];
	if (IsFloat) {
	    float FloatValue = WordToFloat(IntValue);
	    snprintf(
		Output, sizeof(Output), PrintFormat.c_str(), FloatValue
	    );
	}
	else {
	    snprintf(
		Output, sizeof(Output), PrintFormat.c_str(), IntValue
	    );
	}
	Value = Output;
    }
    else {
	ostringstream os;
	if (IsFloat) {
	    float FloatValue = WordToFloat(IntValue);
	    os << FloatValue;
	}
	else {
	    os << IntValue;
	}
	Value = os.str();
    }
}

void TKinokoTaggedDataSectionScanner::MapVariable(const std::string& FieldName, int* Variable) throw(TKinokoException)
{
    _MappedIntList.push_back(
        make_pair(_DataSection->FieldIndexOf(FieldName), Variable)
    );
}

void TKinokoTaggedDataSectionScanner::MapFloatVariable(const std::string& FieldName, float* Variable) throw(TKinokoException)
{
    _MappedFloatList.push_back(
        make_pair(_DataSection->FieldIndexOf(FieldName), Variable)
    );
}

void TKinokoTaggedDataSectionScanner::MapStringVariable(const std::string& FieldName, std::string* Variable) throw(TKinokoException)
{
    _MappedStringList.push_back(
        make_pair(_DataSection->FieldIndexOf(FieldName), Variable)
    );
}

void TKinokoTaggedDataSectionScanner::Update(void* Buffer) const
{
    for (unsigned i = 0; i < _MappedIntList.size(); i++) {
	ReadFrom(
	    Buffer, _MappedIntList[i].first, *_MappedIntList[i].second
	);
    }
    for (unsigned i = 0; i < _MappedFloatList.size(); i++) {
	ReadFloatFrom(
	    Buffer, _MappedFloatList[i].first, *_MappedFloatList[i].second
	);
    }
    for (unsigned i = 0; i < _MappedStringList.size(); i++) {
	ReadStringFrom(
	    Buffer, _MappedStringList[i].first, *_MappedStringList[i].second
	);
    }
}
