/* KinokoTaggedDataSection.hh */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 12 July 2007. */


#ifndef __KinokoTaggedDataSection_hh__
#define __KinokoTaggedDataSection_hh__


#include <iostream>
#include <string>
#include <vector>
#include <map>
#include "MushConfig.hh"
#include "ParaTokenizer.hh"
#include "KinokoDataSection.hh"


class TKinokoTaggedDataSectionFormatter;
class TKinokoTaggedDataSectionScanner;


class TKinokoTaggedDataSection: public TKinokoDataSection {
  public:
    typedef std::map<long, std::string> TValueNameTable;
  public:
    TKinokoTaggedDataSection(TKinokoDataSource* DataSource, const std::string& SectionName);
    TKinokoTaggedDataSection(TKinokoDataSource* DataSource, const std::string& SectionName, int SectionId);
    virtual ~TKinokoTaggedDataSection();
    virtual int SectionType(void) const;
    virtual void ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException);
    virtual void WriteTo(std::ostream& os, const std::string& Indent);
    inline TKinokoTaggedDataSectionFormatter* Formatter(void);
    inline TKinokoTaggedDataSectionScanner* Scanner(void);
    inline const std::vector<int>& FieldValueWidthList(void) const;
    inline const std::vector<int>& FieldValueTypeList(void) const;
  public:
    int AddField(const std::string& FieldName, int ValueWidth, const std::string& PrintFormat = "");
    int AddField(const std::string& FieldName, int ValueWidth, const TValueNameTable& ValueNameTable);
    int AddField(const std::string& FieldName, int ValueWidth, int FieldValueType, const std::string& PrintFormat = "");
    bool HasField(const std::string& FieldName);
    int FieldIndexOf(const std::string& FieldName) throw(TKinokoException);
    std::string FieldNameOf(int FieldIndex) throw(TKinokoException);
    int NumberOfFields(void);
    inline const std::string& PrintFormatOf(int FieldIndex);
    inline bool HasFieldValueNameTableAt(int FieldIndex);
    inline TValueNameTable& FieldValueNameTableOf(int FieldIndex);
  protected:
    int _NumberOfFields;
    std::vector<std::string> _FieldNameList;
    std::map<std::string, long> _FieldNameTable;
    std::vector<int> _FieldValueWidthList;
    std::vector<int> _FieldValueTypeList;
    std::vector<TValueNameTable> _FieldValueNameTableList;
    std::vector<std::string> _PrintFormatList;
    TKinokoTaggedDataSectionFormatter* _Formatter;
    TKinokoTaggedDataSectionScanner* _Scanner;
};


class TKinokoTaggedDataSectionFormatter: public TKinokoDataSectionFormatter {
  public:
    TKinokoTaggedDataSectionFormatter(TKinokoTaggedDataSection* DataSection);
    virtual ~TKinokoTaggedDataSectionFormatter();
  public:
    inline int DataSize(void) const;
    inline void WriteTo(void* Buffer, int FieldIndex, int Value) const;
    inline void WriteFloatTo(void* Buffer, int FieldIndex, float Value) const;
  public:
    void MapVariable(int FieldIndex, int* Variable);
    void MapFloatVariable(int FieldIndex, float* Variable);
    void Update(void* Buffer) const;
  protected:
    int _DataSize;
    int* _FieldBitOffsetList;
    int* _FieldBitWidthList;
  protected:
    std::vector<std::pair<int, int*> > _MappedIntList;
    std::vector<std::pair<int, float*> > _MappedFloatList;
};


class TKinokoTaggedDataSectionScanner: public TKinokoDataSectionScanner {
  public:
    TKinokoTaggedDataSectionScanner(TKinokoTaggedDataSection* DataSection);
    virtual ~TKinokoTaggedDataSectionScanner();
  public:
    inline void ReadFrom(void* Buffer, int FieldIndex, int& Value) const;
    inline void ReadFloatFrom(void* Buffer, int FieldIndex, float& Value) const;
    void ReadStringFrom(void* Buffer, int FieldIndex, std::string& Value) const;
  public:
    void MapVariable(const std::string& FieldName, int* Variable) throw(TKinokoException);
    void MapFloatVariable(const std::string& FieldName, float* Variable) throw(TKinokoException);
    void MapStringVariable(const std::string& FieldName, std::string* Variable) throw(TKinokoException);
    void Update(void* Buffer) const;
  protected:
    TKinokoTaggedDataSection* _DataSection;
    int* _FieldBitOffsetList;
    int* _FieldBitWidthList;
    int* _FieldValueTypeList;
  protected:
    std::vector<std::pair<int, int*> > _MappedIntList;
    std::vector<std::pair<int, float*> > _MappedFloatList;
    std::vector<std::pair<int, std::string*> > _MappedStringList;
};



inline TKinokoTaggedDataSectionFormatter* TKinokoTaggedDataSection::Formatter(void)
{
    if (_Formatter == 0) {
	_Formatter = new TKinokoTaggedDataSectionFormatter(this);
    }

    return _Formatter;
}

inline TKinokoTaggedDataSectionScanner* TKinokoTaggedDataSection::Scanner(void)
{
    if (_Scanner == 0) {
	_Scanner = new TKinokoTaggedDataSectionScanner(this);
    }

    return _Scanner;
}

inline const std::string& TKinokoTaggedDataSection::PrintFormatOf(int FieldIndex)
{
    return _PrintFormatList[FieldIndex];
}

inline bool TKinokoTaggedDataSection::HasFieldValueNameTableAt(int FieldIndex)
{
    return ! _FieldValueNameTableList[FieldIndex].empty();
}

inline TKinokoTaggedDataSection::TValueNameTable& TKinokoTaggedDataSection::FieldValueNameTableOf(int FieldIndex)
{
    return _FieldValueNameTableList[FieldIndex];
}

inline const std::vector<int>& TKinokoTaggedDataSection::FieldValueWidthList(void) const
{
    return _FieldValueWidthList;
}

inline const std::vector<int>& TKinokoTaggedDataSection::FieldValueTypeList(void) const
{
    return _FieldValueTypeList;
}



inline int TKinokoTaggedDataSectionFormatter::DataSize(void) const
{
    return _DataSize;
}

inline void TKinokoTaggedDataSectionFormatter::WriteTo(void* Buffer, int FieldIndex, int Value) const
{
    PutElement(
	Buffer, 
	_FieldBitOffsetList[FieldIndex], _FieldBitWidthList[FieldIndex], 
	Value
    );
}

inline void TKinokoTaggedDataSectionFormatter::WriteFloatTo(void* Buffer, int FieldIndex, float Value) const
{
    PutElement(
	Buffer, 
	_FieldBitOffsetList[FieldIndex], _FieldBitWidthList[FieldIndex], 
	FloatToWord(Value)
    );
}



inline void TKinokoTaggedDataSectionScanner::ReadFrom(void* Buffer, int FieldIndex, int& Value) const
{
    GetElement(
	Buffer, 
	_FieldBitOffsetList[FieldIndex], _FieldBitWidthList[FieldIndex], 
	Value
    );

    if (_FieldValueTypeList[FieldIndex] == TKinokoDataSection::ElementType_Float) {
	Value = (int) WordToFloat(Value);
    }
}

inline void TKinokoTaggedDataSectionScanner::ReadFloatFrom(void* Buffer, int FieldIndex, float& Value) const
{
    int IntValue;

    GetElement(
	Buffer, 
	_FieldBitOffsetList[FieldIndex], _FieldBitWidthList[FieldIndex], 
	IntValue
    );

    if (_FieldValueTypeList[FieldIndex] == TKinokoDataSection::ElementType_Float) {
	Value = WordToFloat(IntValue);
    }
    else {
	Value = IntValue;
    }
}


#endif
