/* KinokoMetaDeviceMessenger.cc */
/* Created by Enomoto Sanshiro on 4 July 2001. */
/* Last updated by Enomoto Sanshiro on 11 July 2007. */


#include <sstream>
#include "MushConfig.hh"
#include "ParaOperator.hh"
#include "KinokoDaqBuilder.hh"
#include "KinokoReadoutAction.hh"
#include "KinokoMetaReadoutAction.hh"
#include "KinokoReadoutCondition.hh"
#include "KinokoMetaDeviceMessenger.hh"

using namespace std;


int TKinokoRegisterMessenger::_RegisterCount = 0;

TKinokoRegisterMessenger::TKinokoRegisterMessenger(TKinokoDaqBuilder* DaqBuilder)
: TParaObjectPrototype("Register")
{
    _DaqBuilder = DaqBuilder;

    _Register = 0;
}

TKinokoRegisterMessenger::~TKinokoRegisterMessenger()
{
}

TParaObjectPrototype* TKinokoRegisterMessenger::Clone(void)
{
    return new TKinokoRegisterMessenger(_DaqBuilder);
}

void TKinokoRegisterMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TParaObjectPrototype::Construct(ClassName, ArgumentList);

    _Register = new int(0);    
    _RegisterName = CreateRegisterName();

    _DaqBuilder->RegisterRegister(_RegisterName, _Register);
}

string TKinokoRegisterMessenger::RegisterName(void)
{
    return _RegisterName;
}

int TKinokoRegisterMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "dump") || (MethodName == "Dump")) {
        ReturnValue = Dump(ArgumentList);
	return 1;
    }

    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _RegisterName + "." + MethodName + "(...)", "too few argument[s]"
	);
    }	    
    TParaValue& Parameter = *ArgumentList[0];

    if ((MethodName == "load") || (MethodName == "Load")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_Load
	);
    }
    else if ((MethodName == "add") || (MethodName == "Add")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_Add
	);
    }
    else if ((MethodName == "subtract") || (MethodName == "Subtract")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_Subtract
	);
    }
    else if ((MethodName == "multiply") || (MethodName == "Multiply")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_Multiply
	);
    }
    else if ((MethodName == "divide") || (MethodName == "Divide")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_Divide
	);
    }
    else if ((MethodName == "modulo") || (MethodName == "Modulo")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_Modulo
	);
    }
    else if ((MethodName == "and") || (MethodName == "And")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_And
	);
    }
    else if ((MethodName == "or") || (MethodName == "Or")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_Or
	);
    }
    else if ((MethodName == "xor") || (MethodName == "Xor")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_Xor
	);
    }
    else if ((MethodName == "shiftLeft") || (MethodName == "ShiftLeft")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_ShiftLeft
	);
    }
    else if ((MethodName == "shiftRight") || (MethodName == "ShiftRight")) {
        ReturnValue = AddRegisterAction(
            MethodName, Parameter, TKinokoRegisterAction::OperationId_ShiftRight
	);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue& TKinokoRegisterMessenger::EvaluateOperator(TParaOperator* Operator, TParaValue& LeftValue, TParaValue& RightValue, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    string Symbol = Operator->Symbol();
    
    if (Symbol == "=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_Load
	);
    }
    else if (Symbol == "+=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_Add
	);
    }
    else if (Symbol == "-=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_Subtract
	);
    }
    else if (Symbol == "*=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_Multiply
	);
    }
    else if (Symbol == "/=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_Divide
	);
    }
    else if (Symbol == "%=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_Modulo
	);
    }
    else if (Symbol == "&=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_And
	);
    }
    else if (Symbol == "|=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_Or
	);
    }
    else if (Symbol == "^=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_Xor
	);
    }
    else if (Symbol == "<<=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_ShiftLeft
	);
    }
    else if (Symbol == ">>=") {
        Result = AddRegisterAction(
            Symbol, RightValue, TKinokoRegisterAction::OperationId_ShiftRight
	);
    }
    else {
	int* ParameterRegister = GetConstRegisterOf(RightValue, _DaqBuilder);

	//... BUG: This ConditionMessenger will not be deleted
	TKinokoRegisterConditionMessenger* ConditionMessenger;

	ConditionMessenger = new TKinokoRegisterConditionMessenger(
	    Symbol, _Register, ParameterRegister
	);

	Result = TParaValue(ConditionMessenger);
    }

    return Result;
}

TParaValue TKinokoRegisterMessenger::AddRegisterAction(const string& ActionName, TParaValue& Parameter, int RegisterOperationId) throw(TScriptException)
{
    int* ParameterRegister;
    try {
	ParameterRegister = GetConstRegisterOf(Parameter, _DaqBuilder);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    _RegisterName + "." + ActionName + "(...): ", e.Message()
	);
    }
    
    TKinokoRegisterAction* RegisterAction = new TKinokoRegisterAction(
	_Register, RegisterOperationId, ParameterRegister
    );

    try {
	_DaqBuilder->AddReadoutAction(RegisterAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
            _RegisterName + "." + ActionName + "()", e.Message()
	);
    }

    return TParaValue((TParaObjectPrototype*) this);
}

TParaValue TKinokoRegisterMessenger::Dump(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoReadoutAction* Action = new TKinokoRegisterDumpAction(
	_Register, ObjectName()
    );

    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_RegisterName + ".dump(): " + e.Message());
    }

    return TParaValue();
}

string TKinokoRegisterMessenger::CreateRegisterName(void)
{
    ostringstream RegisterNameStream;
    RegisterNameStream << "__register_" << _RegisterCount++;

    return RegisterNameStream.str();
}

int* TKinokoRegisterMessenger::GetRegisterOf(TParaValue& Value, TKinokoDaqBuilder* DaqBuilder) throw(TScriptException)
{
    if (! Value.IsObject("Register")) {
	throw TScriptException(
	    "register value is expexted: " + Value.AsString()
	);
    }

    string RegisterName = (
	((TKinokoRegisterMessenger*) Value.AsObject())->RegisterName()
    );

    int* Register = DaqBuilder->DevicePool()->LookupRegister(RegisterName);
    if (Register == 0) {
	throw TScriptException("invalid register name: " + RegisterName);
    }
    
    return Register;
}

int* TKinokoRegisterMessenger::GetConstRegisterOf(TParaValue& Value, TKinokoDaqBuilder* DaqBuilder) throw(TScriptException)
{
    int* Register;

    if (Value.IsObject("Register")) {
	Register = GetRegisterOf(Value, DaqBuilder);
    }
    else if (Value.IsNumeric() || Value.IsString()) {
	int IntValue;
	if (Value.IsDouble()) {
	    IntValue = FloatToWord(Value.AsDouble());
	}
	else if (Value.IsString()) {
	    IntValue = DaqBuilder->HashValueOf(Value.AsString());
	}
	else {
	    IntValue = Value.AsLong();
	}

	Register = new int(IntValue);
	DaqBuilder->RegisterRegister(CreateRegisterName(), Register);
    }
    else {
	throw TScriptException(
	    "integer or register value is expexted: " + Value.AsString()
	);
    }

    return Register;
}



TKinokoReadoutConditionMessenger::TKinokoReadoutConditionMessenger(void)
: TParaObjectPrototype("ReadoutCondition")
{
}

TKinokoReadoutConditionMessenger::~TKinokoReadoutConditionMessenger()
{
}

TParaObjectPrototype* TKinokoReadoutConditionMessenger::Clone(void)
{
    return 0;
}

TParaValue& TKinokoReadoutConditionMessenger::EvaluateOperator(TParaOperator* Operator, TParaValue& LeftValue, TParaValue& RightValue, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    //... BUG: This ConditionMessenger will not be deleted
    TKinokoReadoutConditionMessenger* ConditionMessenger;

    if (Operator->Name() == "Not") {
	ConditionMessenger = new TKinokoComplexReadoutConditionMessenger(
	    Operator->Symbol(), 0, this
	);
    }
    else if (RightValue.IsObject("ReadoutCondition")) {
	ConditionMessenger = new TKinokoComplexReadoutConditionMessenger(
	    Operator->Symbol(), 
	    this, 
	    (TKinokoReadoutConditionMessenger*) RightValue.AsObject()
	);
    }
    else {
	throw TScriptException("invalid operation: " + Operator->Symbol());
    }

    return Result = TParaValue(ConditionMessenger);
}



TKinokoRegisterConditionMessenger::TKinokoRegisterConditionMessenger(const string& OperatorSymbol, int* LeftRegister, int* RightRegister)
{
    _OperatorSymbol = OperatorSymbol;
    _LeftRegister = LeftRegister;
    _RightRegister = RightRegister;
}

TKinokoRegisterConditionMessenger::~TKinokoRegisterConditionMessenger()
{
}

TKinokoReadoutCondition* TKinokoRegisterConditionMessenger::GetReadoutCondition(void) throw(TScriptException)
{
    int OperatorId;
    if (_OperatorSymbol == "==") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_Equal;
    }
    else if (_OperatorSymbol == "!=") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_NotEqual;
    }
    else if (_OperatorSymbol == ">") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_GreaterThan;
    }
    else if (_OperatorSymbol == ">=") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_GreaterEqual;
    }
    else if (_OperatorSymbol == "<") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_LessThan;
    }
    else if (_OperatorSymbol == "<=") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_LessEqual;
    }
    else if (_OperatorSymbol == "&") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_BitAnd;
    }
    else if (_OperatorSymbol == "^") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_BitXor;
    }
    else {
	throw TScriptException("invalid register operator: " + _OperatorSymbol);
    }

    TKinokoReadoutCondition* Condition = new TKinokoRegisterReadoutCondition(
	_LeftRegister, _RightRegister, OperatorId
    );

    return Condition;
}



TKinokoComplexReadoutConditionMessenger::TKinokoComplexReadoutConditionMessenger(const string& OperatorSymbol, TKinokoReadoutConditionMessenger* LeftConditionMessenger, TKinokoReadoutConditionMessenger* RightConditionMessenger)
{
    _OperatorSymbol = OperatorSymbol;
    _LeftConditionMessenger = LeftConditionMessenger;
    _RightConditionMessenger = RightConditionMessenger;
}

TKinokoComplexReadoutConditionMessenger::~TKinokoComplexReadoutConditionMessenger()
{
    delete _LeftConditionMessenger;
    delete _RightConditionMessenger;
}

TKinokoReadoutCondition* TKinokoComplexReadoutConditionMessenger::GetReadoutCondition(void) throw(TScriptException)
{
    int OperatorId;
    if (_OperatorSymbol == "!") { 
	OperatorId = TKinokoComplexReadoutCondition::Operator_Not;
    }
    else if (_OperatorSymbol == "&&") { 
	OperatorId = TKinokoComplexReadoutCondition::Operator_And;
    }
    else if (_OperatorSymbol == "||") { 
	OperatorId = TKinokoComplexReadoutCondition::Operator_Or;
    }
    else {
	throw TScriptException("invalid register operator: " + _OperatorSymbol);
    }

    TKinokoReadoutCondition* LeftCondition = 0;
    TKinokoReadoutCondition* RightCondition = 0;

    if (_LeftConditionMessenger) {
	LeftCondition = _LeftConditionMessenger->GetReadoutCondition();
    }
    if (_RightConditionMessenger) {
	RightCondition = _RightConditionMessenger->GetReadoutCondition();
    }

    TKinokoReadoutCondition* Condition = new TKinokoComplexReadoutCondition(
	LeftCondition, RightCondition, OperatorId
    );

    return Condition;
}



TKinokoDataRecordMessenger::TKinokoDataRecordMessenger(TKinokoDaqBuilder* DaqBuilder)
: TParaObjectPrototype("DataRecord")
{
    _DaqBuilder = DaqBuilder;
    _DataRecord = 0;
}

TKinokoDataRecordMessenger::~TKinokoDataRecordMessenger()
{
}

TParaObjectPrototype* TKinokoDataRecordMessenger::Clone(void)
{
    return new TKinokoDataRecordMessenger(_DaqBuilder);
}

void TKinokoDataRecordMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() > 0) {
	_DataRecordName = ArgumentList[0]->AsString();
    }
    else {
	_DataRecordName = this->ObjectName();
    }

    if (ArgumentList.size() > 1) {
	_SectionId = ArgumentList[1]->AsLong();
    }
    else {
	_SectionId = TKinokoDataSection::SectionId_Unknown;
    }

    TKinokoDataSource* DataSource = _DaqBuilder->DataSource();

    _DataSection = new TKinokoTaggedDataSection(DataSource, _DataRecordName, _SectionId);
    _DataSection->AddAttribute("creator", "DataRecord");

    _DataRecord = new TKinokoDataRecord(_DataSection);

    _DaqBuilder->RegisterDataRecord(_DataRecordName, _DataRecord);
}

int TKinokoDataRecordMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "fill") || (MethodName == "Fill")) {
        ReturnValue = Fill(ArgumentList);
    }
    else if ((MethodName == "send") || (MethodName == "Send")) {
        ReturnValue = Send(ArgumentList);
    }
    else if ((MethodName == "dump") || (MethodName == "Dump")) {
        ReturnValue = Dump(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKinokoDataRecordMessenger::Fill(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few arguments: " + _DataRecordName + 
	    ".fill(string field_name, Register/int value, ...)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "bad argument: " + _DataRecordName + 
	    ".fill(string field_name, Register/int value, ...)"
	);
    }
    if ((ArgumentList.size() > 2) && ! ArgumentList[2]->IsLong()) {
	throw TScriptException(
	    "bad argument: " + _DataRecordName + 
	    ".fill(string field_name, Register/int value, int width, ...)"
	);
    }
    if ((ArgumentList.size() > 3) && ! ArgumentList[3]->IsString()) {
	throw TScriptException(
	    "bad argument: " + _DataRecordName + 
	    ".fill(string field_name, Register/int value, int width, string print_format)"
	);
    }

    string FieldName = ArgumentList[0]->AsString();

    int FieldValueType = TKinokoDataSection::ElementType_Integer;
    if (ArgumentList[1]->IsDouble()) {
	FieldValueType = TKinokoDataSection::ElementType_Float;
    }

    int Width = 32;
    if (ArgumentList.size() > 2) {
	Width = ArgumentList[2]->AsLong();
    }
    
    string PrintFormat;
    if (ArgumentList.size() > 3) {
	PrintFormat = ArgumentList[3]->AsString();
    }

    int FieldIndex = _DataRecord->AddField(
	FieldName, Width, FieldValueType, PrintFormat
    );
    
    int* Register;
    try {
	Register = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(_DataRecordName + ".fill(): " + e.Message());
    }
    
    TKinokoDataRecordFillAction* DataRecordFillAction = (
	new TKinokoDataRecordFillAction(_DataRecord, FieldIndex, Register)
    );
    try {
	_DaqBuilder->AddReadoutAction(DataRecordFillAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_DataRecordName + ".fill(): " + e.Message());
    }

    if (ArgumentList[1]->IsString()) {
	_DataRecord->AddFieldValueAlias(
	    FieldIndex, *Register, ArgumentList[1]->AsString()
	);
    }

    return TParaValue();
}

TParaValue TKinokoDataRecordMessenger::Send(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoDataRecordSendAction* Action = new TKinokoDataRecordSendAction(
	_DataRecord
    );

    try {
	_DaqBuilder->AddReadoutAction(Action, _DataSection);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_DataRecordName + ".send(): " + e.Message());
    }

    return TParaValue();
}

TParaValue TKinokoDataRecordMessenger::Dump(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoDataRecordDumpAction* Action = new TKinokoDataRecordDumpAction(
	_DataRecord
    );

    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_DataRecordName + ".dump(): " + e.Message());
    }

    return TParaValue();
}



TKinokoDaqSystemMessenger::TKinokoDaqSystemMessenger(TKinokoDaqBuilder* DaqBuilder, const string& ScriptFileName)
: TParaObjectPrototype("DaqSystem")
{
    _DaqBuilder = DaqBuilder;
    _ScriptFileName = ScriptFileName;
}

TKinokoDaqSystemMessenger::~TKinokoDaqSystemMessenger()
{
}

TParaObjectPrototype* TKinokoDaqSystemMessenger::Clone(void)
{
    return new TKinokoDaqSystemMessenger(_DaqBuilder, _ScriptFileName);
}

void TKinokoDaqSystemMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
}

int TKinokoDaqSystemMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((MethodName == "getRegistry") || (MethodName == "GetRegistry")) {
        // normal registry access
        ReturnValue = GetRegistry(ArgumentList);
    }
    else if ((MethodName == "setRegistry") || (MethodName == "SetRegistry")) {
        // normal registry access
        ReturnValue = SetRegistry(ArgumentList);
    }
    else if ((MethodName == "readRegistry") || (MethodName == "ReadRegistry")) {
        // registry access action
        ReturnValue = ReadRegistry(ArgumentList);
    }
    else if ((MethodName == "writeRegistry") || (MethodName == "WriteRegistry")) {
        // registry access action
        ReturnValue = WriteRegistry(ArgumentList);
    }
    else if ((MethodName == "suspend") || (MethodName == "Suspend")) {
        ReturnValue = Suspend(ArgumentList);
    }
    else if ((MethodName == "echo") || (MethodName == "Echo")) {
        ReturnValue = Echo(ArgumentList);
    }
    else if ((MethodName == "readTime") || (MethodName == "ReadTime")) {
        ReturnValue = ReadTime(ArgumentList);
    }
    else if ((MethodName == "setPollingInterval") || (MethodName == "SetPollingInterval")) {
        ReturnValue = SetPollingInterval(ArgumentList);
    }
    else if ((MethodName == "forcePolling") || (MethodName == "ForcePolling")) {
        ReturnValue = ForcePolling(ArgumentList);
    }
    else if ((MethodName == "scriptFileName") || (MethodName == "ScriptFileName")) {
        ReturnValue = ScriptFileName(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKinokoDaqSystemMessenger::GetRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "too few arguments: getRegistry(string registry_path)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument:  getRegistry(string registry_path)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    TKinokoRegistry* Registry = _DaqBuilder->Registry();

    string Value = Registry->GetValue(RegistryPath);

    return TParaValue(Value);
}

TParaValue TKinokoDaqSystemMessenger::SetRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few argument[s]: "
	    "setRegistry(string registry_path, string value)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument[s]: "
	    "setRegistry(string registry_path, string value)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();

    TKinokoRegistry* Registry = _DaqBuilder->Registry();
    Registry->SetValue(RegistryPath, Value);

    return TParaValue(Value);
}

TParaValue TKinokoDaqSystemMessenger::ReadRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few arguments: "
	    "readRegistry(string registry_path, Register register)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument: "
	    "readRegistry(string registry_path, Register register)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    int* Register;
    try {
	Register = TKinokoRegisterMessenger::GetRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "invalid argument: "
	    "readRegistry(string registry_path, Register register): " +
	    e.Message()
	);
    }

    TKinokoRegistry* Registry = _DaqBuilder->Registry();

    TKinokoReadoutAction* Action = new TKinokoReadRegistryAction(
	Registry, RegistryPath, Register
    );
    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("readRegistry(): " + e.Message());
    }

    return TParaValue();
}

TParaValue TKinokoDaqSystemMessenger::WriteRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few arguments: "
	    "writeRegistry(string registry_path, Registry register)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument: "
	    "writeRegistry(string registry_path, Register register)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    int* Register;
    try {
	Register = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "invalid argument: "
	    "writeRegistry(string registry_path, Register register): " +
	    e.Message()
	);
    }

    TKinokoRegistry* Registry = _DaqBuilder->Registry();

    TKinokoReadoutAction* Action = new TKinokoWriteRegistryAction(
	Registry, RegistryPath, Register
    );
    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("writeRegistry(): " + e.Message());
    }

    return TParaValue();
}

TParaValue TKinokoDaqSystemMessenger::Suspend(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException("too few arguments: suspend(int sec, int usec)");
    }
    if ((! ArgumentList[0]->IsLong()) || (! ArgumentList[1]->IsLong())) {
	throw TScriptException("bad argument: suspend(int sec, int usec)");
    }

    int Time_sec = ArgumentList[0]->AsLong();
    int Time_usec = ArgumentList[1]->AsLong();
    
    TKinokoReadoutAction* Action = new TKinokoSuspendAction(
	Time_sec, Time_usec
    );

    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("suspend(): " + e.Message());
    }

    return TParaValue();
}

TParaValue TKinokoDaqSystemMessenger::ReadTime(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "too few arguments: ", "readTime(Register register)"
	);
    }

    int* SecRegister = 0;
    int* USecRegister = 0;
    try {
	SecRegister = TKinokoRegisterMessenger::GetRegisterOf(
	    *ArgumentList[0], _DaqBuilder
	);
	if (ArgumentList.size() > 1) {
	    USecRegister = TKinokoRegisterMessenger::GetRegisterOf(
		*ArgumentList[1], _DaqBuilder
	    );
	}
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "invalid argument: readTime(Register register): " + e.Message()
	);
    }

    TKinokoReadoutAction* Action = new TKinokoReadTimeAction(
	SecRegister, USecRegister
    );

    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("readTime(): " + e.Message());
    }

    return TParaValue();
}

TParaValue TKinokoDaqSystemMessenger::Echo(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    string Message;
    if (ArgumentList.size() > 0) {
	Message = ArgumentList[0]->AsString();
    }

    int* Register = 0;
    if (ArgumentList.size() > 1) {
	try {
	    Register = TKinokoRegisterMessenger::GetConstRegisterOf(
		*ArgumentList[1], _DaqBuilder
	    );
	}
	catch (TScriptException &e) {
	    throw TScriptException(
		"invalid argument: "
		"echo(string message, Register register): " +
		e.Message()
	    );
	}
    }

    int Base = 10;
    if (ArgumentList.size() > 2) {
	try {
	    Base = ArgumentList[2]->AsLong();
	}
	catch (TScriptException &e) {
	    throw TScriptException(
		"invalid argument: "
		"echo(string message, Register register, int base): " +
		e.Message()
	    );
	}
    }

    TKinokoReadoutAction* Action = new TKinokoEchoAction(
	Message, Register, Base
    );

    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("echo(): " + e.Message());
    }

    return TParaValue();
}

TParaValue TKinokoDaqSystemMessenger::SetPollingInterval(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsNumeric())) {
	throw TScriptException(
	    "invalid argument[s]: setPollingInterval(int interval_usec)"
	);
    }
    int Interval = ArgumentList[0]->AsLong();

    try {
	_DaqBuilder->Trigger()->SetPollingInterval(Interval);
    }
    catch (TKinokoException &e) {
	throw TScriptException("setPollingInterval(): " + e.Message());
    }

    return TParaValue();
}

TParaValue TKinokoDaqSystemMessenger::ForcePolling(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    int Interval = 0;
    if (ArgumentList.size() > 0) {
	if (! ArgumentList[0]->IsNumeric()) {
	    throw TScriptException(
		"invalid argument[s]: forcePolling(int polling_interval_usec)"
	    );
	}
	Interval = ArgumentList[0]->AsLong();
    }

    try {
	_DaqBuilder->Trigger()->ForcePolling();
	if (Interval > 0) {
	    _DaqBuilder->Trigger()->SetPollingInterval(Interval);
	}
    }
    catch (TKinokoException &e) {
	throw TScriptException("forcePolling(): " + e.Message());
    }

    return TParaValue();
}

TParaValue TKinokoDaqSystemMessenger::ScriptFileName(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    return TParaValue(_ScriptFileName);
}
