/* KinokoReadoutAction.cc */
/* Created by Enomoto Sanshiro on 25 September 1999. */
/* Last updated by Enomoto Sanshiro on 30 September 2002. */


#include <iostream>
#include <string>
#include <vector>
#include <utility>
#include "RoomModule.hh"
#include "RoomCamacAccess.hh"
#include "KinokoDefs.hh"
#include "KinokoStream.hh"
#include "KinokoDataSection.hh"
#include "KinokoTaggedDataSection.hh"
#include "KinokoBlockDataSection.hh"
#include "KinokoIndexedDataSection.hh"
#include "KinokoNestedDataSection.hh"
#include "KinokoReadoutAction.hh"

using namespace std;



TKinokoReadoutAction::TKinokoReadoutAction(void)
{
}

TKinokoReadoutAction::~TKinokoReadoutAction()
{
}

void TKinokoReadoutAction::Initialize(void)
{
}

void TKinokoReadoutAction::Go(TKinokoOutputStream* OutputStream, int& StatusFlag) throw(TKinokoException)
{
    void* DataBuffer = 0;
    int PacketSize = NextPacketSize();

    if (PacketSize > 0) {
	do {
	    OutputStream->NextEntry(DataBuffer, PacketSize);
	} while (DataBuffer == 0);
	
	PacketSize = GoNext(DataBuffer, StatusFlag);
	OutputStream->Flush(DataBuffer, PacketSize);
    }
    else {
	GoNext(DataBuffer, StatusFlag);
    }
}



TKinokoBlockReadAction::TKinokoBlockReadAction(TRoomModule* Module, TKinokoBlockDataSection* DataSection, int Address, int Size)
{
    _Address = Address;
    _Size = Size;

    _Module = Module;
    _DataSection = DataSection;

    _Formatter = 0;
    _DataBufferSize = 0;
}

TKinokoBlockReadAction::~TKinokoBlockReadAction()
{
}

void TKinokoBlockReadAction::Initialize(void)
{
    _Formatter = _DataSection->Formatter();
}

int TKinokoBlockReadAction::NextPacketSize(void)
{
    if (_Size > 0) {
	_DataBufferSize = _Size;
    }
    else {
	_DataBufferSize = _Module->NextDataBlockSize(_Address);
    }

    return _Formatter->PacketSizeFor(_DataBufferSize);
}

int TKinokoBlockReadAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    void* DataArea;
    int DataSize;

    DataArea = _Formatter->DataAreaOf(DataBuffer);
    try {
        DataSize = _Module->BlockRead(_Address, DataArea, _DataBufferSize);
    }
    catch (THardwareException &e) {
         throw TKinokoException(
	    "TKinokoBlockReadAction::GoNext()",
	    "hardware exception: " + e.Message()
        );
    }
    _Formatter->WriteHeaderTo(DataBuffer, DataSize);

    return _Formatter->PacketSizeFor(DataSize);
}

void TKinokoBlockReadAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "BlockRead" << hex << "\t";
    os << _Module->ModuleName() << ", " << _Address << ", " << _Size;
    os << dec << endl;
}



TKinokoSingleReadAction::TKinokoSingleReadAction(TRoomModule* Module, TKinokoIndexedDataSection* DataSection)
{
    _Module = Module;
    _DataSection = DataSection;

    _Formatter = 0;

    _ChannelListCapacity = 32;
    _ChannelListSize = 0;
    _ChannelList = new int[_ChannelListCapacity];
}

TKinokoSingleReadAction::~TKinokoSingleReadAction()
{
    delete[] _ChannelList;
}

void TKinokoSingleReadAction::Initialize(void)
{
    _Formatter = _DataSection->Formatter();
    
    int DataSize = _Formatter->DataSizeFor(_ChannelListSize);
    _DataBufferSize = _Formatter->PacketSizeFor(DataSize);
}

void TKinokoSingleReadAction::AddChannel(int Channel)
{
    if (_ChannelListSize + 1 > _ChannelListCapacity) {
	int* OldChannelList = _ChannelList;

	_ChannelListCapacity *= 2;
	_ChannelList = new int[_ChannelListCapacity];

	for (int i = 0; i < _ChannelListSize; i++) {
	    _ChannelList[i] = OldChannelList[i];
	}

	delete[] OldChannelList;
    }

    _ChannelList[_ChannelListSize] = Channel;
    _ChannelListSize++;
}

int TKinokoSingleReadAction::NextPacketSize(void)
{
    return _DataBufferSize;
}

int TKinokoSingleReadAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    int Index, Channel, Data, DataCount; 

    try {
	DataCount = 0;
	for (Index = 0; Index < _ChannelListSize; Index++) {
	    Channel = _ChannelList[Index];

	    if (_Module->Read(Channel, Data) > 0) {
		_Formatter->WriteTo(DataBuffer, Index, Channel, Data);
		DataCount++;
	    }
	}
    }
    catch (THardwareException &e) {
        throw TKinokoException(
	    "TKinokoSingleReadAction::GoNext()",
	    "hardware exception: " + e.Message()
        );
    }
    _Formatter->WriteHeaderTo(DataBuffer, _Formatter->DataSizeFor(DataCount));

    return _Formatter->PacketSizeFor(_Formatter->DataSizeFor(DataCount));
}

void TKinokoSingleReadAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "SingleRead" << hex << "\t";
    os << _Module->ModuleName() << ", ";
    for (int Index = 0; Index < _ChannelListSize; Index++) {
	if (Index != 0) {
	    os << ":";
	}
	os << _ChannelList[Index];
    }
    os << dec << endl;
}



TKinokoSequentialReadAction::TKinokoSequentialReadAction(TRoomModule* Module, TKinokoIndexedDataSection* DataSection)
{
    _Module = Module;
    _DataSection = DataSection;

    _Formatter = 0;

    _ChannelListCapacity = 32;
    _ChannelListSize = 0;
    _ChannelList = new int[_ChannelListCapacity];

    _BufferLength = 16*1024;
    _Buffer = new int[_BufferLength];
}

TKinokoSequentialReadAction::~TKinokoSequentialReadAction()
{
    delete[] _ChannelList;
    delete[] _Buffer;
}

void TKinokoSequentialReadAction::Initialize(void)
{
    _Formatter = _DataSection->Formatter();
}

void TKinokoSequentialReadAction::AddChannel(int Channel)
{
    if (_ChannelListSize + 1 > _ChannelListCapacity) {
	int* OldChannelList = _ChannelList;

	_ChannelListCapacity *= 2;
	_ChannelList = new int[_ChannelListCapacity];

	for (int i = 0; i < _ChannelListSize; i++) {
	    _ChannelList[i] = OldChannelList[i];
	}

	delete[] OldChannelList;
    }

    _ChannelList[_ChannelListSize] = Channel;
    _ChannelListSize++;
}

int TKinokoSequentialReadAction::NextPacketSize(void)
{
#if 0
    _NumberOfSamples = _Module->NextNumberOfDataElements();
    int NumberOfElements = _NumberOfSamples * _ChannelListSize;
#else
    int NumberOfElements;
    int MaxNumberOfElements = 0;
    int TotalNumberOfElements = 0;
    for (int ChannelIndex = 0; ChannelIndex < _ChannelListSize; ChannelIndex++) {	
	NumberOfElements = (
	    _Module->NextNumberOfDataElements(_ChannelList[ChannelIndex])
	);
	MaxNumberOfElements = max(MaxNumberOfElements, NumberOfElements);
	TotalNumberOfElements += NumberOfElements;
    }
#endif

    int DataSize = _Formatter->DataSizeFor(TotalNumberOfElements);
    _DataBufferSize = _Formatter->PacketSizeFor(DataSize);

    if (MaxNumberOfElements > _BufferLength) {
	_BufferLength = max(MaxNumberOfElements, 2 * _BufferLength);
	delete[] _Buffer;
	_Buffer = new int[_BufferLength];
    }

    return _DataBufferSize;
}

int TKinokoSequentialReadAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    int Index, Channel, DataCount = 0;

    try {
	for (Index = 0; Index < _ChannelListSize; Index++) {
	    Channel = _ChannelList[Index];
	    int DataLength = (
		_Module->SequentialRead(Channel, _Buffer, _BufferLength)
	    );

	    if (DataLength > 0) {
		_Formatter->WriteArrayTo(
		    DataBuffer, DataCount, Channel, _Buffer, DataLength
		);
	    }
	    DataCount += DataLength;
	}
    }
    catch (THardwareException &e) {
        throw TKinokoException(
	    "TKinokoSequentialReadAction::GoNext()",
	    "hardware exception: " + e.Message()
        );
    }
    _Formatter->WriteHeaderTo(DataBuffer, _Formatter->DataSizeFor(DataCount));

    return _Formatter->PacketSizeFor(_Formatter->DataSizeFor(DataCount));
}

void TKinokoSequentialReadAction::Dump(ostream& os, const string& Indent) 
{
    os << Indent << "SequentialRead" << hex << "\t";
    os << _Module->ModuleName() << ", ";
    for (int Index = 0; Index < _ChannelListSize; Index++) {
	if (Index != 0) {
	    os << ":";
	}
	os << _ChannelList[Index];
    }
    os << dec << endl;
}



TKinokoTagReadAction::TKinokoTagReadAction(TRoomModule* Module, TKinokoTaggedDataSection* DataSection)
{
    _Module = Module;
    _DataSection = DataSection;

    _Formatter = 0;

    _ChannelListCapacity = 32;
    _ChannelListSize = 0;
    _ChannelList = new int[_ChannelListCapacity];
    _TagIndexList = new int[_ChannelListCapacity];
}

TKinokoTagReadAction::~TKinokoTagReadAction()
{
    delete[] _TagIndexList;
    delete[] _ChannelList;
}

void TKinokoTagReadAction::AddChannel(int Channel, int TagIndex)
{
    if (_ChannelListSize + 1 > _ChannelListCapacity) {
	int* OldChannelList = _ChannelList;
	int* OldTagIndexList = _TagIndexList;

	_ChannelListCapacity *= 2;
	_ChannelList = new int[_ChannelListCapacity];
	_TagIndexList = new int[_ChannelListCapacity];

	for (int i = 0; i < _ChannelListSize; i++) {
	    _ChannelList[i] = OldChannelList[i];
	    _TagIndexList[i] = OldTagIndexList[i];
	}

	delete[] OldChannelList;
	delete[] OldTagIndexList;
    }

    _ChannelList[_ChannelListSize] = Channel;
    _TagIndexList[_ChannelListSize] = TagIndex;
    _ChannelListSize++;
}

void TKinokoTagReadAction::Initialize(void)
{
    _Formatter = _DataSection->Formatter();
    _DataBufferSize = _Formatter->PacketSizeFor(_Formatter->DataSize());
}

int TKinokoTagReadAction::NextPacketSize(void)
{
    return _DataBufferSize;
}

int TKinokoTagReadAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    int Index, TagIndex, Channel, Data; 

    try {
	for (Index = 0; Index < _ChannelListSize; Index++) {
	    Channel = _ChannelList[Index];
	    TagIndex = _TagIndexList[Index];

	    if (_Module->Read(Channel, Data) == 0) {
		Data = 0;  //... temporary (better than random numbers?)
	    }
	    _Formatter->WriteTo(DataBuffer, TagIndex, Data);
	}
    }
    catch (THardwareException &e) {
        throw TKinokoException(
	    "TKinokoTagReadAction::GoNext()",
	    "hardware exception: " + e.Message()
        );
    }    
    _Formatter->WriteHeaderTo(DataBuffer, _Formatter->DataSize());

    return _Formatter->PacketSizeFor(_Formatter->DataSize());
}

void TKinokoTagReadAction::Dump(ostream& os, const string& Indent) 
{
    os << Indent << "TagRead" << hex << "\t";
    os << _Module->ModuleName() << ", ";
    for (int Index = 0; Index < _ChannelListSize; Index++) {
	if (Index != 0) {
	    os << ":";
	}
	os << _ChannelList[Index] << "/" << _TagIndexList[Index];
    }
    os << dec << endl;
}



TKinokoControlAction::TKinokoControlAction(TRoomModule* Module, int ControlId)
{
    _Module = Module;
    _ControlId = ControlId;

    _ChannelListCapacity = 32;
    _ChannelListSize = 0;
    _ChannelList = new int[_ChannelListCapacity];
}
    
TKinokoControlAction::~TKinokoControlAction()
{
    delete[] _ChannelList;
}

void TKinokoControlAction::AddChannel(int Channel)
{
    if (_ChannelListSize + 1 > _ChannelListCapacity) {
	int* OldChannelList = _ChannelList;

	_ChannelListCapacity *= 2;
	_ChannelList = new int[_ChannelListCapacity];

	for (int i = 0; i < _ChannelListSize; i++) {
	    _ChannelList[i] = OldChannelList[i];
	}

	delete[] OldChannelList;
    }

    _ChannelList[_ChannelListSize] = Channel;
    _ChannelListSize++;
}

void TKinokoControlAction::Transact(int Channel) throw(TKinokoException)
{
    try {
	switch (_ControlId) {
          case ControlId_Enable:
	    _Module->Enable(Channel);
	    break;
          case ControlId_Disable:
	    _Module->Disable(Channel);
	    break;
          case ControlId_Clear:
	    _Module->Clear(Channel);
	    break;
          default:
	    break;
	}
    }
    catch (THardwareException &e) {
         throw TKinokoException(
	    "TControlAction::Transact()",
	    "hardware exception: " + e.Message()
        );
    }
}

int TKinokoControlAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoControlAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    if (_ChannelListSize == 0) {
	Transact(-1);
    }
    else {
	for (int i = 0; i < _ChannelListSize; i++) {
	    Transact(_ChannelList[i]);
	}
    }

    return 0;
}

void TKinokoControlAction::Dump(ostream& os, const string& Indent) 
{
    os << Indent << "CommonControl" << hex << "\t";
    os << _Module->ModuleName() << ", ";
    switch (_ControlId) {
      case ControlId_Enable:
	os << "ENABLE";
	break;
      case ControlId_Disable:
	os << "DISABLE";
	break;
      case ControlId_Clear:
	os << "CLEAR";
	break;
      default:
	;
    }
    os << ", ";

    for (int Index = 0; Index < _ChannelListSize; Index++) {
	if (Index != 0) {
	    os << ":";
	}
	os << _ChannelList[Index];
    }
    os << dec << endl;
}



TKinokoWaitDataAction::TKinokoWaitDataAction(TRoomModule* Module, int* TimeoutRegister, int* ResponseRegister)
{
    _Module = Module;
    _TimeoutRegister = TimeoutRegister;
    _ResponseRegister = ResponseRegister;
}

TKinokoWaitDataAction::~TKinokoWaitDataAction()
{
}

int TKinokoWaitDataAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoWaitDataAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    int Timeout_sec = 1000;
    if (_TimeoutRegister) {
	Timeout_sec = *_TimeoutRegister;
    }

    bool IsDataAvailable;
    try {
        IsDataAvailable = _Module->WaitData(Timeout_sec);
    }
    catch (THardwareException &e) {
	throw TKinokoException(
	    "TKinokoWaitDataAction::GoNext()",
	    "hardware exception: " + e.Message()
	);
    }

    if (_ResponseRegister) {
	*_ResponseRegister = IsDataAvailable ? 1: 0; 
    }

    return 0;
}
    
void TKinokoWaitDataAction::Dump(ostream& os, const string& Indent) 
{
    os << Indent << "WaitData" << "\t" << _Module->ModuleName() << " ";
    os << hex << "[" << _TimeoutRegister << "]";
    os << dec << "(" << *_TimeoutRegister << ")" << ", ";
    os << hex << "[" << _ResponseRegister << "]" << dec << endl;
}



TKinokoWriteRegisterAction::TKinokoWriteRegisterAction(TRoomModule* Module, int* AddressRegister, int* DataRegister)
{
    _Module = Module;

    _AddressRegister = AddressRegister;
    _DataRegister = DataRegister;
}

TKinokoWriteRegisterAction::~TKinokoWriteRegisterAction()
{
}

int TKinokoWriteRegisterAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoWriteRegisterAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    try {
        _Module->WriteRegister(*_AddressRegister, *_DataRegister);
    }
    catch (THardwareException &e) {
         throw TKinokoException(
	    "TKinokoWriteRegisterAction::GoNext()",
	    "hardware exception: " + e.Message()
        );
    }

    return 0;
}
    
void TKinokoWriteRegisterAction::Dump(ostream& os, const string& Indent) 
{
    os << Indent << "WriteRegister" << hex << "\t";
    os << _Module->ModuleName() << ", ";
    os << hex << "[" << _AddressRegister << "]";
    os << dec << "(" << *_AddressRegister << ")" << ", ";
    os << hex << "[" << _DataRegister << "]";
    os << dec << "(" << *_DataRegister << ")" << endl;
}



TKinokoReadRegisterAction::TKinokoReadRegisterAction(TRoomModule* Module, int* AddressRegister, int* DataRegister)
{
    _Module = Module;

    _AddressRegister = AddressRegister;
    _DataRegister = DataRegister;
}

TKinokoReadRegisterAction::~TKinokoReadRegisterAction()
{
}

int TKinokoReadRegisterAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoReadRegisterAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    try {
        _Module->ReadRegister(*_AddressRegister, *_DataRegister);
    }
    catch (THardwareException &e) {
         throw TKinokoException(
	    "TKinokoReadRegisterAction::GoNext()",
	    "hardware exception: " + e.Message()
        );
    }

    return 0;
}

void TKinokoReadRegisterAction::Dump(ostream& os, const string& Indent) 
{
    os << Indent << "ReadRegister" << hex << "\t";
    os << _Module->ModuleName() << ", ";
    os << "[" << _AddressRegister << "]";
    os << dec << "(" << *_AddressRegister << ")" << ", ";
    os << hex << "[" << _DataRegister << "]";
    os << dec << endl;
}



TKinokoMiscControlAction::TKinokoMiscControlAction(TRoomModule* Module, int ControlId, const vector<int*>& ParameterRegisterList, const std::string& ControlName)
{
    _Module = Module;
    _ControlId = ControlId;
    _ControlName = ControlName;
    
    _NumberOfParameters = ParameterRegisterList.size();
    _ParameterRegisterList = new int* [_NumberOfParameters];
    _ParameterList = new int [_NumberOfParameters];

    for (int i = 0; i < _NumberOfParameters; i++) {
	_ParameterRegisterList[i] = ParameterRegisterList[i];
    }
}
    
TKinokoMiscControlAction::~TKinokoMiscControlAction()
{
    delete[] _ParameterRegisterList;
    delete[] _ParameterList;
}

int TKinokoMiscControlAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoMiscControlAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    for (int i = 0; i < _NumberOfParameters; i++) {
	_ParameterList[i] = *_ParameterRegisterList[i];
    }

    try {
	_Module->MiscControl(
	    _ControlId, _ParameterList, _NumberOfParameters
	);
    }
    catch (THardwareException &e) {
	throw TKinokoException(
	    "TMiscControlAction::GoNext()",
	    "hardware exception: " + e.Message()
        );
    }

    for (int i = 0; i < _NumberOfParameters; i++) {
	*_ParameterRegisterList[i] = _ParameterList[i];
    }

    return 0;
}

void TKinokoMiscControlAction::Dump(ostream& os, const string& Indent) 
{
    os << Indent << "MiscControl" << "\t";
    os << _Module->ModuleName() << ", ";
    os << _ControlId << "(" << _ControlName << ")";
    for (int Index = 0; Index < _NumberOfParameters; Index++) {
	os << hex << ", [" << _ParameterRegisterList[Index] << "]";
	os << dec << "(" << *_ParameterRegisterList[Index] << ")";
    }
    os << dec << endl;
}



TKinokoCamacControllerAction::TKinokoCamacControllerAction(TRoomCamacController* Controller, const string& ControllerName, int ActionId)
{
    _CamacController = Controller;
    _ControllerName = ControllerName;
    _ActionId = ActionId;
}

TKinokoCamacControllerAction::~TKinokoCamacControllerAction()
{
}

int TKinokoCamacControllerAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoCamacControllerAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    switch (_ActionId) {
      case CamacControllerAction_Initialize:
	_CamacController->Initialize();
	break;
      case CamacControllerAction_Clear:
	_CamacController->Clear();
	break;
      case CamacControllerAction_SetInhibit:
	_CamacController->SetInhibit();
	break;
      case CamacControllerAction_ReleaseInhibit:
	_CamacController->ReleaseInhibit();
	break;
      default:
        ;
    }

    return 0;
}

void TKinokoCamacControllerAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "CamacControllerAction" << "\t";
    os << _ControllerName << ", ";  

    switch (_ActionId) {
      case CamacControllerAction_Initialize:
	os << "INITIALIZE";
	break;
      case CamacControllerAction_Clear:
	os << "CLEAR";
	break;
      case CamacControllerAction_SetInhibit:
	os << "SET_INHIBIT";
	break;
      case CamacControllerAction_ReleaseInhibit:
	os << "RELEASE_INHIBIT";
	break;
      default:
	;
    }
    os << endl;
}



TKinokoCamacTransactAction::TKinokoCamacTransactAction(TRoomCamacModule* CamacModule, int Function, int Address, int* DataRegister, int* QResponseRegister, int* XResponseRegister)
{
    _CamacModule = CamacModule;
    _Function = Function;
    _Address = Address;
    _DataRegister = DataRegister;
    _QResponseRegister = QResponseRegister;
    _XResponseRegister = XResponseRegister;

    if (_DataRegister == 0) {
	_DataRegister = &_DummyDataRegister;
    }
    if (_QResponseRegister == 0) {
	_QResponseRegister = &_DummyQResponseRegister;
    }
    if (_XResponseRegister == 0) {
	_XResponseRegister = &_DummyXResponseRegister;
    }
}

TKinokoCamacTransactAction::~TKinokoCamacTransactAction()
{
}

int TKinokoCamacTransactAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoCamacTransactAction::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    int Data = *_DataRegister;
    try {
        _CamacModule->Transact(
	    _Function, _Address, Data, 
	    *_QResponseRegister, *_XResponseRegister
	);
    }
    catch (THardwareException &e) {
         throw TKinokoException(
	    "TKinokoCamacTransactAction::GoNext()",
	    "hardware exception: " + e.Message()
        );
    }

    if (_Function < 8) {
        *_DataRegister = Data;
    }

    return 1;
}

void TKinokoCamacTransactAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "CamacTransact" << "\t";
    os << _CamacModule->ModuleName() << ", F" << _Function << ", A" << _Address;

    if (_DataRegister != &_DummyDataRegister) {
	os << hex << ", [" << _DataRegister << "]";
    }
    if (_QResponseRegister != &_DummyQResponseRegister) {
	os << hex << ", [" << _QResponseRegister << "]";
    }
    if (_XResponseRegister != &_DummyXResponseRegister) {
	os << hex << ", [" << _XResponseRegister << "]";
    }

    os << dec << endl;
}
