/* ReadoutSequence.cc */
/* Created by Enomoto Sanshiro on 25 September 1999. */
/* Last updated by Enomoto Sanshiro on 6 November 2001. */


#include <vector>
#include "KinokoStream.hh"
#include "KinokoReadoutAction.hh"
#include "KinokoReadoutSequence.hh"

using namespace std;



TKinokoReadoutSequence::TKinokoReadoutSequence(void)
{
    _ActionListCapacity = 64;
    _ActionList = new TKinokoReadoutAction* [_ActionListCapacity];
    _ActionListSize = 0;

    _ArgumentRegisterList = 0;
    _ArgumentListSize = 0;
}

TKinokoReadoutSequence::TKinokoReadoutSequence(int* ArgumentRegister)
{
    _ActionListCapacity = 64;
    _ActionList = new TKinokoReadoutAction* [_ActionListCapacity];
    _ActionListSize = 0;

    _ArgumentListSize = 1;
    _ArgumentRegisterList = new int* [1];
    _ArgumentRegisterList[0] = ArgumentRegister;
}

TKinokoReadoutSequence::TKinokoReadoutSequence(std::vector<int*> ArgumentRegisterList)
{
    _ActionListCapacity = 64;
    _ActionList = new TKinokoReadoutAction* [_ActionListCapacity];
    _ActionListSize = 0;

    _ArgumentListSize = ArgumentRegisterList.size();
    _ArgumentRegisterList = new int* [_ArgumentListSize];

    for (int i = 0; i < _ArgumentListSize; i++) {
	_ArgumentRegisterList[i] = ArgumentRegisterList[i];
    }
}

TKinokoReadoutSequence::~TKinokoReadoutSequence()
{
    delete[] _ArgumentRegisterList;

    for (unsigned i = 0; i < _ActionListSize; i++) {
        delete _ActionList[i];
    }
    delete[] _ActionList;
}

void TKinokoReadoutSequence::AddReadoutAction(TKinokoReadoutAction* ReadoutAction)
{
    if (_ActionListSize + 1 > _ActionListCapacity) {
	_ActionListCapacity *= 2;
	TKinokoReadoutAction** OldActionList = _ActionList;
	_ActionList = new TKinokoReadoutAction* [_ActionListCapacity];
	for (unsigned i = 0; i < _ActionListSize; i++) {
	    _ActionList[i] = OldActionList[i];
	}
	delete[] OldActionList;
    }

    _ActionList[_ActionListSize] = ReadoutAction;
    _ActionListSize++;
}

void TKinokoReadoutSequence::Initialize(void)
{
    for (unsigned i = 0; i < _ActionListSize; i++) {
        _ActionList[i]->Initialize();
    }
}

void TKinokoReadoutSequence::Go(TKinokoOutputStream* OutputStream, int& StatusFlag) throw(TKinokoException)
{
    for (unsigned i = 0; i < _ActionListSize; i++) {
	_ActionList[i]->Go(OutputStream, StatusFlag);
	if (StatusFlag & TKinokoReadoutAction::Status_Break) {
	    break;
	}
    }
}

// The following method might be called by TKinokoUnitedReadoutActions::Go().
int TKinokoReadoutSequence::NextPacketSize(void)
{
    int DataSize = 0;
    for (unsigned i = 0; i < _ActionListSize; i++) {
	DataSize += _ActionList[i]->NextPacketSize();
    }
    
    return DataSize;
}

// The following method might be called by TKinokoUnitedReadoutActions::Go().
int TKinokoReadoutSequence::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    int ThisDataSize, DataSize = 0;

    for (unsigned i = 0; i < _ActionListSize; i++) {
	ThisDataSize = _ActionList[i]->GoNext(
	    (U8bit*) DataBuffer + DataSize, StatusFlag
	);
	DataSize += ThisDataSize;
	if (StatusFlag & TKinokoReadoutAction::Status_Break) {
	    break;
	}
    }

    return DataSize;
}

void TKinokoReadoutSequence::LoadArguments(int* ArgumentList, int NumberOfArguments)
{
    if (NumberOfArguments > _ArgumentListSize) {
	NumberOfArguments = _ArgumentListSize;
    }

    for (int i = 0; i < NumberOfArguments; i++) {
	*_ArgumentRegisterList[i] = ArgumentList[i];
    }
}

void TKinokoReadoutSequence::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence";
    if (_ArgumentListSize > 0) {
	os << " loads";
	for (int i = 0; i < _ArgumentListSize; i++) {
	    os << " [" << _ArgumentRegisterList[i] << "]";
	}
    }
    os << endl;

    for (unsigned i = 0; i < _ActionListSize; i++) {
	_ActionList[i]->Dump(os, Indent + "->\t");
    }

    os << Indent << ".end" << endl;
}



TKinokoUnitedReadoutActions::TKinokoUnitedReadoutActions(TKinokoNestedDataSection* DataSection)
{
    _DataSection = DataSection;
    _Formatter = 0;
}

TKinokoUnitedReadoutActions::~TKinokoUnitedReadoutActions()
{
}

void TKinokoUnitedReadoutActions::Initialize(void)
{
    _Formatter = _DataSection->Formatter();
    TKinokoReadoutSequence::Initialize();
}

void TKinokoUnitedReadoutActions::Go(TKinokoOutputStream* OutputStream, int& StatusFlag) throw(TKinokoException)
{
    TKinokoReadoutAction::Go(OutputStream, StatusFlag);
}

int TKinokoUnitedReadoutActions::NextPacketSize(void)
{
    int DataSize = TKinokoReadoutSequence::NextPacketSize();
    
    return _Formatter->PacketSizeFor(DataSize);
}

int TKinokoUnitedReadoutActions::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    void* SubDataBuffer = _Formatter->DataAreaOf(DataBuffer);
    int DataSize = TKinokoReadoutSequence::GoNext(SubDataBuffer, StatusFlag);

    _Formatter->WriteHeaderTo(DataBuffer, DataSize);
    
    return _Formatter->PacketSizeFor(DataSize);
}

void TKinokoUnitedReadoutActions::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence united" << endl;

    for (unsigned i = 0; i < _ActionListSize; i++) {
	_ActionList[i]->Dump(os, Indent + "->\t");
    }

    os << Indent << ".end" << endl;
}



TKinokoConditionalReadoutSequence::TKinokoConditionalReadoutSequence(TKinokoReadoutCondition* Condition)
{
    _Condition = Condition;
}

TKinokoConditionalReadoutSequence::~TKinokoConditionalReadoutSequence()
{
    delete _Condition;
}

void TKinokoConditionalReadoutSequence::Go(TKinokoOutputStream* OutputStream, int& StatusFlag) throw(TKinokoException)
{
    if (_Condition->IsSatisfied()) {
	TKinokoReadoutSequence::Go(OutputStream, StatusFlag);
    }
}

// The following method might be called by TKinokoUnitedReadoutActions::Go().
int TKinokoConditionalReadoutSequence::NextPacketSize(void)
{
    _IsConditionSatisfied = _Condition->IsSatisfied();
    if (_IsConditionSatisfied) {
	return TKinokoReadoutSequence::NextPacketSize();
    }
    else {
	return 0;
    }
}

// The following method might be called by TKinokoUnitedReadoutActions::Go().
int TKinokoConditionalReadoutSequence::GoNext(void* DataBuffer, int& StatusFlag) throw(TKinokoException)
{
    if (_IsConditionSatisfied) {
	return TKinokoReadoutSequence::GoNext(DataBuffer, StatusFlag);
    }
    else {
	return 0;
    }
}

void TKinokoConditionalReadoutSequence::Dump(ostream& os, const string& Indent)
{
    os << Indent << ".sequence conditional (";
    _Condition->Dump(os);
    os << ")" << endl;

    for (unsigned i = 0; i < _ActionListSize; i++) {
	_ActionList[i]->Dump(os, Indent + "->\t");
    }

    os << Indent << ".end" << endl;
}

