/* KinokoSocketStream.cc */
/* Created by Enomoto Sanshiro on 24 February 1998. */
/* Last updated by Enomoto Sanshiro on 17 April 2001. */


#include "MushDecoratedSocket.hh"
#include "KinokoStream.hh"
#include "KinokoSocketStream.hh"

using namespace std;



TKinokoInputSocketStream::TKinokoInputSocketStream(TMushFramedSocket *Socket, int LocalBufferSize, int LocalBufferEntryTableSize) throw(TKinokoException)
: TKinokoInputStream(LocalBufferSize, LocalBufferEntryTableSize)
{
    _Socket = Socket;

    if (_Socket->Status() != TMushSocket::ssOpened) {
	try {
	    _Socket->Open();
	}
	catch (TSystemCallException &e) {
            throw TKinokoException(
	        "TKinokoInputSocketStream::TKinokoInputSocketStream()",
	        "system call exception: " + e.Message()
            );
	}
    }

    _SocketSelector = new TMushSocketSelector();
    _SocketSelector->AddSocket(Socket);
}

TKinokoInputSocketStream::~TKinokoInputSocketStream() 
{
    delete _SocketSelector;
}

void TKinokoInputSocketStream::Destroy(void)
{
    _SocketSelector->RemoveSocket(_Socket);
    _Socket->Close();

    delete _Socket;
    _Socket = 0;
}

bool TKinokoInputSocketStream::HasData(void)
{
    TMushSocket* Socket = _SocketSelector->WaitEvent().first;

    return (Socket != 0);
}

size_t TKinokoInputSocketStream::NextDataSize(void)
{
    size_t Result = _Socket->NextPacketSize();
    if (Result < 0) {
	Result = 0;
    }

    return Result;
}

int TKinokoInputSocketStream::Read(void *Address, size_t MaxSize) throw(TKinokoException)
{
    int ReadLength;

    try {
	ReadLength = _Socket->Receive(Address, MaxSize);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoInputSocketStream::Read()",
	    "system call exception: " + e.Message()
        );
    }
    if (ReadLength < 0) {
	// interrupted system call
	ReadLength = 0;
    }

    return ReadLength;
}

int TKinokoInputSocketStream::NextEntry(void *&Address) throw(TKinokoException)
{
    size_t NextPacketSize = (size_t) _Socket->NextPacketSize();

    if (NextPacketSize <= 0) {
	Address = 0;
	return 0;
    }

    if (NextPacketSize > _WorkAreaSize) {
	ResizeWorkArea(NextPacketSize);
    }

    int ReadLength;
    try {
	ReadLength = _Socket->Receive(_WorkArea, NextPacketSize);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TInputFileStream::NextEntry()",
	    "system call exception: " + e.Message()
	);
    }

    if (ReadLength < 0) {
	// interrupted system call
	Address = 0;
	ReadLength = 0;
    }
    else {
	Address = _WorkArea;
    }

    return ReadLength;
}

void TKinokoInputSocketStream::Flush(void* Address) throw(TKinokoException)
{
}

int TKinokoInputSocketStream::GetEntry(void *&Address) throw(TKinokoException)
{
    if (_LocalBuffer == 0) {
	CreateLocalBuffer();
    }

    int PacketSize = _Socket->NextPacketSize();
    if (PacketSize <= 0) {
	Address = 0;
	return 0;
    }

    if (! _LocalBuffer->Allocate(Address, PacketSize)) {
	_IsLocalBufferFull = true;
	Address = 0;
	return 0;
    }

    int ReadLength;
    try {
	ReadLength = _Socket->Receive(Address, PacketSize);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoInputSocketStream::GetEntry()",
	    "system call exception: " + e.Message()
        );
    }
    if (ReadLength < 0) {
	// interrupted system call
	ReadLength = 0;
    }

    return ReadLength;
}

void TKinokoInputSocketStream::ReleaseEntry(void* Address) throw(TKinokoException)
{
    if (Address != 0) {
	_LocalBuffer->Release(Address);
	_IsLocalBufferFull = false;
    }
}



TKinokoOutputSocketStream::TKinokoOutputSocketStream(TMushFramedSocket *Socket, int LocalBufferSize, int LocalBufferEntryTableSize) throw(TKinokoException)
: TKinokoOutputStream(LocalBufferSize, LocalBufferEntryTableSize)
{
    _Socket = Socket;

    if (_Socket->Status() != TMushSocket::ssOpened) {
	try {
	    _Socket->Open();
	}
	catch (TSystemCallException &e) {
            throw TKinokoException(
	        "TKinokoOutputSocketStream::TKinokoOutputSocketStream()",
	        "system call exception: " + e.Message()
            );
	}
    }
}

TKinokoOutputSocketStream::~TKinokoOutputSocketStream() 
{
}

void TKinokoOutputSocketStream::Destroy(void)
{
    _Socket->Close();

    delete _Socket;
    _Socket = 0;
}

int TKinokoOutputSocketStream::Write(void *Address, size_t Size) throw(TKinokoException)
{
    int WrittenLength;

    try {
	WrittenLength = _Socket->Send(Address, Size);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoOutputSocketStream::Write()",
	    "system call exception: " + e.Message()
        );
    }

    return WrittenLength;
}

int TKinokoOutputSocketStream::NextEntry(void *&Address, size_t MaxSize) throw(TKinokoException)
{
    if (MaxSize > _WorkAreaSize) {
	ResizeWorkArea(MaxSize);
    }
    Address = _WorkArea;

    return MaxSize;
}

int TKinokoOutputSocketStream::Flush(void* Address, size_t Size) throw(TKinokoException)
{
    int WrittenLength;

    try {
        WrittenLength = _Socket->Send(_WorkArea, Size);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TOutputSocketStream::Flush()",
	    "system call exception: " + e.Message()
        );
    }
    
    return WrittenLength;
}

int TKinokoOutputSocketStream::GetEntry(void *&Address, size_t MaxSize) throw(TKinokoException)
{
    if (_LocalBuffer == 0) {
	CreateLocalBuffer();
    }

    if (! _LocalBuffer->Allocate(Address, MaxSize)) {
	_IsLocalBufferFull = true;
	Address = 0;
	return 0;
    }

    return MaxSize;
}

int TKinokoOutputSocketStream::ReleaseEntry(void* Address, size_t Size) throw(TKinokoException)
{
    if (Address == 0) {
	return 0;
    }

    int WrittenLength;
    try {
        WrittenLength = _Socket->Send(Address, Size);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoOutputSocketStream::Flush()",
	    "system call exception: " + e.Message()
        );
    }
    
    _LocalBuffer->Release(Address);
    _IsLocalBufferFull = false;

    return WrittenLength;
}
