/* KinokoViewSequence.cc */
/* Created by Enomoto Sanshiro on 16 June 2002. */
/* Last updated by Enomoto Sanshiro on 16 June 2002. */


#include <string>
#include <map>
#include "MushMisc.hh"
#include "KinokoView.hh"
#include "KinokoViewAction.hh"
#include "KinokoViewSequence.hh"

using namespace std;


TKinokoViewSequence::TKinokoViewSequence(void)
{
    _ActionListCapacity = 64;
    _ActionList = new TKinokoViewAction* [_ActionListCapacity];
    _ActionListSize = 0;
}

TKinokoViewSequence::~TKinokoViewSequence()
{
    for (int i = 0; i < _ActionListSize; i++) {
	delete _ActionList[i];
    }
    
    delete[] _ActionList;
}

void TKinokoViewSequence::AddAction(TKinokoViewAction* Action)
{
    if (_ActionListCapacity == _ActionListSize) {
	TKinokoViewAction** OldActionList = _ActionList;
	_ActionListCapacity *= 2;
	_ActionList = new TKinokoViewAction* [_ActionListCapacity];
	for (int i = 0; i < _ActionListSize; i++) {
	     _ActionList[i] = OldActionList[i];
	}
	delete[] OldActionList;
    }

    _ActionList[_ActionListSize] = Action;
    _ActionListSize++;
}

void TKinokoViewSequence::Execute(void) throw(TKinokoException)
{
    for (int i = 0; i < _ActionListSize; i++) {
	_ActionList[i]->Execute();
    }
}

void TKinokoViewSequence::Dump(ostream& os, const std::string& Indent)
{
    for (int i = 0; i < _ActionListSize; i++) {
	_ActionList[i]->Dump(os, Indent);
    }
}



TKinokoViewSequenceTable::TKinokoViewSequenceTable(void)
{
}

TKinokoViewSequenceTable::~TKinokoViewSequenceTable()
{
    map<string, TKinokoViewSequence*>::iterator SequenceIterator;
    for (
        SequenceIterator = _CommandSequenceTable.begin();
        SequenceIterator != _CommandSequenceTable.end();
        SequenceIterator++
    ){
        delete (*SequenceIterator).second;
    }
}

void TKinokoViewSequenceTable::AddTimeSequence(long Interval_sec, TKinokoViewSequence* Sequence)
{
    _TimeSequenceTable.push_back(make_pair(Interval_sec, Sequence));

    long PresentTime = TMushDateTime::SecSinceEpoch();
    _NextExecutionTimeTable.push_back(PresentTime);
}

void TKinokoViewSequenceTable::AddTrapSequence(int TrapId, TKinokoViewSequence* Sequence)
{
    _TrapSequenceTable[TrapId] = Sequence;
}

void TKinokoViewSequenceTable::AddCommandSequence(const string& Command, TKinokoViewSequence* Sequence)
{
    _CommandSequenceTable[Command] = Sequence;
}

int TKinokoViewSequenceTable::ExecuteTimeSequence(void) throw(TKinokoException)
{
    int Result = 0;

    long PresentTime = TMushDateTime::SecSinceEpoch();
    for (unsigned i = 0; i < _TimeSequenceTable.size(); i++) {
	long NextTime = _NextExecutionTimeTable[i];
	if (NextTime <= PresentTime) {
	    _TimeSequenceTable[i].second->Execute();

	    long Interval = _TimeSequenceTable[i].first;
	    NextTime += Interval * ((PresentTime - NextTime) / Interval + 1);
	    _NextExecutionTimeTable[i] = NextTime;

	    Result++;
	}
    }

    return Result;
}

int TKinokoViewSequenceTable::ExecuteTrapSequence(int TrapId) throw(TKinokoException)
{
    int Result = 0;

    if (_TrapSequenceTable.count(TrapId)) {
	_TrapSequenceTable[TrapId]->Execute();
	Result = 1;
    }
    
    return Result;
}

int TKinokoViewSequenceTable::ExecuteCommandSequence(const string& Command) throw(TKinokoException)
{
    int Result = 0;

    if (_CommandSequenceTable.count(Command)) {
	_CommandSequenceTable[Command]->Execute();
	Result = 1;
    }
    
    return Result;
}

void TKinokoViewSequenceTable::Dump(ostream& os)
{
    for (unsigned i = 0; i < _TimeSequenceTable.size(); i++) {
        os << "time (" << _TimeSequenceTable[i].first << "):" << endl;
	os << "\t.sequence" << endl;
        _TimeSequenceTable[i].second->Dump(os, "\t->\t");
	os << "\t.end" << endl << endl;
    }

    map<int, TKinokoViewSequence*>::iterator TrapSequenceIterator;
    for (
        TrapSequenceIterator = _TrapSequenceTable.begin();
        TrapSequenceIterator != _TrapSequenceTable.end();
        TrapSequenceIterator++
    ){
	os << "trap ";
        int TrapId = (*TrapSequenceIterator).first;
	switch (TrapId) {
          case TrapId_Construct:
	    os << "CONSTRUCT:" << endl;
	    break;
          case TrapId_Destruct:
	    os << "DESTRUCT:" << endl;
	    break;
          case TrapId_RunBegin:
	    os << "RUN_BEGIN:" << endl;
	    break;
          case TrapId_RunEnd:
	    os << "RUN_END:" << endl;
	    break;
          case TrapId_RunSuspend:
            os << "RUN_SUSPEND:" << endl;
	    break;
          case TrapId_RunResume:
	    os << "RUN_RESUME:" << endl;
	    break;
          case TrapId_PreClear:
	    os << "PRE_CLEAR:" << endl;
	    break;
          case TrapId_PostClear:
	    os << "POST_CLEAR:" << endl;
	    break;

          default:
	    os << "UNKNOWN (" << TrapId << "):" << endl;
	}
	os << "\t.sequence" << endl;
        (*TrapSequenceIterator).second->Dump(os, "\t->\t");
	os << "\t.end" << endl << endl;
    }

    map<string, TKinokoViewSequence*>::iterator CommandSequenceIterator;
    for (
        CommandSequenceIterator = _CommandSequenceTable.begin();
        CommandSequenceIterator != _CommandSequenceTable.end();
        CommandSequenceIterator++
    ){
	os << "command \"" << (*CommandSequenceIterator).first << "\":";
	os << endl;
	os << "\t.sequence" << endl;
        (*CommandSequenceIterator).second->Dump(os, "\t->\t");
	os << "\t.end" << endl << endl;
    }
}
