/* kdfconvert.cc */
/* Created by Enomoto Sanshiro on 1 February 2002. */
/* Last updated by Enomoto Sanshiro on 21 Fabruary 2002. */


#include <cstdlib>
#include <iostream>
#include <string>
#include "MushFile.hh"
#include "MushFileSystem.hh"
#include "MushArgumentList.hh"
#include "KinokoKdfStorage.hh"

using namespace std;


class TKinokoKdfConvert {
  public:
    TKinokoKdfConvert(TMushArgumentList& ArgumentList);
    virtual ~TKinokoKdfConvert();
    virtual void Start(void) throw(TKinokoException);
  protected:
    virtual void Construct(void) throw(TKinokoException);
    virtual void Destruct(void) throw(TKinokoException);
  protected:
    string _InputFileName;
    string _OutputFileName;
    TKinokoStorage* _InputStorage;
    TKinokoStorage* _OutputStorage;
    TKinokoInputFileStream* _InputStream;
    TKinokoOutputFileStream* _OutputStream;
    bool _IsDataCompressionEnabled;
    bool _IsIndexEnabled;
    bool _IsOverWriteAllowed;
};



TKinokoKdfConvert::TKinokoKdfConvert(TMushArgumentList& ArgumentList)
{
    _InputFileName = ArgumentList[0];
    _OutputFileName = ArgumentList[1];

    if (ArgumentList.IsOptionSpecified("--compress")) {
	_IsDataCompressionEnabled = true;
    }
    else {
	_IsDataCompressionEnabled = false;
    }

    if (ArgumentList.IsOptionSpecified("--make-index")) {
	_IsIndexEnabled = true;
    }
    else {
	_IsIndexEnabled = false;
    }

    if (ArgumentList.IsOptionSpecified("-f")) {
	_IsOverWriteAllowed = true;
    }
    else {
	_IsOverWriteAllowed = false;
	if (TMushFileAttribute(_OutputFileName).IsReadable()) {
	    throw TKinokoException(
		"file already exists: " + _OutputFileName + "\n" +
		"use -f option to overwrite"
	    );
	}
    }

    _InputStorage = 0;
    _OutputStorage = 0;

    _InputStream = 0;
    _OutputStream = 0;
}

TKinokoKdfConvert::~TKinokoKdfConvert()
{
    delete _InputStorage;
    delete _OutputStorage;
}

void TKinokoKdfConvert::Start(void) throw(TKinokoException)
{
    Construct();

    void* Packet;
    int PacketSize;
    while ((PacketSize = _InputStream->NextEntry(Packet)) > 0) {
	_OutputStream->Write(Packet, PacketSize);
	_InputStream->Flush(Packet);
    }

    Destruct();
}

void TKinokoKdfConvert::Construct(void) throw(TKinokoException)
{
    _InputStorage = new TKinokoKdfStorage(_InputFileName);
    _OutputStorage = new TKinokoKdfStorage(_OutputFileName);

    if (_IsDataCompressionEnabled) {
	_OutputStorage->EnableDataCompression();
    }
    if (_IsIndexEnabled) {
	_OutputStorage->EnableIndex();
    }
    if (_IsOverWriteAllowed) {
	_OutputStorage->AllowOverWrite();
    }

    TKinokoStorageHeader StorageHeader;
    try {
        _InputStorage->ReadHeader(StorageHeader);
    }
    catch (TKinokoException &e) {
        /* version number mismatch */
        cout << endl; 
	cout << "WARNING: " << e << endl << endl;
    }
    _OutputStorage->WriteHeader(StorageHeader);

    try {
        _InputStream = _InputStorage->GetInputStream();
    }
    catch (TKinokoException &e) {
	throw e;
    }
    _OutputStream = _OutputStorage->GetOutputStream();
}

void TKinokoKdfConvert::Destruct(void) throw(TKinokoException)
{
}



int main(int argc, char** argv)
{
    TMushArgumentList ArgumentList(argc, argv);

    if (ArgumentList.NumberOfParameters() < 2) {
	cerr << "Usage: " << argv[0];
	cerr << " [Options] InputFileName OutputFileName" << endl;
	cerr << "Options:" << endl;
	cerr << "  --compress  compress" << endl;
	cerr << "  --make-index  make index" << endl;
	cerr << "  -f  allow file overwrite" << endl;
        return EXIT_FAILURE;
    }

    try {
	TKinokoKdfConvert(ArgumentList).Start();
    }
    catch (TKinokoException& e) {
	cerr << "ERROR: " << e << endl;
        return EXIT_FAILURE;
    }

    return 0;
}
