/* kdfextract.cc */
/* Created by Enomoto Sanshiro on 4 October 2002. */
/* Last updated by Enomoto Sanshiro on 2 August 2007. */


#include <cstdlib>
#include <cstring>
#include <iostream>
#include <string>
#include "MushFile.hh"
#include "MushFileSystem.hh"
#include "MushArgumentList.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataProcessor.hh"

using namespace std;


class TKinokoKdfExtract {
  public:
    TKinokoKdfExtract(const TMushArgumentList& ArgumentList);
    virtual ~TKinokoKdfExtract();
    virtual void Start(void) throw(TKinokoException);
  protected:
    string _InputFileName;
    bool _IsOverWriteAllowed;
    TKinokoStorage* _InputStorage;
    TKinokoStorage *_OutputStorage_RunBegin;
    TKinokoStorage *_OutputStorage_RunEnd;
    TKinokoStorageHeader* _InputStorageHeader;
    TKinokoInputStream* _InputStream;
    TKinokoOutputStream* _OutputStream_RunBegin;
    TKinokoOutputStream* _OutputStream_RunEnd;
    map<int, int> _DescriptorCounts;
};



TKinokoKdfExtract::TKinokoKdfExtract(const TMushArgumentList& ArgumentList)
{
    _InputFileName = ArgumentList[0];
    _IsOverWriteAllowed = ArgumentList.IsOptionSpecified("-f");

    _InputStorage = 0;

    _OutputStream_RunBegin = 0;
    _OutputStream_RunEnd = 0;

    _OutputStorage_RunBegin = 0;
    _OutputStorage_RunEnd = 0;
}

TKinokoKdfExtract::~TKinokoKdfExtract()
{
    delete _InputStorageHeader;
    delete _InputStorage;

    delete _OutputStorage_RunBegin;
    delete _OutputStorage_RunEnd;
}

void TKinokoKdfExtract::Start(void) throw(TKinokoException)
{
    _InputStorage = new TKinokoKdfStorage(_InputFileName);
    _InputStorageHeader = new TKinokoStorageHeader();

    try {
        _InputStorage->ReadHeader(*_InputStorageHeader);
    }
    catch (TKinokoException &e) {
        /* version number mismatch ? */
	cout << "WARNING: " << e << endl << endl;
    }
    _InputStream = _InputStorage->GetInputStream();

    TMushFileAttribute InputFileAttr(_InputFileName);
    string OutputFileName_RunBegin = InputFileAttr.FileRootName() + "-RunBegin." + InputFileAttr.Extension();
    string OutputFileName_RunEnd = InputFileAttr.FileRootName() + "-RunEnd." + InputFileAttr.Extension();
    cout << "making " << OutputFileName_RunBegin << "..." << endl;
    cout << "making " << OutputFileName_RunEnd << "..." << endl;

    if (! _IsOverWriteAllowed) {
	if (TMushFileAttribute(OutputFileName_RunBegin).IsReadable()) {
	    throw TKinokoException(
		"file already exists: " + OutputFileName_RunBegin + "\n" +
		"use -f option to overwrite"
	    );
	}
	if (TMushFileAttribute(OutputFileName_RunEnd).IsReadable()) {
	    throw TKinokoException(
		"file already exists: " + OutputFileName_RunEnd + "\n" +
		"use -f option to overwrite"
	    );
	}
    }

    _OutputStorage_RunBegin = new TKinokoKdfStorage(OutputFileName_RunBegin);
    _OutputStorage_RunEnd = new TKinokoKdfStorage(OutputFileName_RunEnd, true);

    if (_IsOverWriteAllowed) {
	_OutputStorage_RunBegin->AllowOverWrite();
	_OutputStorage_RunEnd->AllowOverWrite();
    }
    if (_InputStorage->IsDataCompressionEnabled()) {
	_OutputStorage_RunBegin->EnableDataCompression();
	_OutputStorage_RunEnd->EnableDataCompression();
    }

    _OutputStorage_RunBegin->WriteHeader(*_InputStorageHeader);
    _OutputStream_RunBegin = _OutputStorage_RunBegin->GetOutputStream();
    _OutputStream_RunEnd = _OutputStorage_RunEnd->GetOutputStream();

    void* Packet;
    int PacketSize;
    while ((PacketSize = _InputStream->NextEntry(Packet)) > 0) {
	//... correct byte order ...
	long DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);

	if (TKinokoDataStreamScanner::IsDataDescriptorPacket(Packet)) {
	    _OutputStream_RunBegin->Write(Packet, PacketSize);
	    _DescriptorCounts[DataSourceId] += 1;
	}
	else if (TKinokoDataStreamScanner::IsCommandPacket(Packet)) {
	    int CommandValue = TKinokoDataStreamScanner::CommandValueOf(Packet);

	    if (CommandValue == TKinokoDataStreamScanner::Command_RunBegin) {
               if (_DescriptorCounts[DataSourceId] == 1) {
	            _OutputStream_RunBegin->Write(Packet, PacketSize);
                    _DescriptorCounts[DataSourceId] -= 1;

		    ((U32bit*) Packet)[TKinokoDataStreamFormatter::Offset_PacketType] = TKinokoDataStreamFormatter::PacketType_Command | TKinokoDataStreamFormatter::Command_RunEnd;
		    _OutputStream_RunEnd->Write(Packet, PacketSize);
	       }
	    }
	}

	_InputStream->Flush(Packet);
    }
}


int main(int argc, char** argv)
{
    if (argc < 2) {
        cerr << argv[0] << ": extracts DataDescriptor and RunBegin packets from a kdf file" << endl;
	cerr << "Usage: " << argv[0] << " SoureFileName" << endl;
        return EXIT_FAILURE;
    }

    TMushArgumentList ArgumentList(argc, argv);

    try {
	TKinokoKdfExtract(ArgumentList).Start();
    }
    catch (TKinokoException& e) {
	cerr << "ERROR: " << e << endl;
        return EXIT_FAILURE;
    }

    return 0;
}
