/* kdfsplit.cc */
/* Created by Enomoto Sanshiro on 26 November 2001. */
/* Last updated by Enomoto Sanshiro on 26 November 2001. */


#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>
#include "MushFile.hh"
#include "MushFileSystem.hh"
#include "MushArgumentList.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataProcessor.hh"

using namespace std;


class TKinokoKdfSplit {
  public:
    TKinokoKdfSplit(const TMushArgumentList& ArgumentList);
    virtual ~TKinokoKdfSplit();
    virtual void Start(void) throw(TKinokoException);
  protected:
    string _InputFileName;
    bool _IsOverWriteAllowed;
    TKinokoDataDescriptor* _InputDataDescriptor;
    TKinokoStorage* _InputStorage;
    vector<TKinokoStorage*> _OutputStorageList;
    TKinokoInputStream* _InputStream;
    map<long, TKinokoOutputStream*> _OutputStreamTable;
};



TKinokoKdfSplit::TKinokoKdfSplit(const TMushArgumentList& ArgumentList)
{
    _InputFileName = ArgumentList[0];
    _IsOverWriteAllowed = ArgumentList.IsOptionSpecified("-f");

    _InputStorage = 0;
    _InputDataDescriptor = 0;
}

TKinokoKdfSplit::~TKinokoKdfSplit()
{
    delete _InputDataDescriptor;
    delete _InputStorage;

    for (unsigned i = 0; i < _OutputStorageList.size(); i++) {
	delete _OutputStorageList[i];
    }
}

void TKinokoKdfSplit::Start(void) throw(TKinokoException)
{
    _InputDataDescriptor = new TKinokoDataDescriptor();
    _InputStorage = new TKinokoKdfStorage(_InputFileName);

    TKinokoStorageHeader InputStorageHeader;
    try {
        _InputStorage->ReadHeader(InputStorageHeader);
    }
    catch (TKinokoException &e) {
        /* version number mismatch ? */
	cout << "WARNING: " << e << endl << endl;
    }

    TKinokoStorageHeader StorageHeader;
    StorageHeader.AddEntry("Creator", "kdfsplit");
    StorageHeader.AddEntry("DateTime", TMushDateTime().AsString());
    StorageHeader.AddEntry("UserName", TMushUser().Name());
    StorageHeader.AddEntry("Comment", "created by kdfsplit from " + _InputFileName);

    _InputStream = _InputStorage->GetInputStream();

    void* Packet;
    int PacketSize;
    TKinokoOutputStream* OutputStream;
    while ((PacketSize = _InputStream->NextEntry(Packet)) > 0) {
	//... correct byte order ...

	if (TKinokoDataStreamScanner::IsDataDescriptorPacket(Packet)) {
	    char* DescriptorText = TKinokoDataStreamScanner::DataDescriptorOf(Packet);
	    istringstream DescriptorTextStream(DescriptorText);
	    try {
		_InputDataDescriptor->ReadFrom(DescriptorTextStream);
	    }
	    catch (TKinokoException &e) {
		throw TKinokoException("badly formatted data descriptor");
	    }
	}

	long DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);

	OutputStream = _OutputStreamTable[DataSourceId];
	if (OutputStream == 0) {
	    string DataSourceName = _InputDataDescriptor->DataSource(DataSourceId)->DataSourceName();

	    TMushFileAttribute InputFileAttr(_InputFileName);
	    string OutputFileName = InputFileAttr.FileRootName() + "-" + DataSourceName + "." + InputFileAttr.Extension();

	    cout << "making " << OutputFileName << "..." << endl;

	    if (
		(! _IsOverWriteAllowed) &&
		(TMushFileAttribute(OutputFileName).IsReadable())
	    ){
		throw TKinokoException(
		    "file already exists: " + OutputFileName + "\n" +
		    "use -f option to overwrite"
		);
	    }

	    TKinokoKdfStorage* Storage = new TKinokoKdfStorage(OutputFileName);

	    if (_IsOverWriteAllowed) {
		Storage->AllowOverWrite();
	    }
	    Storage->WriteHeader(StorageHeader);
	    OutputStream = Storage->GetOutputStream();

	    _OutputStorageList.push_back(Storage);
	    _OutputStreamTable[DataSourceId] = OutputStream;
	}
	
	OutputStream->Write(Packet, PacketSize);
	_InputStream->Flush(Packet);
    }
}


int main(int argc, char** argv)
{
    if (argc < 2) {
	cerr << "Usage: " << argv[0] << " SoureFileName" << endl;
        return EXIT_FAILURE;
    }

    TMushArgumentList ArgumentList(argc, argv);

    try {
	TKinokoKdfSplit(ArgumentList).Start();
    }
    catch (TKinokoException& e) {
	cerr << "ERROR: " << e << endl;
        return EXIT_FAILURE;
    }

    return 0;
}
