/* kinoko-cleanup.cc */
/* Created by Enomoto Sanshiro on 11 February 2010. */
/* Last updated by Enomoto Sanshiro on 11 February 2010. */


#include <iostream>
#include <string>
#include <cstdlib>
#include "MushArgumentList.hh"
#include "MushXml.hh"
#include "MushNetworkSocket.hh"

using namespace std;
using namespace mush;


class TKinokoCleaner {
  public:
    TKinokoCleaner(void);
    virtual ~TKinokoCleaner();
    virtual void SetDate(const string& Date);
    virtual void KillKcomProcess(const string& Name, const string& Host, const string& ProcessId);
    virtual void RemoveMessageQueue(const string& Name, const string& Host, const string& ProjectPath, const std::string& ProjectId);
    virtual void RemoveSharedMemory(const string& Name, const string& Host, const string& ProjectPath, const std::string& ProjectId);
  protected:
    string _LocalHostAddress;
};


class TKinokoRegistryRepositoryHandler: public sax::DocumentHandler {
  public:
    TKinokoRegistryRepositoryHandler(TKinokoCleaner* Cleaner);
    virtual ~TKinokoRegistryRepositoryHandler();
  public:
    virtual void startElement(const std::string& name, const sax::AttributeList& attribute_list);
    virtual void endElement(const std::string& name);
    virtual void characters(const std::string& text);
  protected:
    TKinokoCleaner* _Cleaner;
    string _ResourceType, _Creator;
    string _Host, _ProcessId, _ProjectPath, _ProjectId;
  private:
    int _Depth;
    string _CurrentValue;
};


TKinokoRegistryRepositoryHandler::TKinokoRegistryRepositoryHandler(TKinokoCleaner* Cleaner)
{
    _Cleaner = Cleaner;
    _Depth = 0;

    cout << "#! /bin/sh" << endl;
}

TKinokoRegistryRepositoryHandler::~TKinokoRegistryRepositoryHandler()
{
}

void TKinokoRegistryRepositoryHandler::startElement(const std::string& name, const sax::AttributeList& attribute_list)
{
    if (name == "__resources") {
	_Depth = 1;
    }
    else if (_Depth > 0) {
	_Depth++;
    }

    if (_Depth == 2) {
	_ResourceType = name;
    }
    else if (_Depth == 3) {
	_Creator = name;
    }

    _CurrentValue.clear();
}

void TKinokoRegistryRepositoryHandler::endElement(const std::string& name)
{
    if (_Depth > 0) {
	_Depth--;
    }
    if (_Depth <= 0) {
	if (name == "calendar_time") {
	    _Cleaner->SetDate(_CurrentValue);
	}

	return;
    }

    if (_Depth == 1) {
	;
    }
    else if (_Depth == 2) {
	if (_ResourceType == "kcom_process") {
	    _Cleaner->KillKcomProcess(_Creator, _Host, _ProcessId);
	}
	else if (_ResourceType == "shared_memory") {
	    _Cleaner->RemoveSharedMemory(
		_Creator, _Host, _ProjectPath, _ProjectId
	    );
	}
	else if (_ResourceType == "message_queue") {
	    _Cleaner->RemoveMessageQueue(
		_Creator, _Host, _ProjectPath, _ProjectId
	    );
	}
    }
    else if (_Depth == 3) {
	if (name == "process_id") {
	    _ProcessId = _CurrentValue;
	}
	else if (name == "host") {
	    _Host = _CurrentValue;
	}
	else if (name == "project_path") {
	    _ProjectPath = _CurrentValue;
	}
	else if (name == "project_id") {
	    _ProjectId = _CurrentValue;
	}
    }
}

void TKinokoRegistryRepositoryHandler::characters(const std::string& text)
{
    _CurrentValue += text;
}



TKinokoCleaner::TKinokoCleaner(void)
{
    _LocalHostAddress = TMushNetworkSocket::LocalIPAddress();
}

TKinokoCleaner::~TKinokoCleaner()
{
    cout << endl;
}

void TKinokoCleaner::SetDate(const string& Date)
{
    cout << "# clean-up script for Kinoko started on " << Date << endl;
    cout << "# automatically generated by kinoko-cleanup" << endl;
}

void TKinokoCleaner::KillKcomProcess(const string& Name, const string& Host, const string& ProcessId)
{
    static bool IsFirst = true;
    if (IsFirst) {
	cout << endl;
	cout << "# Component Process #" << endl;
	IsFirst = false;
    }

    if (Host != _LocalHostAddress) {
	cout << "ssh " << Host << " ";
    }
    cout << "kill -9 " << ProcessId;
    cout << "\t\t# " << Name << " @ " << Host << endl;
}

void TKinokoCleaner::RemoveMessageQueue(const string& Name, const string& Host, const string& ProjectPath, const std::string& ProjectId)
{
    static bool IsFirst = true;
    if (IsFirst) {
	cout << endl;
	cout << "# Message Queue #" << endl;
	IsFirst = false;
    }

    if (Host != _LocalHostAddress) {
	cout << "ssh " << Host << " ";
    }
    cout << "msqrm " << ProjectPath << " " << ProjectId;
    cout << "\t\t# " << Name << " @ " << Host << endl;
}

void TKinokoCleaner::RemoveSharedMemory(const string& Name, const string& Host, const string& ProjectPath, const std::string& ProjectId)
{
    static bool IsFirst = true;
    if (IsFirst) {
	cout << endl;
	cout << "# Shared Memory #" << endl;
	IsFirst = false;
    }

    if (Host != _LocalHostAddress) {
	cout << "ssh " << Host << " ";
    }
    cout << "shmrm " << ProjectPath << " " << ProjectId;

    cout << "\t\t# " << Name << " @ " << Host << endl;
}



int main(int argc, char** argv)
{
    TMushArgumentList ArgumentList(argc, argv);

    if (ArgumentList.NumberOfParameters() < 1) {
	cerr << "Usage: " << ArgumentList.ProgramName();
	cerr << " KinokoRegistryFile" << endl;
        return EXIT_FAILURE;
    }
    string RegistryFile = ArgumentList[0];

    try {
	TKinokoCleaner Cleaner;
	TKinokoRegistryRepositoryHandler Handler(&Cleaner);
	sax::Parser Parser;
	Parser.setDocumentHandler(&Handler);
	Parser.parse(RegistryFile);
    }
    catch (TSystemCallException &e) {
	cerr << "ERROR: " << e << endl;
	return EXIT_FAILURE;
    }

    return 0;
}
