/* kinoko-viewer.cc */
/* Created by Enomoto Sanshiro on 1 September 2001. */
/* Last updated by Enomoto Sanshiro on 16 February 2008. */


#include <iostream>
#include <cstdlib>
#include "MushArgumentList.hh"
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "KinokoStandaloneComponent.hh"
#include "KinokoViewer.hh"

using namespace std;


class TKinokoOfflineViewer: public TKinokoViewer {
  public:
    TKinokoOfflineViewer(bool IsPlaybackMode = false);
    virtual ~TKinokoOfflineViewer();
    virtual void OnReceiveCommandPacket(void* Packet, long PacketSize) throw(TKinokoException);
    virtual void OnRunEnd(void) throw(TKinokoException);
    virtual void OnDestruct(void) throw(TKinokoException);
  private:
    bool _IsPlaybackMode;
    int _DisplayStartTime, _DataStartTime;
};


TKinokoOfflineViewer::TKinokoOfflineViewer(bool IsPlaybackMode)
{
    _IsPlaybackMode = IsPlaybackMode;

    _DataStartTime = 0;
    _DisplayStartTime = TMushDateTime::SecSinceEpoch();
}

TKinokoOfflineViewer::~TKinokoOfflineViewer()
{
}

void TKinokoOfflineViewer::OnReceiveCommandPacket(void* Packet, long PacketSize) throw(TKinokoException) 
{
    TKinokoViewer::OnReceiveCommandPacket(Packet, PacketSize);

    if (! _IsPlaybackMode) {
	return;
    }

    int EventTime = TKinokoDataStreamScanner::CommandTimeOf(Packet);

    if (_DataStartTime == 0) {
	_DataStartTime = EventTime;
    }

    int WaitTime = (EventTime - _DataStartTime) - TMushDateTime::SecSince(_DisplayStartTime);
    if (WaitTime > 0) {
	TMushRealTimeTimer(WaitTime, 0).Suspend();
    }
}

void TKinokoOfflineViewer::OnRunEnd(void) throw(TKinokoException)
{
    DrawAll();
    TKinokoViewer::OnRunEnd();
}

void TKinokoOfflineViewer::OnDestruct(void) throw(TKinokoException)
{
    DrawAll();
    TKinokoViewer::OnDestruct();
}



int main(int argc, char** argv)
{
    TMushArgumentList ArgumentList(argc, argv);
    if (ArgumentList.NumberOfParameters() < 2) {
	cerr << "Usage: " << argv[0];
	cerr << " [Options] DataFileName ScriptFileName" << endl;
	cerr << "Options:" << endl;
	cerr << "  --port=PortNumber" << endl;
	cerr << "  --delay=EventInterval_msec" << endl;
	cerr << "  --registry=RegistryFileName" << endl;
	cerr << "  --playback" << endl;
	return EXIT_FAILURE;
    }
    string ScriptFileName = ArgumentList[1];
    
    TKinokoViewer* Viewer = (
	new TKinokoOfflineViewer(
	    ArgumentList.IsOptionSpecified("--playback")
	)
    );
    TKinokoStandaloneComponent* StandaloneComponent = (
	new TKinokoStandaloneDataConsumer(Viewer, "Viewer")
    );

    try {
	Viewer->SetScript(ScriptFileName);
	StandaloneComponent->Start(ArgumentList);
    }
    catch (TKinokoException &e) {
	cerr << "ERROR: " << argv[0] << ": " << e << endl;
    }

    delete StandaloneComponent;
    delete Viewer;

    return 0;
}
