/* KinokoCanvas.cc */
/* Created by Enomoto Sanshiro on 9 July 2000. */
/* Last updated by Enomoto Sanshiro on 19 March 2008. */


#include <iostream>
#include <vector>
#include <map>
#include "KinokoShellConnector.hh"
#include "KinokoCanvas.hh"
#include "KinokoCanvasObject.hh"
#include "KinokoCanvasPictureObject.hh"
#include "KinokoCanvasPlotObject.hh"
#include "KinokoCanvas3dPlotObject.hh"
#include "KinokoCanvasTextBoxObject.hh"
#include "KinokoCanvasSchematicObject.hh"
#include "KinokoCanvasImageArea.hh"
#include "KinokoCanvasImageAreaEps.hh"

using namespace std;


TKinokoCanvas::TKinokoCanvas(TKinokoShellConnector* ShellConnector)
: TKinokoShell(ShellConnector)
{
    _IsAutoRedrawEnabled = true;
    _CurrentPage = 0;
    _CurrentPageNumber = 0;
}

TKinokoCanvas::~TKinokoCanvas()
{
    for (unsigned i = 0; i < _CanvasPageList.size(); i++) {
	delete _CanvasPageList[i];
    }
}

void TKinokoCanvas::ConstructPrototypeTable(void)
{
    RegisterObjectPrototype(
	"picture", new TKinokoCanvasPictureObject(this)
    );
    RegisterObjectPrototype(
	"textbox", new TKinokoCanvasTextBoxObject(this)
    );
    RegisterObjectPrototype(
	"plot", new TKinokoCanvasPlotObject(this)
    );
    RegisterObjectPrototype(
	"plot3dnoscale", new TKinokoCanvas3dPlotObject(this)
    );
    RegisterObjectPrototype(
	"plot3dmonochrome", new TKinokoCanvasZScaled3dPlotObject(this, true)
    );
    RegisterObjectPrototype(
	"plot3d", new TKinokoCanvasZScaled3dPlotObject(this)
    );
    RegisterObjectPrototype(
	"mapplot", new TKinokoCanvasMapPlotObject(this)
    );
    RegisterObjectPrototype(
	"schematic", new TKinokoCanvasSchematicObject(this)
    );
}

int TKinokoCanvas::AddPage(TKinokoCanvasImageArea* ImageArea, bool IsEpsEnabled)
{
    TKinokoCanvasPage* Page = new TKinokoCanvasPage(
	this, ImageArea, IsEpsEnabled
    );

    Page->Clear();
    _CanvasPageList.push_back(Page);

    return _CanvasPageList.size() - 1;
}

void TKinokoCanvas::AddCanvasObject(TKinokoCanvasObject* CanvasObject)
{
    _CurrentPage->AddCanvasObject(CanvasObject);
}

TKinokoCanvasImageArea* TKinokoCanvas::ImageArea(void)
{
    return _CurrentPage->ImageArea();
}

int TKinokoCanvas::Reset(void) 
{
    for (unsigned i = 0; i < _CanvasPageList.size(); i++) {
	delete _CanvasPageList[i];
    }
    _CanvasPageList.erase(_CanvasPageList.begin(), _CanvasPageList.end());

    _IsAutoRedrawEnabled = true;
    _CurrentPage = 0;
    _CurrentPageNumber = 0;

    return 1;
}

int TKinokoCanvas::Clear(int PageNumber)
{
    if (PageNumber >= 0) {
	_CanvasPageList[PageNumber]->Clear();
    }
    else if (PageNumber == -1) {
	for (unsigned i = 0; i < _CanvasPageList.size(); i++) {
	    _CanvasPageList[i]->Clear();
	}
    }
    else {
	_CurrentPage->Clear();
    }

    return 1;
}

int TKinokoCanvas::Redraw(int PageNumber)
{
    if ((PageNumber >= 0) && (PageNumber < (int) _CanvasPageList.size())) {
	_CanvasPageList[PageNumber]->Redraw();
    }
    else if (PageNumber == -1) {
	for (unsigned i = 0; i < _CanvasPageList.size(); i++) {
	    _CanvasPageList[i]->Redraw();
	}
    }
    else if (_CurrentPage != 0) {
	_CurrentPage->Redraw();
    }

    return 1;
}

int TKinokoCanvas::SetBackgroundColor(const std::string& ColorName)
{
    _CurrentPage->SetBackgroundColor(ColorName);
    return 1;
}

int TKinokoCanvas::SetTitle(const string& Title)
{
    return 1;
}

int TKinokoCanvas::SetPageTitle(const string& Title)
{
    return 1;
}

int TKinokoCanvas::SaveImageTo(const string& FileName)
{
    return _CurrentPage->SaveImageTo(FileName);
}

int TKinokoCanvas::CreatePage(const std::string& PageName)
{
    int PageNumber = _CanvasPageList.size() - 1;
    SelectPage(PageNumber);

    return 1;
}

int TKinokoCanvas::SelectPage(int PageNumber)
{
    if ((PageNumber >=0) && (PageNumber < (int) _CanvasPageList.size())) {
	_CurrentPage = _CanvasPageList[PageNumber];
	_CurrentPageNumber = PageNumber;
	return 1;
    }
    else {
	return 0;
    }
}

void TKinokoCanvas::OnButtonPress(int PageNumber, int ButtonNumber, int X, int Y)
{
    _CanvasPageList[PageNumber]->OnButtonPress(ButtonNumber, X, Y);
}

void TKinokoCanvas::OnButtonRelease(int PageNumber, int ButtonNumber, int X, int Y)
{
    _CanvasPageList[PageNumber]->OnButtonRelease(ButtonNumber, X, Y);
}

void TKinokoCanvas::OnMouseMove(int PageNumber, int X, int Y)
{
    _CanvasPageList[PageNumber]->OnMouseMove(X, Y);
}

int TKinokoCanvas::ProcessCommand(std::istream& InputStream)
{
    int Result = TKinokoShell::ProcessCommand(InputStream);

    if ((Result > 0) && _IsAutoRedrawEnabled) {
	Redraw();
    }

    return Result;
}

int TKinokoCanvas::ProcessSystemCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == ".set") {
	Result = ProcessSetCommand(InputStream);
    }
    else if (Command == ".redraw") {
	Result = ProcessRedrawCommand(InputStream);
    }
    else if (Command == ".clear") {
	Result = ProcessClearCommand(InputStream);
    }
    else if (Command == ".reset") {
	Result = ProcessResetCommand(InputStream);
    }
    else if (Command == ".saveImage") {
	Result = ProcessSaveImageCommand(InputStream);
    }
    else if (Command == ".beginEps") {
	Result = ProcessBeginEpsCommand(InputStream);
    }
    else if (Command == ".endEps") {
	Result = ProcessEndEpsCommand(InputStream);
    }
    else if (Command == ".setEpsUnitSize") {
	Result = ProcessSetEpsUnitSizeCommand(InputStream);
    }
    else if (Command == ".createPage") {
	Result = ProcessCreatePageCommand(InputStream);
    }
    else if (Command == ".selectPage") {
	Result = ProcessSelectPageCommand(InputStream);
    }
    else {
	Result = TKinokoShell::ProcessSystemCommand(Command, InputStream);
    }
    
    return Result;
}

int TKinokoCanvas::ProcessSetCommand(istream& InputStream)
{
    int Result = 0;

    string Name;
    if (InputStream >> Name) {
	if (Name == "redraw") {
	    string Value;
	    if (InputStream >> Value) {
		if (Value == "auto") {
		    _IsAutoRedrawEnabled = true;
		    Result = 1;
		}
		else if (Value == "manual") {
		    _IsAutoRedrawEnabled = false;
		    Result = 1;
		}
	    }
	}
	else if (Name == "title") {
	    string Title;
	    InputStream >> ws;
	    if (getline(InputStream, Title, '\n')) {
		Result = SetTitle(Title);
	    }
	}
	else if (Name == "pagetitle") {
	    string Title;
	    InputStream >> ws;
	    if (getline(InputStream, Title, '\n')) {
		Result = SetPageTitle(Title);
	    }
	}
	else if (Name == "background") {
	    string Color;
	    InputStream >> ws;
	    if (getline(InputStream, Color, '\n')) {
		Result = SetBackgroundColor(Color);
	    }
	}
    }

    return Result;
}

int TKinokoCanvas::ProcessRedrawCommand(istream& InputStream)
{
    int PageNumber;
    if (InputStream >> PageNumber) {
	return Redraw(PageNumber);
    }
    else {
	return Redraw();
    }
}

int TKinokoCanvas::ProcessResetCommand(std::istream& InputStream)
{
    return Reset();
}

int TKinokoCanvas::ProcessClearCommand(std::istream& InputStream)
{
    int PageNumber;
    if (InputStream >> PageNumber) {
	return Clear(PageNumber);
    }
    else {
	return Clear();
    }
}

int TKinokoCanvas::ProcessSaveImageCommand(istream& InputStream)
{
    int Result = 0;
    string FileName;
    if (InputStream >> FileName) {
	Result = SaveImageTo(FileName);
    }

    return Result;
}

int TKinokoCanvas::ProcessBeginEpsCommand(istream& InputStream)
{
    int Result = 0;
    string FileName;
    if (InputStream >> FileName) {
	Result = _CurrentPage->BeginEps(FileName);
    }

    return Result;
}

int TKinokoCanvas::ProcessEndEpsCommand(istream& InputStream)
{
    return _CurrentPage->EndEps();
}

int TKinokoCanvas::ProcessSetEpsUnitSizeCommand(istream& InputStream)
{
    int Result = 0;
    double PixelsPerInch;
    if (InputStream >> PixelsPerInch) {
	Result = _CurrentPage->SetEpsUnitSize(PixelsPerInch);
    }

    return Result;
}

int TKinokoCanvas::ProcessCreatePageCommand(istream& InputStream)
{
    string PageName;
    getline(InputStream, PageName);

    return CreatePage(PageName);
}

int TKinokoCanvas::ProcessSelectPageCommand(istream& InputStream)
{
    int Result = 0;

    int PageNumber;
    if (InputStream >> PageNumber) {
	Result = SelectPage(PageNumber);
    }

    return Result;
}



TKinokoCanvasPage::TKinokoCanvasPage(TKinokoCanvas* Canvas, TKinokoCanvasImageArea* ImageArea, bool IsEpsEnabled)
{
    _Canvas = Canvas;

    if (IsEpsEnabled) {
	_EpsImageArea = new TKinokoCanvasImageAreaEps(
	    ImageArea->Width(), ImageArea->Height(), ImageArea
	);
	_ImageArea = _EpsImageArea;
    }
    else {
	_EpsImageArea = 0;
	_ImageArea = ImageArea;
    }

    _IsInRangeSelectionMode = false;
}

TKinokoCanvasPage::~TKinokoCanvasPage()
{
    delete _EpsImageArea;
}

TKinokoCanvasImageArea* TKinokoCanvasPage::ImageArea(void)
{
    return _ImageArea;
}

void TKinokoCanvasPage::AddCanvasObject(TKinokoCanvasObject* CanvasObject)
{
    _CanvasObjectList.push_back(CanvasObject);
}

int TKinokoCanvasPage::Clear(void)
{
    _ImageArea->Clear();
    return 1;
}

int TKinokoCanvasPage::Redraw(void)
{
    _ImageArea->Redraw();
    return 1;
}

int TKinokoCanvasPage::SetBackgroundColor(const std::string& ColorName)
{
    _ImageArea->SetBackgroundColor(_ImageArea->AllocateColor(ColorName));
    return 1;
}

int TKinokoCanvasPage::SaveImageTo(const string& FileName)
{
    return _ImageArea->SaveImageTo(FileName);
}

int TKinokoCanvasPage::BeginEps(const std::string& FileName)
{
    if (_EpsImageArea) {
	return _EpsImageArea->Open(FileName);
    }

    return 1;
}

int TKinokoCanvasPage::EndEps(void)
{
    if (_EpsImageArea) {
	return _EpsImageArea->Close();
    }

    return 1;
}

int TKinokoCanvasPage::SetEpsUnitSize(double PixelsPerInch)
{
    if (_EpsImageArea) {
	return _EpsImageArea->SetUnitSize(PixelsPerInch);
    }

    return 1;
}

void TKinokoCanvasPage::OnButtonPress(int ButtonNumber, int X, int Y)
{
    if (ButtonNumber == 1) {
	_X0 = _X1 = X;
	_Y0 = _Y1 = Y;
	_IsInRangeSelectionMode = true;
    }
    if (ButtonNumber == 3) {
	for (int i = _CanvasObjectList.size() - 1; i >= 0; i--) {
	    if (_CanvasObjectList[i]->Includes(X, Y)) {
		if (_Canvas->OpenContextMenu(_CanvasObjectList[i])) {
		    break;
		}
	    }
	}
    }
}

void TKinokoCanvasPage::OnButtonRelease(int ButtonNumber, int X, int Y)
{
    if ((ButtonNumber == 1) && _IsInRangeSelectionMode) {
	int DrawingFunction = _ImageArea->SetDrawingFunction(
	    TKinokoCanvasImageArea::DrawingFunction_Invert
	);
	_ImageArea->DrawRect(_X0, _Y0, _X1, _Y1);
	_ImageArea->SetDrawingFunction(DrawingFunction);
	_ImageArea->Redraw();

	_IsInRangeSelectionMode = false;

	for (int i = _CanvasObjectList.size() - 1; i >= 0; i--) {
	    if (_CanvasObjectList[i]->Includes(_X0, _Y0)) {
		if (_CanvasObjectList[i]->ProcessRangeSelect(_X0, _Y0, _X1, _Y1)) {
		    break;
		}
	    }
	}
    }
}

void TKinokoCanvasPage::OnMouseMove(int X, int Y)
{
    if (! _IsInRangeSelectionMode) {
	return;
    }

    int DrawingFunction = _ImageArea->SetDrawingFunction(
	TKinokoCanvasImageArea::DrawingFunction_Invert
    );

    _ImageArea->DrawRect(_X0, _Y0, _X1, _Y1);
    _ImageArea->DrawRect(_X0, _Y0, _X1 = X, _Y1 = Y);

    _ImageArea->SetDrawingFunction(DrawingFunction);
    _ImageArea->Redraw();
}

