/* KinokoCanvasColorScale.cc */
/* Created by Enomoto Sanshiro on 14 January 2001. */
/* Last updated by Enomoto Sanshiro on 14 January 2001. */


#include <cmath>
#include <cstdlib>
#include "KinokoCanvas.hh"
#include "KinokoCanvasImageArea.hh"
#include "KinokoCanvasColorScale.hh"

using namespace std;



TKinokoCanvasColorScale::TKinokoCanvasColorScale(TKinokoCanvas* Canvas, int NumberOfColors)
{
    _Canvas = Canvas;
    _ImageArea = 0;

    _NumberOfColors = NumberOfColors;
    _ColorTable = 0;
}

TKinokoCanvasColorScale::~TKinokoCanvasColorScale()
{
    delete[] _ColorTable;
}

void TKinokoCanvasColorScale::Initialize(void)
{
    if (_ColorTable) {
	delete[] _ColorTable;
    }

    _ImageArea = _Canvas->ImageArea();
    _ColorTable = new int[_NumberOfColors];

    CreateColorTable();
}

int TKinokoCanvasColorScale::ColorIndexOf(double Value)
{
    int ScaleIndex;
    if (Value < 0) {
	ScaleIndex = 0;
    }
    else if (Value >= 1) {
	ScaleIndex = _NumberOfColors - 1;
    }
    else {
	ScaleIndex = (int) (Value * _NumberOfColors);
    }

    return _ColorTable[ScaleIndex];
}



TKinokoCanvasGrayColorScale::TKinokoCanvasGrayColorScale(TKinokoCanvas* Canvas, int NumberOfColors)
: TKinokoCanvasColorScale(Canvas, NumberOfColors)
{
}

TKinokoCanvasGrayColorScale::~TKinokoCanvasGrayColorScale()
{
}

void TKinokoCanvasGrayColorScale::CreateColorTable(void)
{
    for (int i = 0; i < _NumberOfColors; i++) {
	//float v = 1.0 - (float) i / (_NumberOfColors - 1);
	float v = (float) i / (_NumberOfColors - 1);
	float r = v;
	float g = v;
	float b = v;

	int Color = _ImageArea->AllocateColorRgb(r, g, b);
	_ColorTable[i] = Color;
    }
}



TKinokoCanvasTwoColorScale::TKinokoCanvasTwoColorScale(TKinokoCanvas* Canvas, int NumberOfColors)
: TKinokoCanvasColorScale(Canvas, NumberOfColors)
{
}

TKinokoCanvasTwoColorScale::~TKinokoCanvasTwoColorScale()
{
}

void TKinokoCanvasTwoColorScale::CreateColorTable(void)
{
    float StartR = 0.1, StartG = 0.1, StartB = 1.0;
    float EndR = 1.0, EndG = 0, EndB = 0;
    
    float WidthR = EndR - StartR;
    float WidthG = EndG - StartG;
    float WidthB = EndB - StartB;

    for (int i = 0; i < _NumberOfColors; i++) {
	float Value = (float) i / (_NumberOfColors - 1);
	float r = Value * WidthR + StartR;
	float g = Value * WidthG + StartG;
	float b = Value * WidthB + StartB;

	int Color = _ImageArea->AllocateColorRgb(r, g, b);
	_ColorTable[i] = Color;
    }
}



TKinokoCanvasRainbowColorScale::TKinokoCanvasRainbowColorScale(TKinokoCanvas* Canvas, int NumberOfColors)
: TKinokoCanvasColorScale(Canvas, NumberOfColors)
{
}

TKinokoCanvasRainbowColorScale::~TKinokoCanvasRainbowColorScale()
{
}

void TKinokoCanvasRainbowColorScale::CreateColorTable(void)
{
    static const float HueCutOff = 4.0/6.0;
    static const float Saturation = 1.0;
    static const float Brightness = 0.5;
    
    for (int i = 0; i < _NumberOfColors; i++) {
	float Value = (float) i / (_NumberOfColors - 1);
	float h = HueCutOff * (1.0 - Value);
	float s = Saturation;
	float v = Brightness;

	h = (
	    +1.00 * h
	    +0.40 * sin(2*M_PI * 3*h)/(2*M_PI * 3)
	    +0.05 * sin(2*M_PI * 6*h)/(2*M_PI * 6)
	);

	float r, g, b;
	HSV2RGB(h, s, v, r, g, b);

	int Color = _ImageArea->AllocateColorRgb(r, g, b);
	_ColorTable[i] = Color;
    }
}

static float HSV2RGB_aux(float rm1, float rm2, float h);

void TKinokoCanvasRainbowColorScale::HSV2RGB(float h, float s, float v, float& r, float& g, float& b)
{
    // see HIGZ routine IGHTOR

    // normalization
    h = ((h >= 0) && (h <= 1.0)) ? h : fmod(h + abs((int) h) + 1, 1.0f);
    s = ((s >= 0) && (s <= 1.0)) ? s : fmod(s + abs((int) s) + 1, 1.0f);
    v = ((v >= 0) && (v <= 1.0)) ? v : fmod(v + abs((int) v) + 1, 1.0f);

    // use degree unit for convenience...
    h *= 360;

    float rm1, rm2;
    if (v < 0.5) {
	rm2 = v * (1.0 + s);
    }
    else {
	rm2 = v + s - v * s;
    }
    rm1 = 2.0 * v - rm2;

    if (s == 0) {
	r = g = b = v;
    }
    else {
	r = HSV2RGB_aux(rm1, rm2, h + 120);
	g = HSV2RGB_aux(rm1, rm2, h);
	b = HSV2RGB_aux(rm1, rm2, h - 120);
    }
}

static float HSV2RGB_aux(float rm1, float rm2, float h)
{
    // see HIGZ routine IGHR01
    h = (h > 360) ? (h - 360) : ((h < 0) ? (h + 360) : h);

    if (h < 60) {
	return rm1 + (rm2 - rm1) * h / 60;
    }
    else if (h < 180) {
	return rm2;
    }
    else if (h < 240) {
	return rm1 + (rm2 - rm1) * (240 - h) / 60;
    }
    else {
	return rm1;
    }
}



TKinokoCanvasWhiteBackgroundRainbowColorScale::TKinokoCanvasWhiteBackgroundRainbowColorScale(TKinokoCanvas* Canvas, int NumberOfColors)
: TKinokoCanvasRainbowColorScale(Canvas, NumberOfColors)
{
}

TKinokoCanvasWhiteBackgroundRainbowColorScale::~TKinokoCanvasWhiteBackgroundRainbowColorScale()
{
}

void TKinokoCanvasWhiteBackgroundRainbowColorScale::CreateColorTable(void)
{
    TKinokoCanvasRainbowColorScale::CreateColorTable();
    _BackgroundColor = _Canvas->ImageArea()->AllocateColorRgb(1.0, 1.0, 1.0);
}

int TKinokoCanvasWhiteBackgroundRainbowColorScale::ColorIndexOf(double Value)
{
    if (Value <= 0) {
	return _BackgroundColor;
    }
    else {
	return TKinokoCanvasRainbowColorScale::ColorIndexOf(Value);
    }
}



TKinokoCanvasBlackBackgroundRainbowColorScale::TKinokoCanvasBlackBackgroundRainbowColorScale(TKinokoCanvas* Canvas, int NumberOfColors)
: TKinokoCanvasRainbowColorScale(Canvas, NumberOfColors)
{
}

TKinokoCanvasBlackBackgroundRainbowColorScale::~TKinokoCanvasBlackBackgroundRainbowColorScale()
{
}

void TKinokoCanvasBlackBackgroundRainbowColorScale::CreateColorTable(void)
{
    TKinokoCanvasRainbowColorScale::CreateColorTable();
    _BackgroundColor = _Canvas->ImageArea()->AllocateColorRgb(0, 0, 0);
}

int TKinokoCanvasBlackBackgroundRainbowColorScale::ColorIndexOf(double Value)
{
    if (Value <= 0) {
	return _BackgroundColor;
    }
    else {
	return TKinokoCanvasRainbowColorScale::ColorIndexOf(Value);
    }
}



TKinokoCanvasGrayBackgroundRainbowColorScale::TKinokoCanvasGrayBackgroundRainbowColorScale(TKinokoCanvas* Canvas, int NumberOfColors)
: TKinokoCanvasRainbowColorScale(Canvas, NumberOfColors)
{
}

TKinokoCanvasGrayBackgroundRainbowColorScale::~TKinokoCanvasGrayBackgroundRainbowColorScale()
{
}

void TKinokoCanvasGrayBackgroundRainbowColorScale::CreateColorTable(void)
{
    TKinokoCanvasRainbowColorScale::CreateColorTable();
    _BackgroundColor = _Canvas->ImageArea()->AllocateColorRgb(0.5, 0.5, 0.5);
}

int TKinokoCanvasGrayBackgroundRainbowColorScale::ColorIndexOf(double Value)
{
    if (Value <= 0) {
	return _BackgroundColor;
    }
    else {
	return TKinokoCanvasRainbowColorScale::ColorIndexOf(Value);
    }
}
