/* KinokoCanvasDrawableObject.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 12 June 2004. */


#include <sstream>
#include <algorithm>
#include <cstring>
#include <cmath>
#include "KinokoCanvasObject.hh"
#include "KinokoCanvasDrawableObject.hh"

using namespace std;


#define clip(x, lower, upper) (min(max((lower), (x)), (upper)))


TKinokoCanvasDrawableObject::TKinokoCanvasDrawableObject(TKinokoCanvas* Canvas)
: TKinokoCanvasObject(Canvas)
{
    _MathTextComposer = 0;
    _MarkerFactory = 0;
    _Marker = 0;
}

TKinokoCanvasDrawableObject::~TKinokoCanvasDrawableObject()
{
    delete _MathTextComposer;
    delete _MarkerFactory;
    delete _Marker;
}

void TKinokoCanvasDrawableObject::Initialize(void)
{
    TKinokoCanvasObject::Initialize();

    _MathTextComposer = new TKinokoCanvasMathTextComposer(
	*_MathTextComposerList[Font_Normal]
    );

    _MarkerFactory = new TKinokoCanvasMarkerFactory();

    SetColor();
    SetFont();
    _TextAdjustment = "tl";
    _LineWidth = 1;
    _LineStyleIndex = 0;
    _Marker = 0;
    _MarkerSize = 3;

    SetCoordinate(0, 0, 1, 1);

    _CursorX = 0;
    _CursorY = 0;
    _CursorX0 = 0;
}

void TKinokoCanvasDrawableObject::SetPosition(int OffsetX, int OffsetY, int Width, int Height)
{
    TKinokoCanvasObject::SetPosition(OffsetX, OffsetY, Width, Height);

    _DrawableXMin = OffsetX;
    _DrawableXMax = OffsetX + Width;
    _DrawableYMin = OffsetY;
    _DrawableYMax = OffsetY + Height;
}

int TKinokoCanvasDrawableObject::ProcessCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "set") {
	string Name, Value;
	InputStream >> Name >> ws;
	if (! Name.empty()) {
	    getline(InputStream, Value, '\n');
	    Result = ProcessSetCommand(Name, Value);
	}
    }
    else if (Command == "drawpoint") {
	Result = ProcessDrawPointCommand(InputStream);
    }
    else if (Command == "drawline") {
	Result = ProcessDrawLineCommand(InputStream);
    }
    else if (Command == "drawrect") {
	Result = ProcessDrawRectCommand(InputStream, false);
    }
    else if (Command == "drawrectfill") {
	Result = ProcessDrawRectCommand(InputStream, true);
    }
    else if (Command == "drawpolygon") {
	Result = ProcessDrawPolygonCommand(InputStream, false);
    }
    else if (Command == "drawpolygonfill") {
	Result = ProcessDrawPolygonCommand(InputStream, true);
    }
    else if (Command == "drawcircle") {
	Result = ProcessDrawCircleCommand(InputStream, false);
    }
    else if (Command == "drawcirclefill") {
	Result = ProcessDrawCircleCommand(InputStream, true);
    }
    else if (Command == "drawellipse") {
	Result = ProcessDrawEllipseCommand(InputStream, false);
    }
    else if (Command == "drawellipsefill") {
	Result = ProcessDrawEllipseCommand(InputStream, true);
    }
    else if (Command == "drawarrowtip") {
	Result = ProcessDrawArrowTipCommand(InputStream, false);
    }
    else if (Command == "drawarrowtipfill") {
	Result = ProcessDrawArrowTipCommand(InputStream, true);
    }
    else if (Command == "drawtext") {
	Result = ProcessDrawTextCommand(InputStream);
    }
    else if (Command == "drawimage") {
	Result = ProcessDrawImageCommand(InputStream);
    }
    else if (Command == "locate") {
	Result = ProcessLocateCommand(InputStream);
    }
    else if (Command == "print") {
	Result = ProcessPrintCommand(InputStream, false);
    }
    else if (Command == "println") {
	Result = ProcessPrintCommand(InputStream, true);
    }
    else if (Command == "clear") {
	Result = ProcessClearCommand(InputStream);
    }
    else if (Command == "setfont") {
	Result = ProcessSetFontCommand(InputStream);
    }
    else if (Command == "savecontext") {
	Result = ProcessSaveContextCommand(InputStream);
    }
    else if (Command == "restorecontext") {
	Result = ProcessRestoreContextCommand(InputStream);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "foreground") {
	// color for frame, label, etc.
	Result = SetForegroundColor(Value);
    }
    else if (Name == "background") {
	Result = SetBackgroundColor(Value);
    }
    else if (Name == "color") {
	// color for line, text, etc.
	Result = SetColor(Value);
    }
    else if (Name == "linewidth") {
	Result = SetLineWidth(Value);
    }
    else if (Name == "linestyle") {
	Result = SetLineStyle(Value);
    }
    else if (Name == "textadjustment") {
	Result = SetTextAdjustment(Value);
    }
    else if (Name == "marker") {
	Result = SetMarker(Value);
    }
    else if (Name == "markersize") {
	Result = SetMarkerSize(Value);
    }
    else {
	Result = 0;
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawPointCommand(std::istream& InputStream)
{
    int Result = 0;
    
    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();

    float x, y;
    if (InputStream >> x >> y) {
	Result = 1;

	if ((x >= _XMin) && (x <= _XMax) && (y >= _YMin) && (y <= _YMax)) {
	    float cx = CanvasXOf(x);
	    float cy = CanvasYOf(y);
	
	    if (_Marker) {
		_Marker->Draw(_ImageArea, cx, cy, _MarkerSize);
	    }
	    else {
		_ImageArea->DrawPoint(cx, cy);
	    }
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawLineCommand(istream& InputStream)
{
    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    float x0, y0, x1, y1;
    if (InputStream >> x0 >> y0 >> x1 >> y1) {
	Result = 1;

	float cx0 = CanvasXOf(x0);
	float cy0 = CanvasYOf(y0);
	float cx1 = CanvasXOf(x1);
	float cy1 = CanvasYOf(y1);
	
	_ImageArea->DrawLine(cx0, cy0, cx1, cy1);
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawRectCommand(istream& InputStream, bool IsFilled)
{
    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    float x0, y0, x1, y1;
    if (InputStream >> x0 >> y0 >> x1 >> y1) {
	Result = 1;

	float cx0 = CanvasXOf(x0);
	float cy0 = CanvasYOf(y0);
	float cx1 = CanvasXOf(x1);
	float cy1 = CanvasYOf(y1);
#if 0
	cx0 = clip(cx0, _DrawableXMin, _DrawableXMax);
	cy0 = clip(cy0, _DrawableYMin, _DrawableYMax);
	cx1 = clip(cx1, _DrawableXMin, _DrawableXMax);
	cy1 = clip(cy1, _DrawableYMin, _DrawableYMax);
#endif
	
	if (IsFilled) {
	    _ImageArea->DrawRectFill(cx0, cy0, cx1, cy1);
	}
	else {
	    _ImageArea->DrawRect(cx0, cy0, cx1, cy1);
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawPolygonCommand(istream& InputStream, bool IsFilled)
{
    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    float x, y;
    vector<pair<float, float> > PointList;
    while (InputStream >> x >> y) {
	Result = 1;

	float cx = CanvasXOf(x);
	float cy = CanvasYOf(y);
	PointList.push_back(make_pair(cx, cy));
    }
	
    if (PointList.size() > 0) {
	if (IsFilled) {
	    _ImageArea->DrawPolygon(PointList);
	}
	else {
	    _ImageArea->DrawPolygonFill(PointList);
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawCircleCommand(istream& InputStream, bool IsFilled)
{
    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    float x, y, r;
    if (InputStream >> x >> y >> r) {
	Result = 1;

	float cx = CanvasXOf(x);
	float cy = CanvasYOf(y);
	float crx = fabs(CanvasXOf(x + r) - CanvasXOf(x));
	float cry = fabs(CanvasYOf(y + r) - CanvasYOf(y));
	
	if (IsFilled) {
	    _ImageArea->DrawEllipseFill(cx, cy, crx, cry);
	}
	else {
	    _ImageArea->DrawEllipse(cx, cy, crx, cry);
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawEllipseCommand(istream& InputStream, bool IsFilled)
{
    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    float x, y, rx, ry, angle;
    if (InputStream >> x >> y >> rx >> ry) {
	Result = 1;
	
	if (! (InputStream >> angle)) {
	    angle = 0;
	}

	float cx = CanvasXOf(x);
	float cy = CanvasYOf(y);
	float crx = fabs(CanvasXOf(x + rx) - CanvasXOf(x));
	float cry = fabs(CanvasYOf(y + ry) - CanvasYOf(y));
	
	if (IsFilled) {
	    _ImageArea->DrawEllipseFill(cx, cy, crx, cry, angle);
	}
	else {
	    _ImageArea->DrawEllipse(cx, cy, crx, cry, angle);
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawArrowTipCommand(istream& InputStream, bool IsFilled)
{
    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();

    float x0, y0, x1, y1, TipWidth, TipLength;
    if (InputStream >> x0 >> y0 >> x1 >> y1 >> TipWidth >> TipLength) {
	Result = 1;

	float cx0 = CanvasXOf(x0);
	float cy0 = CanvasYOf(y0);
	float cx1 = CanvasXOf(x1);
	float cy1 = CanvasYOf(y1);
	
	float Angle = atan2(cy1 - cy0, cx1 - cx0);
	float xa0 = cx1 + (-TipLength * cos(Angle) - TipWidth/2 * sin(Angle));
	float ya0 = cy1 + (-TipLength * sin(Angle) + TipWidth/2 * cos(Angle));
	float xa1 = cx1 + (-TipLength * cos(Angle) + TipWidth/2 * sin(Angle));
	float ya1 = cy1 + (-TipLength * sin(Angle) - TipWidth/2 * cos(Angle));

	vector<pair<float, float> > PointList;
	PointList.push_back(make_pair(xa0, ya0));
	PointList.push_back(make_pair(cx1, cy1));
	PointList.push_back(make_pair(xa1, ya1));

	if (IsFilled) {
	    _ImageArea->DrawPolygonFill(PointList);
	}
	else {
	    _ImageArea->DrawLines(PointList);
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawTextCommand(istream& InputStream)
{
    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldFontIndex = _ImageArea->SetFont(_FontIndex);

    float x, y;
    if (InputStream >> x >> y) {
	Result = 1;

	float cx = CanvasXOf(x);
	float cy = CanvasYOf(y);
	
	string Text;
	InputStream >> ws;
	if (getline(InputStream, Text, '\n')) {
	    if (strncmp(Text.c_str(), "math:", 5) == 0) {
		Text.erase(0, 5);
		_Typesetter->Typeset(
		    cx, cy, _MathTextComposer->Compose(Text), _TextAdjustment
		);
	    }
	    else {
		_ImageArea->DrawText(cx, cy, Text, _TextAdjustment);
	    }
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetFont(OldFontIndex);

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawImageCommand(std::istream& InputStream)
{
    int Result = 0;

    float x, y;
    string FileName;
    if (InputStream >> x >> y >> FileName) {
	TKinokoCanvasImage* Image = _ImageArea->LoadImage(FileName);
	if (Image == 0) {
	    return Result;
	}
	Result = 1;

	float cx0 = CanvasXOf(x);
	float cy0 = CanvasYOf(y);
	float cx1 = cx0 + Image->Width();
	float cy1 = cy0 + Image->Height();

	float cxmax = max(CanvasXOf(_XMin), CanvasXOf(_XMax));
	float cxmin = min(CanvasXOf(_XMin), CanvasXOf(_XMax));
	float cymax = max(CanvasYOf(_YMin), CanvasYOf(_YMax));
	float cymin = min(CanvasYOf(_YMin), CanvasYOf(_YMax));

	if ((cx0 < cxmin) || (cx1 > cxmax) || (cy0 < cymin) || (cy1 > cymax)) {
	    return Result;
	}

	_ImageArea->DrawImage(cx0, cy0, Image);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessLocateCommand(istream& InputStream)
{
    int Result = 0;

    float x, y;
    if (InputStream >> x) {
	_CursorX0 = CanvasXOf(x);
	_CursorX = _CursorX0;

	if (InputStream >> y) {
	    _CursorY = CanvasYOf(y);
	}

	Result = 1;
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessPrintCommand(istream& InputStream, bool FeedsLine)
{
    int OldFontIndex = _ImageArea->SetFont(_FontIndex);
    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    string TextAdjustment = "tl";

    string Text;
    InputStream >> ws;
    if (getline(InputStream, Text, '\n')) {
	if (strncmp(Text.c_str(), "math:", 5) == 0) {
	    Text.erase(0, 5);
	    const TKinokoCanvasTextComposition& Composition = (
		_MathTextComposer->Compose(Text)
	    );
	    _Typesetter->Typeset(
		_CursorX, _CursorY, Composition, TextAdjustment
	    );
	    _CursorX += Composition.Width();
	}
	else {
	    _ImageArea->DrawText(_CursorX, _CursorY, Text, TextAdjustment);
	    _CursorX += _ImageArea->TextWidthOf(Text);
	}
    }
    
    if (FeedsLine) {
	_CursorX = _CursorX0;
	_CursorY += 1.2 * _LineHeight;
    }

    _ImageArea->SetFont(OldFontIndex);
    _ImageArea->SetColor(OldColorIndex);

    return 1;
}

int TKinokoCanvasDrawableObject::ProcessClearCommand(istream& InputStream)
{
    Clear();
    return 1;
}

int TKinokoCanvasDrawableObject::ProcessSetFontCommand(istream& InputStream)
{
    int Result = 0;

    string FontName;
    int FontSize;
    if (InputStream >> FontName >> FontSize) {
	Result = SetFont(FontName, FontSize);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessSaveContextCommand(istream& InputStream)
{
    return SaveContext();
}

int TKinokoCanvasDrawableObject::ProcessRestoreContextCommand(istream& InputStream)
{
    return RestoreContext();
}

float TKinokoCanvasDrawableObject::CanvasXOf(float x)
{
    return (x - _XMin) / _XWidth * _Width + _OffsetX;
}

float TKinokoCanvasDrawableObject::CanvasYOf(float y)
{
    return (y - _YMin) / _YWidth * _Height + _OffsetY;
}

float TKinokoCanvasDrawableObject::XOf(float CanvasX)
{
    return (CanvasX - _OffsetX) / _Width * _XWidth + _XMin;
}

float TKinokoCanvasDrawableObject::YOf(float CanvasY)
{
    return (CanvasY - _OffsetY) / _Height * _YWidth + _YMin;
}

int TKinokoCanvasDrawableObject::SetCoordinate(float XMin, float YMin, float XWidth, float YWidth)
{
    _XMin = XMin;
    _YMin = YMin;
    _XWidth = XWidth;
    _YWidth = YWidth;

    _XMax = XMin + XWidth;
    _YMax = YMin + YWidth;

    return 1;
}

int TKinokoCanvasDrawableObject::SetForegroundColor(const string& ColorName)
{
    int ColorIndex = _ImageArea->AllocateColor(ColorName);

    if (ColorIndex >= 0) {
	_ColorIndexList[Color_Foreground] = ColorIndex;
	return 1;
    }
    else {
	return 0;
    }
}

int TKinokoCanvasDrawableObject::SetBackgroundColor(const string& ColorName)
{
    int ColorIndex = _ImageArea->AllocateColor(ColorName);

    if (ColorIndex >= 0) {
	_ColorIndexList[Color_Background] = ColorIndex;
	return 1;
    }
    else {
	return 0;
    }
}

int TKinokoCanvasDrawableObject::SetColor(const string& ColorName)
{
    int ColorIndex;
    if (ColorName.empty() || (ColorName == "default")) {
	ColorIndex = _ColorIndexList[Color_Foreground];
    }
    else {
	ColorIndex = _ImageArea->AllocateColor(ColorName);
    }

    if (ColorIndex >= 0) {
	_ColorIndex = ColorIndex;
	return 1;
    }
    else {
	return 0;
    }
}

int TKinokoCanvasDrawableObject::SetFont(string FontName, int FontSize)
{
    int FontIndex;
    if (FontName.empty() || (FontName == "default")) {
	FontIndex = _FontIndexList[Font_Normal];
	FontName = _FontList[Font_Normal].first;
	FontSize = _FontList[Font_Normal].second;
    }
    else {
	FontIndex = _ImageArea->LoadFont(FontName, FontSize);
    }

    if (FontIndex >= 0) {
	_FontIndex = FontIndex;
    }
    else {
	return 0;
    }

    if (_LineHeightTable.count(_FontIndex) == 0) {
	int OldFontIndex = _ImageArea->SetFont(_FontIndex);
	_LineHeightTable[_FontIndex] = _ImageArea->TextHeightOf("Ap");
	_ImageArea->SetFont(OldFontIndex);
    }
    _LineHeight = _LineHeightTable[_FontIndex];

    _MathTextComposer->SetFont(FontName, FontSize);

    return 1;
}

int TKinokoCanvasDrawableObject::SetLineWidth(const string& LineWidthString)
{
    int Value;
    if (istringstream(LineWidthString) >> Value) {
	_LineWidth = Value;
    }
    else {
	return 0;
    }

    return 1;
}

int TKinokoCanvasDrawableObject::SetLineStyle(const string& LineStyleString)
{
    _LineStyleIndex = _ImageArea->CreateLineStyle(LineStyleString);

    return 1;
}

int TKinokoCanvasDrawableObject::SetTextAdjustment(const std::string& AdjustmentString)
{
    _TextAdjustment = AdjustmentString;

    return 1;
}

int TKinokoCanvasDrawableObject::SetMarker(string MarkerName)
{
    string Parameter;
    size_t NameLength = MarkerName.find_first_of(":");
    if (NameLength != string::npos) {
	Parameter = MarkerName.substr(NameLength + 1, string::npos);
	MarkerName = MarkerName.substr(0, NameLength);
    }

    if (_Marker) {
	delete _Marker;
    }
    _Marker = _MarkerFactory->CreateMarker(MarkerName, Parameter);

    return _Marker ? 1 : 0;
}

int TKinokoCanvasDrawableObject::SetMarkerSize(const string& SizeString)
{
    int Value = 0;
    istringstream(SizeString) >> Value;

    if (Value > 0) {
	_MarkerSize = Value;
    }
    else {
	return 0;
    }

    return 1;
}

int TKinokoCanvasDrawableObject::SaveContext(void)
{
    _ColorIndexStack.push_back(_ColorIndex);
    _LineWidthStack.push_back(_LineWidth);
    _LineStyleIndexStack.push_back(_LineStyleIndex);
    _FontIndexStack.push_back(_FontIndex);
    _TextAdjustmentStack.push_back(_TextAdjustment);
    _LineHeightStack.push_back(_LineHeight);

    if (_Marker) {
	_MarkerStack.push_back(_Marker->Clone());
    }
    else {
	_MarkerStack.push_back(0);
    }
    _MarkerSizeStack.push_back(_MarkerSize);

    return 1;
}

int TKinokoCanvasDrawableObject::RestoreContext(void)
{
    if (_ColorIndexStack.empty()) {
	return 0;
    }

    delete _Marker;

    _ColorIndex = _ColorIndexStack.back();
    _LineWidth = _LineWidthStack.back();
    _LineStyleIndex = _LineStyleIndexStack.back();
    _FontIndex = _FontIndexStack.back();
    _TextAdjustment = _TextAdjustmentStack.back();
    _LineHeight = _LineHeightStack.back();
    _Marker = _MarkerStack.back();
    _MarkerSize = _MarkerSizeStack.back();

    _ColorIndexStack.pop_back();
    _LineWidthStack.pop_back();
    _LineStyleIndexStack.pop_back();
    _FontIndexStack.pop_back();
    _TextAdjustmentStack.pop_back();
    _LineHeightStack.pop_back();
    _MarkerStack.pop_back();
    _MarkerSizeStack.pop_back();

    return 1;
}

int TKinokoCanvasDrawableObject::PopContextStack(void)
{
    if (_ColorIndexStack.empty()) {
	return 0;
    }

    delete _MarkerStack.back();

    _ColorIndexStack.pop_back();
    _LineWidthStack.pop_back();
    _LineStyleIndexStack.pop_back();
    _FontIndexStack.pop_back();
    _TextAdjustmentStack.pop_back();
    _LineHeightStack.pop_back();
    _MarkerStack.pop_back();
    _MarkerSizeStack.pop_back();

    return 1;
}

void TKinokoCanvasDrawableObject::Clear(void)
{
    _CursorX = 0;
    _CursorY = 0;
    _CursorX0 = 0;

    TKinokoCanvasObject::Clear();
}

