/* KinokoCanvasPlotObject.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 23 January 2008. */


#include <iostream>
#include <sstream>
#include <iomanip>
#include <algorithm>
#include <cmath>
#include "KinokoCanvas.hh"
#include "KinokoCanvasDataScale.hh"
#include "KinokoCanvasColorScale.hh"
#include "KinokoCanvasFramedObject.hh"
#include "KinokoCanvasPlotObject.hh"

using namespace std;


#define clip(x, lower, upper) (min(max((lower), (x)), (upper)))
inline float max(float x, double y) { return max((double) x, y); }


TKinokoCanvasPlotObject::TKinokoCanvasPlotObject(TKinokoCanvas* Canvas)
: TKinokoCanvasFramedObject(Canvas)
{
    _XScale = 0;
    _YScale = 0;

    _Y2Scale = 0;

    _IsCommentDisabled = false;
    _IsUpdatingSuspended = false;
    _CommentBoxColorIndex = -1;

    _IsCommandReplayEnabled = false;
    _IsCommandReplaySuspended = false;
}

TKinokoCanvasPlotObject::~TKinokoCanvasPlotObject()
{
    delete _XScale;
    delete _YScale;

    delete _Y2Scale;
}

TKinokoShellObject* TKinokoCanvasPlotObject::Clone(void)
{
    return new TKinokoCanvasPlotObject(_Canvas);
}

void TKinokoCanvasPlotObject::Initialize(void)
{
    TKinokoCanvasFramedObject::Initialize();

    _BottomMargin += (int) (2.2 * _LabelCharacterHeight);
    _RightMargin += (int) (1.5 * _LabelCharacterWidth);
    _LeftMargin += (int) (4.5 * _LabelCharacterWidth);

    _IsXScaleLog = false;
    _IsYScaleLog = false;

    _IsXScaleLabelDisabled = false;
    _IsYScaleLabelDisabled = false;

    _IsYTitleVertical = true;
    _IsScaleFactoringEnabled = /**/ false /*/ true /**/;

    _NumberOfXScaleLabels = 0;
    _NumberOfYScaleLabels = 0;
    SetCoordinate(0, 0, 1, 1);
    _XScale = new TKinokoCanvasDataScale(0, 1, _NumberOfXScaleLabels);
    _YScale = new TKinokoCanvasDataScale(0, 1, _NumberOfYScaleLabels);
}

void TKinokoCanvasPlotObject::Clear(void)
{
    TKinokoCanvasFramedObject::Clear();

    if (_IsCommandReplayEnabled) {
	ClearReplayCommandList();
    }
}

vector<string> TKinokoCanvasPlotObject::ActionList(void)
{
    vector<string> ActionList = TKinokoCanvasFramedObject::ActionList();

    ActionList.push_back("setYScaleLinear");
    ActionList.push_back("setYScaleLog");
    //ActionList.push_back("suspendUpdating");
    //ActionList.push_back("restartUpdating");
    ActionList.push_back("hideCommentBox");
    ActionList.push_back("showCommentBox");
    ActionList.push_back("resetRange");

    return ActionList;
}

int TKinokoCanvasPlotObject::ProcessAction(const string& ActionName)
{
    int Result = 1;

    if (ActionName == "suspendUpdating") {
	_IsUpdatingSuspended = true;
    }
    else if (ActionName == "restartUpdating") {
	_IsUpdatingSuspended = false;
    }
    else {
	Result = ProcessDrawingAction(ActionName);
	if (Result <= 0) {
	    Result = TKinokoCanvasFramedObject::ProcessAction(ActionName);
	}
    }

    if ((Result > 0) && (_IsCommandReplayEnabled)) {
	ReplayCommand();
    }

    return Result;
}

int TKinokoCanvasPlotObject::ProcessDrawingAction(const string& ActionName)
{
    int Result = 1;

    if (ActionName == "setYScaleLinear") {
	SetYScaleLinear();
    }
    else if (ActionName == "setYScaleLog") {
	SetYScaleLog();
    }
    else if (ActionName == "hideCommentBox") {
	_IsCommentDisabled = true;
    }
    else if (ActionName == "showCommentBox") {
	_IsCommentDisabled = false;
    }
    else if (ActionName == "resetRange") {
	ThawFrame();
    }
    else {
	Result = 0;
    }

    return Result;
}

int TKinokoCanvasPlotObject::ProcessRangeSelect(int X0, int Y0, int X1, int Y1)
{
    int Result = 0;

    if (! _IsFrameValid) {
	return Result;
    }

    float XMin = XOf(min(X0, X1)), XMax = XOf(max(X0, X1));
    float YMin = YOf(max(Y0, Y1)), YMax = YOf(min(Y0, Y1));
    
    if ((X0 == X1) && (Y0 == Y1)) {
	ostringstream os;
	os << "(" << setprecision(4) << XMin << ", ";
	os << setprecision(4) << YMin << ")";

	//... BUG: this will disappear on next "drawXXX"
	_ImageArea->DrawCircle(X0, Y0, 1);
	_ImageArea->DrawText(X0+2, Y0+2, os.str(), "tl");
	_ImageArea->Redraw();

	return Result = 1;
    }
    else if ((X0 == X1) || (Y0 == Y1)) {
	return Result;
    }

    if ((XMin > _XMax) || (XMax < _XMin) || (YMin > _YMax) || (YMax < _YMin)) {
	return Result;
    }
	
    FreezeFrame(
	clip(XMin, _XMin, _XMax), clip(XMax, _XMin, _XMax),
	clip(YMin, _YMin, _YMax), clip(YMax, _YMin, _YMax)
    );
    Result = 1;

    //... BUG? Check This ...//
    if (_Y2Scale) {
	float Y2Min = Y2Of(min(Y0, Y1)), Y2Max = Y2Of(max(Y0, Y1));
	if ((Y2Min > _Y2Max) || (Y2Max < _Y2Min)) {
	    ;
	}
	else {
	    SetY2Axis(Y2Min, Y2Max - Y2Min, _Y2Title);
	}
    }

    if (_IsCommandReplayEnabled) {
	ReplayCommand();
    }

    return Result;
}

int TKinokoCanvasPlotObject::ProcessCommand(const string& Command, istream& InputStream)
{
    istream* ParameterStream = &InputStream;
    string ParameterString;
    istream* DuplicatedInputStream = 0;
    if (_IsCommandReplayEnabled && ! _IsCommandReplaySuspended) {
	getline(InputStream, ParameterString, ';');
	DuplicatedInputStream = new istringstream(ParameterString);
	ParameterStream = DuplicatedInputStream;
    }

    int Result;
    if (_IsUpdatingSuspended) {
	Result = 1;
    }
    else {
	Result = ProcessDrawingCommand(Command, *ParameterStream);
	if (Result <= 0) {
	    Result = TKinokoCanvasFramedObject::ProcessCommand(
		Command, *ParameterStream
	    );
	}
    }

    if (DuplicatedInputStream) {
	AddReplayCommand(Command, ParameterString);
	delete DuplicatedInputStream;
    }

    return Result;
}

int TKinokoCanvasPlotObject::ProcessDrawingCommand(const std::string& Command, std::istream& InputStream)
{
    int Result = 0;

    if (Command == "plot") {
	Result = ProcessPlotCommand(InputStream, Style_Points | Style_Lines);
    }
    else if (Command == "linesplot") {
	Result = ProcessPlotCommand(InputStream, Style_Lines);
    }
    else if (Command == "pointsplot") {
	Result = ProcessPlotCommand(InputStream, Style_Points);
    }
    else if (Command == "filledplot") {
	Result = ProcessPlotCommand(InputStream, Style_Filled);
    }
    else if (Command == "errorplot") {
	Result = ProcessErrorPlotCommand(InputStream);
    }
    else if (Command == "minmaxplot") {
	Result = ProcessMinmaxPlotCommand(InputStream);
    }
    else if (Command == "hist") {
	Result = ProcessHistCommand(InputStream);
    }
    else if (Command == "histfill") {
	Result = ProcessHistCommand(InputStream, Style_Filled);
    }
    else if (Command == "bandhist") {
	Result = ProcessBandHistCommand(InputStream);
    }
    else if (Command == "comment") {
	Result = ProcessCommentCommand(InputStream);
    }
    else if (Command == "drawxgrid") {
	Result = ProcessDrawGridCommand(InputStream, Axis_X);
    }
    else if (Command == "drawygrid") {
	Result = ProcessDrawGridCommand(InputStream, Axis_Y);
    }
    else if (Command == "drawxscalelabel") {
	Result = ProcessDrawXScaleLabelCommand(InputStream);
    }
    else if (Command == "drawyscalelabel") {
	Result = ProcessDrawYScaleLabelCommand(InputStream);
    }
    else if (Command == "setxscaletimetick") {
	Result = ProcessSetXScaleTimeTickCommand(InputStream);
    }
    else if (Command == "setxscaleelapsetick") {
	Result = ProcessSetXScaleElapseTickCommand(InputStream);
    }
    else if (Command == "sety2axis") {
	Result = ProcessSetY2AxisCommand(InputStream);
    }

    return Result;
}

int TKinokoCanvasPlotObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "xscale") {
	if (Value == "log") {
	    SetXScaleLog();
	}
	else {
	    SetXScaleLinear();
	}
    }
    else if (Name == "yscale") {
	if (Value == "log") {
	    SetYScaleLog();
	}
	else {
	    SetYScaleLinear();
	}
    }
    else if (Name == "xtitle") {
	_XTitle = Value;
    }
    else if (Name == "ytitle") {
	_YTitle = Value;
    }
    else if (Name == "xscalelabel") {
	_IsXScaleLabelDisabled = (Value == "disabled");
    }
    else if (Name == "yscalelabel") {
	_IsYScaleLabelDisabled = (Value == "disabled");
    }
    else if (Name == "commentbox") {
	_IsCommentDisabled = (Value == "hidden");
    }
    else if (Name == "commentboxcolor") {
	if (Value.empty()) {
	    _CommentBoxColorIndex = -1;
	}
	else if (Value == "background") {
	    _CommentBoxColorIndex = _ColorIndexList[Color_Background];
	}
	else {
	    _CommentBoxColorIndex = _ImageArea->AllocateColor(Value);
	}
    }
    else if (Name == "commandreplay") {
	_IsCommandReplayEnabled = (Value == "enabled");
    }
    else if (Name == "scalefactoring") {
	bool IsScaleFactoringEnabled = (Value == "enabled");
	if (IsScaleFactoringEnabled && (! _IsScaleFactoringEnabled)) {
	    _RightMargin += (int) (2.3 * _LabelCharacterWidth);
	}
	if ((! IsScaleFactoringEnabled) && _IsScaleFactoringEnabled) {
	    _RightMargin -= (int) (2.3 * _LabelCharacterWidth);
	}
	_IsScaleFactoringEnabled = IsScaleFactoringEnabled;
	SetPosition(_OffsetX, _OffsetY, _Width, _Height);
    }
    else if (Name == "ytitleorientation") {
	//... temporary
	bool IsOldYTitleVertical = _IsYTitleVertical;
	_IsYTitleVertical = (Value == "vertical");
	if ((! IsOldYTitleVertical) && _IsYTitleVertical) {
	    _LeftMargin += 2 * _LabelCharacterWidth;
	    _RightMargin += 2 * _LabelCharacterWidth;
	}
	else if (IsOldYTitleVertical && (! _IsYTitleVertical)) {
	    _LeftMargin -= 2 * _LabelCharacterWidth;
	    _RightMargin -= 2 * _LabelCharacterWidth;
	}
	SetPosition(_OffsetX, _OffsetY, _Width, _Height);
    }
    else {
	Result = TKinokoCanvasFramedObject::ProcessSetCommand(Name, Value);
    }

    return Result;
}

float TKinokoCanvasPlotObject::CanvasXOf(float x)
{
    if (! _IsXScaleLog) {
	return (x - _XMin) / _XWidth * _FrameWidth + _FrameOffsetX;
    }
    else {
	return (log10(x) - _LogXMin) / _LogXWidth * _FrameWidth + _FrameOffsetX;
    }
}

float TKinokoCanvasPlotObject::CanvasYOf(float y)
{
    if (! _IsYScaleLog) {
	return (1.0 - (y - _YMin) / _YWidth) * _FrameHeight + _FrameOffsetY;
    }
    else {
	return (1.0 - (log10(y) - _LogYMin) / _LogYWidth) * _FrameHeight + _FrameOffsetY;
    }
}

float TKinokoCanvasPlotObject::CanvasY2Of(float y)
{
    if (_Y2Scale == 0) {
	return CanvasYOf(y);
    }

    if (! _IsYScaleLog) {
	return (1.0 - (y - _Y2Min) / _Y2Width) * _FrameHeight + _FrameOffsetY;
    }
    else {
	return (1.0 - (log10(y) - _LogY2Min) / _LogY2Width) * _FrameHeight + _FrameOffsetY;
    }
}

float TKinokoCanvasPlotObject::XOf(float CanvasX)
{
    if (! _IsXScaleLog) {
	return (CanvasX - _FrameOffsetX) / _FrameWidth * _XWidth + _XMin;
    }
    else {
	return pow(10, (CanvasX - _FrameOffsetX) / _FrameWidth * _LogXWidth + _LogXMin);
    }
}

float TKinokoCanvasPlotObject::YOf(float CanvasY)
{
    if (! _IsYScaleLog) {
	return (1.0 - (CanvasY - _FrameOffsetY) / _FrameHeight) * _YWidth + _YMin;
    }
    else {
	return pow(10, (1.0 - (CanvasY - _FrameOffsetY) / _FrameHeight) * _LogYWidth + _LogYMin);    
    }
}

float TKinokoCanvasPlotObject::Y2Of(float CanvasY)
{
    if (_Y2Scale == 0) {
	return YOf(CanvasY);
    }

    if (! _IsYScaleLog) {
	return (1.0 - (CanvasY - _FrameOffsetY) / _FrameHeight) * _Y2Width + _Y2Min;
    }
    else {
	return pow(10, (1.0 - (CanvasY - _FrameOffsetY) / _FrameHeight) * _LogY2Width + _LogY2Min);    
    }
}

int TKinokoCanvasPlotObject::SetCoordinate(float XMin, float YMin, float XWidth, float YWidth)
{
    _XMinOrg = XMin;
    _XWidthOrg = XWidth;
    _YMinOrg = YMin;
    _YWidthOrg = YWidth;

    if (_IsXScaleLog) {
	if (XMin <= 0) {
	    float XMax = XMin + XWidth;
	    float XMaxForLog = (XMax <= 0) ? 1.0 : XMax;
	    float XMinForLog = min(XMaxForLog / 100.0, 0.5);
	    XMin = XMinForLog;
	    XWidth = XMaxForLog - XMinForLog;
	}

	_LogXMin = log10(XMin);
	_LogXMax = log10(XMin + XWidth);
	_LogXWidth = _LogXMax - _LogXMin;
    }
    
    if (_IsYScaleLog) {
	if (YMin <= 0) {
	    float YMax = YMin + YWidth;
	    float YMaxForLog = (YMax <= 0) ? 1.0 : YMax;
	    float YMinForLog = min(YMaxForLog / 100.0, 0.5);
	    YMin = YMinForLog;
	    YWidth = YMaxForLog - YMinForLog;
	}

	_LogYMin = log10(YMin);
	_LogYMax = log10(YMin + YWidth);
	_LogYWidth = _LogYMax - _LogYMin;
    }

    TKinokoCanvasFramedObject::SetCoordinate(XMin, YMin, XWidth, YWidth);

    return 1;
}

int TKinokoCanvasPlotObject::SetY2Axis(float YMin, float YWidth, const string& Title)
{
    _Y2Min = YMin;
    _Y2Max = YMin + YWidth;
    _Y2Width = YWidth;

    _LogY2Min = log10(YMin);
    _LogY2Max = log10(YMin + YWidth);
    _LogY2Width = log10(YWidth);
    
    _Y2Title = Title;

    delete _Y2Scale;
    _Y2Scale = new TKinokoCanvasDataScale(0, 1, _NumberOfYScaleLabels);

    return 1;
}

int TKinokoCanvasPlotObject::SetXScaleLinear(void)
{
    _IsXScaleLog = false;
    SetCoordinate(_XMinOrg, _YMinOrg, _XWidthOrg, _YWidthOrg);
    delete _XScale;
    _XScale = new TKinokoCanvasDataScale(
	_XMin, _XMax, _NumberOfXScaleLabels
    );

    return 1;
}

int TKinokoCanvasPlotObject::SetXScaleLog(void)
{
    _IsXScaleLog = true;
    SetCoordinate(_XMinOrg, _YMinOrg, _XWidthOrg, _YWidthOrg);
    delete _XScale;
    _XScale = new TKinokoCanvasLogDataScale(
	_XMin, _XMax, _NumberOfXScaleLabels
    );

    return 1;
}

int TKinokoCanvasPlotObject::SetXScaleTime(long Epoch, const string& Format, int Unit, int Step)
{
    _IsXScaleLog = false;
    SetCoordinate(_XMinOrg, _YMinOrg, _XWidthOrg, _YWidthOrg);

    TKinokoCanvasTimeDataScale* Scale = new TKinokoCanvasTimeDataScale(
	_XMin, _XMax, _NumberOfXScaleLabels
    );
    Scale->SetTimeTick(Epoch, Format, Unit, Step);

    delete _XScale;
    _XScale = Scale;

    return 1;
}

int TKinokoCanvasPlotObject::SetXScaleElapse(void)
{
    _IsXScaleLog = false;
    SetCoordinate(_XMinOrg, _YMinOrg, _XWidthOrg, _YWidthOrg);

    TKinokoCanvasElapseDataScale* Scale = new TKinokoCanvasElapseDataScale(
	_XMin, _XMax, _NumberOfXScaleLabels
    );

    delete _XScale;
    _XScale = Scale;

    return 1;
}

int TKinokoCanvasPlotObject::SetYScaleLinear(void)
{
    _IsYScaleLog = false;
    SetCoordinate(_XMinOrg, _YMinOrg, _XWidthOrg, _YWidthOrg);
    delete _YScale;
    _YScale = new TKinokoCanvasDataScale(
	_YMin, _YMax, _NumberOfYScaleLabels
    );

    if (_Y2Scale) {
	delete _Y2Scale;
	_Y2Scale = new TKinokoCanvasDataScale(
	    _Y2Min, _Y2Max, _NumberOfYScaleLabels
	);
    }

    return 1;
}

int TKinokoCanvasPlotObject::SetYScaleLog(void)
{
    _IsYScaleLog = true;
    SetCoordinate(_XMinOrg, _YMinOrg, _XWidthOrg, _YWidthOrg);
    delete _YScale;
    _YScale = new TKinokoCanvasLogDataScale(
	_YMin, _YMax, _NumberOfYScaleLabels
    );

    if (_Y2Scale) {
	delete _Y2Scale;
	_Y2Scale = new TKinokoCanvasLogDataScale(
	    _Y2Min, _Y2Max, _NumberOfYScaleLabels
	);
    }

    return 1;
}

int TKinokoCanvasPlotObject::ProcessPlotCommand(istream& InputStream, int Style)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);

    double x1, y1;
    float x0, y0, xc, yc;
    float cx, cy;
    bool IsFirstPoint = true, IsLastPointInRange, IsThisPointInRange;
    vector<pair<float, float> > PointList;
    while (InputStream >> x1 >> y1) {
	IsThisPointInRange = (
	    (x1 >= _XMin) && (x1 <= _XMax) && (y1 >= _YMin) && (y1 <= _YMax)
	);

	if (IsThisPointInRange) {
	    cx = CanvasXOf(x1);
	    cy = CanvasYOf(y1);
	}

	if (Style & Style_Points) {
	    if (IsThisPointInRange) {
		if (_Marker) {
		    _Marker->Draw(_ImageArea, cx, cy, _MarkerSize);
		}
		else {
		    _ImageArea->DrawRect(cx - 2, cy - 2, cx + 2, cy + 2);
		}
	    }
	}

	if (Style & (Style_Lines | Style_Filled)) {
	    if (IsFirstPoint) {
		IsFirstPoint = false;
		if (IsThisPointInRange) {
		    PointList.push_back(make_pair(cx, cy));
		}
	    }
	    else if (IsLastPointInRange && IsThisPointInRange) {
		PointList.push_back(make_pair(cx, cy));
	    }
	    else if (IsLastPointInRange) {
		xc = x1; yc = y1;
		if (xc > _XMax) {
		    yc = (_XMax - x0) / (xc - x0) * (yc - y0) + y0;
		    xc = _XMax;
		}
		else if (xc < _XMin) {
		    yc = (_XMin - x0) / (xc - x0) * (yc - y0) + y0;
		    xc = _XMin;
		}
		if (yc > _YMax) {
		    xc = (_YMax - y0) / (yc - y0) * (xc - x0) + x0;
		    yc = _YMax;
		}
		else if (yc < _YMin) {
		    xc = (_YMin - y0) / (yc - y0) * (xc - x0) + x0;
		    yc = _YMin;
		}
		PointList.push_back(
		    make_pair(CanvasXOf(xc), CanvasYOf(yc))
		);
	    }
	    else if (IsThisPointInRange) {
		xc = x0; yc = y0;
		if (xc > _XMax) {
		    yc = (_XMax - x1) / (xc - x1) * (yc - y1) + y1;
		    xc = _XMax;
		}
		else if (xc < _XMin) {
		    yc = (_XMin - x1) / (xc - x1) * (yc - y1) + y1;
		    xc = _XMin;
		}
		if (yc > _YMax) {
		    xc = (_YMax - y1) / (yc - y1) * (xc - x1) + x1;
		    yc = _YMax;
		}
		else if (yc < _YMin) {
		    xc = (_YMin - y1) / (yc - y1) * (xc - x1) + x1;
		    yc = _YMin;
		}
		//... BUG:
		// (xc, yc) might be on a different side from the exit point.
		PointList.push_back(
		    make_pair(CanvasXOf(xc), CanvasYOf(yc))
		);
		PointList.push_back(make_pair(cx, cy));
	    }
	    else {
		//... BUG: the segment might clip a corner.
	    }
	}

	x0 = x1; y0 = y1;
	IsLastPointInRange = IsThisPointInRange;
    }

    if (! PointList.empty()) {
	int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
	int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

	if (Style & Style_Lines) {
	    _ImageArea->DrawLines(PointList);
	}
	if (Style & Style_Filled) {
	    _ImageArea->DrawPolygonFill(PointList);
	}

	_ImageArea->SetLineWidth(OldLineWidth);
	_ImageArea->SetLineStyle(OldLineStyleIndex);
    }

    _ImageArea->SetColor(OldColorIndex);

    return Result = 1;
}

int TKinokoCanvasPlotObject::ProcessErrorPlotCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();

    double x, y, ex, ey;
    float cx, cy, cx0, cy0, cx1, cy1;
    while (InputStream >> x >> y >> ex >> ey) {
	if ((x < _XMin) || (x > _XMax)) {
	    continue;
	}
	if ((y < _YMin) || (y > _YMax)) {
	    continue;
	}

	cx = CanvasXOf(x);
	cx0 = CanvasXOf(clip(x - ex, _XMin, _XMax));
	cx1 = CanvasXOf(clip(x + ex, _XMin, _XMax));

	cy = CanvasYOf(y);
	cy0 = CanvasYOf(clip(y - ey, _YMin, _YMax));
	cy1 = CanvasYOf(clip(y + ey, _YMin, _YMax));

	if (cx0 != cx1) {
	    _ImageArea->DrawLine(cx0, cy, cx1, cy);
	}
	if (cy0 != cy1) {
	    _ImageArea->DrawLine(cx, cy0, cx, cy1);
	}

	if (_Marker) {
	    _Marker->Draw(_ImageArea, cx, cy, _MarkerSize);
	}
	else if ((cx0 == cx1) || (cy0 == cy1)) {
	    _ImageArea->DrawRect(cx - 2, cy - 2, cx + 2, cy + 2);
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result = 1;
}

int TKinokoCanvasPlotObject::ProcessMinmaxPlotCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();

    double x, y, ymin, ymax, deviation;
    float cx, cy, cymin, cymax, cy0, cy1;
    while (InputStream >> x >> y >> ymin >> ymax >> deviation) {
	if ((x < _XMin) || (x > _XMax)) {
	    continue;
	}
	if ((y < _YMin) || (y > _YMax)) {
	    continue;
	}

	cx = CanvasXOf(x);
	cy = CanvasYOf(y);
	cymin = CanvasYOf(clip(ymin, _YMin, _YMax));
	cymax = CanvasYOf(clip(ymax, _YMin, _YMax));
	cy0 = CanvasYOf(clip(y - deviation, _YMin, _YMax));
	cy1 = CanvasYOf(clip(y + deviation, _YMin, _YMax));

	_ImageArea->DrawRect(cx - 2, cy - 2, cx + 2, cy + 2);
	if (cy0 != cy1) {
	    _ImageArea->DrawLine(cx, cy0, cx, cy1);
	}

	_ImageArea->DrawLine(cx - 2, cymin - 2, cx + 2, cymin + 2);
	_ImageArea->DrawLine(cx - 2, cymin + 2, cx + 2, cymin - 2);
	_ImageArea->DrawLine(cx - 2, cymax - 2, cx + 2, cymax + 2);
	_ImageArea->DrawLine(cx - 2, cymax + 2, cx + 2, cymax - 2);
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result = 1;
}

int TKinokoCanvasPlotObject::ProcessHistCommand(istream& InputStream, int Style)
{
    // syntax: hist xmin step y0 y1 y2 y3 ...
    // [xmin, xmin + step) -> y0, [xmin + step, xmin + 2 step) -> y1, ...

    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    double xmin, xstep;
    if (! (InputStream >> xmin >> xstep)) {
	return 0;
    }

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    vector<pair<float, float> > PointList;

    double x = xmin, y;
    float cx0, cy0, cx1, cy1;
    while (InputStream >> y) {
	x += xstep;
	if ((x-xstep < _XMin) || (x > _XMax)) {
	    continue;
	}
	if (PointList.empty()) {
	    cx0 = CanvasXOf(x-xstep);
	    cy0 = CanvasYOf(_YMin);
	    PointList.push_back(make_pair(cx0, cy0));
	}
	y = clip(y, _YMin, _YMax);

	cx1 = CanvasXOf(x);
	cy1 = CanvasYOf(y);

	PointList.push_back(make_pair(cx0, cy1));
	PointList.push_back(make_pair(cx1, cy1));

	cx0 = cx1;
	cy0 = cy1;
    }

    cy1 = CanvasYOf(_YMin);
    PointList.push_back(make_pair(cx0, cy1));

    if (Style & Style_Filled) {
	if (PointList.size() > 2) {
	    PointList.push_back(PointList[0]);
	    _ImageArea->DrawPolygonFill(PointList);
	}
    }
    else {
	_ImageArea->DrawLines(PointList);
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result = 1;
}

int TKinokoCanvasPlotObject::ProcessBandHistCommand(istream& InputStream, int Style)
{
    // syntax: hist xmin step y0L, y0U, y1L, y1U, y2L, y2U, ...
    // [xmin, xmin + step) -> y0, [xmin + step, xmin + 2 step) -> y1, ...

    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    double xmin, xstep;
    if (! (InputStream >> xmin >> xstep)) {
	return 0;
    }

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    float cx0 = CanvasXOf(_XMin), cx1, cyL, cyU;
    vector<pair<float, float> > PointList, PointListL, PointListU;

    PointList.push_back(make_pair(cx0, CanvasYOf(_YMin)));

    float x;
    double yL, yU;
    while (InputStream >> yL >> yU) {
	x += xstep;
	if ((x-xstep < _XMin) || (x > _XMax)) {
	    continue;
	}
	yL = clip(yL, _YMin, _YMax);
	yU = clip(yU, _YMin, _YMax);

	cx1 = CanvasXOf(x);
	cyL = CanvasYOf(yL);
	cyU = CanvasYOf(yU);

	PointList.push_back(make_pair(cx0, cyL));
	PointListL.push_back(make_pair(cx0, cyL));
	PointListU.push_back(make_pair(cx0, cyU));

	PointList.push_back(make_pair(cx1, cyL));
	PointListL.push_back(make_pair(cx1, cyL));
	PointListU.push_back(make_pair(cx1, cyU));

	cx0 = cx1;
    }

    PointList.push_back(make_pair(cx1, CanvasYOf(_YMin)));
    _ImageArea->DrawPolygon(PointList);

    reverse(PointListL.begin(), PointListL.end());
    PointListU.insert(PointListU.end(), PointListL.begin(), PointListL.end());

    _ImageArea->DrawPolygonFill(PointListU);

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result = 1;
}

int TKinokoCanvasPlotObject::ProcessCommentCommand(istream& InputStream)
{
    static const int SideMargin = 7;

    if (! _IsFrameValid) {
	return 1;
    }

    if (_IsCommentDisabled) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Normal]);

    enum TTextType {
	TextType_Plain,
	TextType_Math,
	NumberOfTextTypes
    };

    double x, y;
    if (InputStream >> x >> y >> ws) {
	float x0 = fabs((x / 100.0) * _FrameWidth) + _FrameOffsetX;
	float y0 = fabs((y / 100.0) * _FrameHeight) + _FrameOffsetY;
	
	char Buffer[256];
	vector<string> TextList;
	vector<float> YOffsetList;
	vector<float> TextTypeList;
	float MaxWidth = 0, TotalHeight = 0;
	while (InputStream.getline(Buffer, sizeof(Buffer), '\n')) {
	    int Offset, TextType;
	    float Width;
	    if (strncmp(Buffer, "math:", 5) == 0) {
		TextType = TextType_Math;
		Offset = 5;
		const TKinokoCanvasTextComposition& Composition = (
		    _MathTextComposerList[Font_Normal]->Compose(Buffer + Offset)
		);
		Width = Composition.Width();
	    }
	    else if (strncmp(Buffer, "plain:", 6) == 0) {
		TextType = TextType_Plain;
		Offset = 6;
		Width = _ImageArea->TextWidthOf(Buffer + Offset);
	    }
	    else {
		TextType = TextType_Plain;
		Offset = 0;
		Width = _ImageArea->TextWidthOf(Buffer + Offset);
	    }

	    MaxWidth = max(MaxWidth, Width);
	    TotalHeight += 1.2 * _NormalFontLineHeight;

	    TextList.push_back(Buffer + Offset);
	    YOffsetList.push_back(TotalHeight);
	    TextTypeList.push_back(TextType);
	}
        MaxWidth += 2 * SideMargin;
	TotalHeight += SideMargin;

	if (x < 0) {
	    x0 = x0 - MaxWidth;
	}
	if (y < 0) {
	    y0 = y0 - TotalHeight;
	}

	if (_CommentBoxColorIndex > 0) {
	    int OldColor = _ImageArea->SetColor(_CommentBoxColorIndex);
	    _ImageArea->DrawRectFill(
		x0+1, y0+1, x0 + MaxWidth-1, y0 + TotalHeight-1
	    );
	    _ImageArea->SetColor(OldColor);
	}
	_ImageArea->DrawRect(x0, y0, x0 + MaxWidth, y0 + TotalHeight);

	for (unsigned i = 0; i < TextList.size(); i++) {
	    if (TextTypeList[i] == TextType_Math) {
		_Typesetter->Typeset(
		    x0 + SideMargin, y0 + YOffsetList[i],
		    _MathTextComposerList[Font_Normal]->Compose(
			TextList[i]
		    ), 
		    "bl"
		);
	    }
	    else {
		_ImageArea->DrawText(
		    x0 + SideMargin, y0 + YOffsetList[i], TextList[i], "bl"
		);
	    }
	}

	Result = 1;
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return Result;
}

int TKinokoCanvasPlotObject::ProcessDrawGridCommand(istream& InputStream, int AxisIndex)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(
	_LineStyleIndexList[LineStyle_Dot]
    );

    vector<double> TickList;
    
    double t;
    while (InputStream >> t) {
	TickList.push_back(t);
    }
    if (TickList.empty()) {
	if (AxisIndex == Axis_X) {
	    for (int i = 0; i < _XScale->NumberOfDivisions(); i++) {
		TickList.push_back(_XScale->DivisionValueOf(i));
	    }
	}
	else if (AxisIndex == Axis_Y) {
	    for (int i = 0; i < _YScale->NumberOfDivisions(); i++) {
		TickList.push_back(_YScale->DivisionValueOf(i));
	    }
	}
    }

    for (unsigned i = 0; i < TickList.size(); i++) {
	t = TickList[i];
	if (AxisIndex == Axis_X) {
	    if ((t > _XMin) && (t < _XMax)) {
		float cx = CanvasXOf(t);
		_ImageArea->DrawLine(
		    cx, _FrameOffsetY, cx, _FrameOffsetY + _FrameHeight
		);
	    }
	}
	else if (AxisIndex == Axis_Y) {
	    if ((t > _YMin) && (t < _YMax)) {
		float cy = CanvasYOf(t);
		_ImageArea->DrawLine(
		    _FrameOffsetX, cy, _FrameOffsetX + _FrameWidth, cy
		);
	    }
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return 1;
}

int TKinokoCanvasPlotObject::ProcessDrawXScaleLabelCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    double x;
    if ((InputStream >> x) && ((x >= _XMin) || (x <= _XMax))) {
	float cx = CanvasXOf(x);
	float y1 = _FrameOffsetY + _FrameHeight;

	string Label;
	InputStream >> ws;
	if (getline(InputStream, Label)) {
	    if (strncmp(Label.c_str(), "math:", 5) == 0) {
		_Typesetter->Typeset(
		    cx, y1 + 3,
		    _MathTextComposerList[Font_Label]->Compose(
			Label.substr(5, string::npos)
		    ), 
		    "tc"
		);
	    }
	    else {
		_ImageArea->DrawText(cx, y1 + 3, Label, "tc");
	    }
	}
	else {
	    _Typesetter->Typeset(
		cx, y1 + 3,
		_MathTextComposerList[Font_Label]->ComposeNumber(x),
		"tc"
	    );
	}
	_ImageArea->DrawLine(cx, y1 - 8, cx, y1);

	Result = 1;
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return Result;
}

int TKinokoCanvasPlotObject::ProcessDrawYScaleLabelCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    double y;
    if ((InputStream >> y) && ((y >= _YMin) || (y <= _YMax))) {
	float cy = CanvasYOf(y);
	float x0 = _FrameOffsetX;

	InputStream >> ws;
	string Label;
	if (getline(InputStream, Label)) {
	    if (strncmp(Label.c_str(), "math:", 5) == 0) {
		_Typesetter->Typeset(
		    x0 - 3, cy,
		    _MathTextComposerList[Font_Label]->Compose(
			Label.substr(5, string::npos)
		    ), 
		    "cr"
		);
	    }
	    else {
		_ImageArea->DrawText(x0 - 3, cy, Label, "cr");
	    }
	}
	else {
	    _Typesetter->Typeset(
		x0 - 3, cy,
		_MathTextComposerList[Font_Label]->ComposeNumber(y),
		"cr"
	    );
	}
	_ImageArea->DrawLine(x0, cy, x0 + 8, cy);

	Result = 1;
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return Result;
}

int TKinokoCanvasPlotObject::ProcessSetXScaleTimeTickCommand(std::istream& InputStream)
{
    long Epoch = 0;  // zero for absolute time
    string Format;   // "%H:%M", "%m %y" etc, empty for auto
    string UnitStr;  // "sec", "day", etc, empty or "-" for auto
    long Step = 0;   // zero for auto

    InputStream >> Epoch >> UnitStr >> Step >> ws;
    getline(InputStream, Format);

    int Unit;
    if ((UnitStr == "sec") || (UnitStr == "second")) {
	Unit = TKinokoCanvasTimeDataScale::TimeUnit_Second;
    }
    else if ((UnitStr == "min") || (UnitStr == "minute")) {
	Unit = TKinokoCanvasTimeDataScale::TimeUnit_Minute;
    }
    else if (UnitStr == "hour") {
	Unit = TKinokoCanvasTimeDataScale::TimeUnit_Hour;
    }
    else if (UnitStr == "day") {
	Unit = TKinokoCanvasTimeDataScale::TimeUnit_Day;
    }
    else if (UnitStr == "month") {
	Unit = TKinokoCanvasTimeDataScale::TimeUnit_Month;
    }
    else if (UnitStr == "year") {
	Unit = TKinokoCanvasTimeDataScale::TimeUnit_Year;
    }
    else {
	Unit = TKinokoCanvasTimeDataScale::TimeUnit_Undefined;
    }

    SetXScaleTime(Epoch, Format, Unit, Step);

    return 1;
}

int TKinokoCanvasPlotObject::ProcessSetXScaleElapseTickCommand(std::istream& InputStream)
{
    SetXScaleElapse();

    return 1;
}

int TKinokoCanvasPlotObject::ProcessSetY2AxisCommand(std::istream& InputStream)
{
    int Result = 0;

    double YMin, YMax;
    string Title;
    if (InputStream >> YMin >> YMax) {
	InputStream >> ws;
	getline(InputStream, Title);

	SetY2Axis(min(YMin, YMax), fabs(YMax - YMin), Title);
	Result = 1;
    }

    return Result;
}

int TKinokoCanvasPlotObject::DrawFrame(void)
{
    TKinokoCanvasFramedObject::DrawFrame();

    if (! _IsFrameValid) {
	return 1;
    }

    _NumberOfXScaleLabels = min(
	12, _FrameWidth / (6 * _LabelCharacterWidth)
    );
    _NumberOfXScaleLabels = max(3, _NumberOfXScaleLabels);

    _NumberOfYScaleLabels = min(
	_FrameHeight / (2 * _LabelCharacterHeight),
	(int) (((float)_FrameHeight/_FrameWidth) * _NumberOfXScaleLabels) + 1
    );
    _NumberOfYScaleLabels = max(3, _NumberOfYScaleLabels);

    int x0 = _FrameOffsetX;
    int y0 = _FrameOffsetY;
    int x1 = _FrameOffsetX + _FrameWidth;
    int y1 = _FrameOffsetY + _FrameHeight;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldBGColorIndex = _ImageArea->SetBackgroundColor(_ColorIndexList[Color_Background]);
    int OldLineWidth = _ImageArea->SetLineWidth(2);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    _ImageArea->DrawRect(x0, y0, x1, y1);
    _ImageArea->SetLineWidth(1);

    // X Title //
    if (strncmp(_XTitle.c_str(), "math:", 5) == 0) {
	_Typesetter->Typeset(
	    x1, (y1 + 1.8 * _LabelCharacterHeight),
	    _MathTextComposerList[Font_Label]->Compose(
		_XTitle.substr(5, string::npos)
	    ), 
	    "tr"
	);
    }
    else {
	_ImageArea->DrawText(
	    x1, y1 + 1.8 * _LabelCharacterHeight, 
	    _XTitle, 
	    "tr"
	);
    }

    // Y Title //
    if (! _IsYTitleVertical) {
	_ImageArea->DrawText(
	    x0, y0 - _LabelCharacterHeight, 
	    _YTitle, 
	    "br"
	);
    }
    else {
	if (strncmp(_YTitle.c_str(), "math:", 5) == 0) {
	    _Typesetter->SetTextOrientation(90);
	    _Typesetter->Typeset(
		_OffsetX + 0.2 * _LabelCharacterHeight, y0,
		_MathTextComposerList[Font_Label]->Compose(
		    _YTitle.substr(5, string::npos)
		), 
		"tl"
	    );
	    _Typesetter->SetTextOrientation(0);
	}
	else {
	    int OldTextOrientation = _ImageArea->SetTextOrientation(90);
	    _ImageArea->DrawText(
		_OffsetX + 0.2 * _LabelCharacterHeight, y0, 
		_YTitle, 
		"tl"
	    );
	    _ImageArea->SetTextOrientation(OldTextOrientation);
	}
    }

    // Y2 Scale Title //
    if (_Y2Scale != 0) {
	if (strncmp(_Y2Title.c_str(), "math:", 5) == 0) {
	    _Typesetter->SetTextOrientation(270);
	    _Typesetter->Typeset(
		_OffsetX + _Width, y0,
		_MathTextComposerList[Font_Label]->Compose(
		    _Y2Title.substr(5, string::npos)
		), 
		"tr"
	    );
	    _Typesetter->SetTextOrientation(0);
	}
	else {
	    int OldTextOrientation = _ImageArea->SetTextOrientation(270);
	    _ImageArea->DrawText(
		_OffsetX + _Width, y0, 
		_Y2Title, 
		"tr"
	    );
	    _ImageArea->SetTextOrientation(OldTextOrientation);
	}
    }

    // X Scale //
    if (! _IsXScaleLabelDisabled) {
	_XScale->Rescale(_XMin, _XMax, _NumberOfXScaleLabels);
	// X Scale Factoring //
	if (_IsScaleFactoringEnabled && _IsYTitleVertical) {
	    string FactorLabel = _XScale->FactorizeScaleLabel();
	    if (! FactorLabel.empty()) {
		_Typesetter->Typeset(
		    x1 + 3, y1, 
		    _MathTextComposerList[Font_Label]->Compose(FactorLabel),
		    "bl"
		);
	    }
	}

	// X Scale Labels //
	int NumberOfXScaleDivisions = _XScale->NumberOfDivisions();
	for (int i = 0; i < NumberOfXScaleDivisions; i++) {
	    float xs = _XScale->DivisionValueOf(i);
	    float x = CanvasXOf(xs);
	    _ImageArea->DrawLine(x, y1 - 8, x, y1);

	    string Label = _XScale->DivisionLabelOf(i);
	    _Typesetter->Typeset(
		x, y1 + 3, 
		_MathTextComposerList[Font_Label]->Compose(Label),
		"tc"
	    );
	}

	// X Scale Ticks //
	int NumberOfXScaleTicks = _XScale->NumberOfTicks();
	for (int i = 0; i < NumberOfXScaleTicks; i++) {
	    float xs = _XScale->TickValueOf(i);
	    float x = CanvasXOf(xs);
	    _ImageArea->DrawLine(x, y1 - 5, x, y1);
	}
    }

    // Y Scale //
    if (! _IsYScaleLabelDisabled) {
	_YScale->Rescale(_YMin, _YMax, _NumberOfYScaleLabels);

	// Y Scale Factoring //
	if (_IsScaleFactoringEnabled) {
	    string FactorLabel = _YScale->FactorizeScaleLabel();
	    if (! FactorLabel.empty()) {
		_Typesetter->Typeset(
		    x0, y0 - 3, 
		    _MathTextComposerList[Font_Label]->Compose(FactorLabel),
		    "bl"
		);
	    }
	}

	// Y Scale Labels //
	int NumberOfYScaleDivisions = _YScale->NumberOfDivisions();
	for (int i = 0; i < NumberOfYScaleDivisions; i++) {
	    float ys = _YScale->DivisionValueOf(i);
	    float y = CanvasYOf(ys);
	    _ImageArea->DrawLine(x0, y, x0 + 8, y);

	    string Label = _YScale->DivisionLabelOf(i);
	    _Typesetter->Typeset(
		x0 - 3, y, 
		_MathTextComposerList[Font_Label]->Compose(Label),
		"cr"
	    );
	}
	
	// Y Scale Ticks //
	int NumberOfYScaleTicks = _YScale->NumberOfTicks();
	for (int i = 0; i < NumberOfYScaleTicks; i++) {
	    float ys = _YScale->TickValueOf(i);
	    float y = CanvasYOf(ys);
	    _ImageArea->DrawLine(x0, y, x0 + 5, y);
	}
    }
	
    //Y2 Scale //
    if (_Y2Scale != 0) {
	_Y2Scale->Rescale(_Y2Min, _Y2Max, _NumberOfYScaleLabels);
#if 1
	// Y2 Scale Factoring //
	string FactorLabel = _Y2Scale->FactorizeScaleLabel();
	if (! FactorLabel.empty()) {
	    _Typesetter->Typeset(
		x1, y0 - 3, 
		_MathTextComposerList[Font_Label]->Compose(FactorLabel),
		"br"
	    );
	}
#endif

	// Y2 Scale Labels //
	int NumberOfYScaleDivisions = _Y2Scale->NumberOfDivisions();
	for (int i = 0; i < NumberOfYScaleDivisions; i++) {
	    float ys = _Y2Scale->DivisionValueOf(i);
	    float y = CanvasY2Of(ys);
	    _ImageArea->DrawLine(x1 - 8, y, x1, y);

	    string Label = _Y2Scale->DivisionLabelOf(i);
	    _Typesetter->Typeset(
		x1 + 3, y, 
		_MathTextComposerList[Font_Label]->Compose(Label),
		"cl"
	    );
	}
	
	// Y2 Scale Ticks //
	int NumberOfYScaleTicks = _Y2Scale->NumberOfTicks();
	for (int i = 0; i < NumberOfYScaleTicks; i++) {
	    float ys = _Y2Scale->TickValueOf(i);
	    float y = CanvasY2Of(ys);
	    _ImageArea->DrawLine(x1 - 5, y, x1, y);
	}
    }
	
    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetBackgroundColor(OldBGColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return 1;
}

void TKinokoCanvasPlotObject::AddReplayCommand(const std::string& Command, const std::string& Parameter)
{
    if (_IsCommandReplaySuspended) {
	return;
    }

    if (_ReplayCommandList.empty()) {
	SaveContext();
    }

    _ReplayCommandList.push_back(make_pair(Command, Parameter));
}

void TKinokoCanvasPlotObject::ClearReplayCommandList(void)
{
    if (_IsCommandReplaySuspended) {
	return;
    }

    if (! _ReplayCommandList.empty()) {
	PopContextStack();
    }

    _ReplayCommandList.erase(
	_ReplayCommandList.begin(), _ReplayCommandList.end()
    );
}

void TKinokoCanvasPlotObject::ReplayCommand(void)
{
    if (! _IsCommandReplayEnabled) {
	return;
    }

    _IsCommandReplaySuspended = true;

    Clear();

    RestoreContext();
    SaveContext();

    for (unsigned i = 0; i < _ReplayCommandList.size(); i++) {
	string Command = _ReplayCommandList[i].first;
	istringstream ParameterStream(_ReplayCommandList[i].second);
	ProcessCommand(Command, ParameterStream);
    }
    _IsCommandReplaySuspended = false;

    _Canvas->Redraw();
}



TKinokoCanvasMapPlotObject::TKinokoCanvasMapPlotObject(TKinokoCanvas* Canvas)
: TKinokoCanvasFramedObject(Canvas), TKinokoCanvasColorScaledObject(Canvas, new TKinokoCanvasGrayBackgroundRainbowColorScale(Canvas))
{
}

TKinokoCanvasMapPlotObject::~TKinokoCanvasMapPlotObject()
{
}

TKinokoShellObject* TKinokoCanvasMapPlotObject::Clone(void)
{
    return new TKinokoCanvasMapPlotObject(_Canvas);
}

void TKinokoCanvasMapPlotObject::Initialize(void)
{
    TKinokoCanvasFramedObject::Initialize();
    TKinokoCanvasColorScaledObject::Initialize();
}

float TKinokoCanvasMapPlotObject::CanvasXOf(float x)
{
    return (x - _XMin) / _XWidth * _FrameWidth + _FrameOffsetX;
}

float TKinokoCanvasMapPlotObject::CanvasYOf(float y)
{
    return (1.0 - (y - _YMin) / _YWidth) * _FrameHeight + _FrameOffsetY;
}

int TKinokoCanvasMapPlotObject::ProcessCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "plot") {
	Result = ProcessPlotCommand(InputStream);
    }
    else {
	Result = TKinokoCanvasFramedObject::ProcessCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoCanvasMapPlotObject::ProcessPlotCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    double Diameter;
    if (! (InputStream >> Diameter)) {
	return 0;
    }
    Diameter *= _FrameWidth / 100.0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);

    double x, y, z;
    float cx, cy;
    while (InputStream >> x >> y >> z) {
	if ((x < _XMin) || (x > _XMax) || (y < _YMin) || (y > _YMax)) {
	    continue;
	}
	cx = CanvasXOf(x);
	cy = CanvasYOf(y);

	z = clip(z, _ZMin, _ZMax);

	_ImageArea->SetColor(
	    _ColorScale->ColorIndexOf((z - _ZMin) / _ZWidth)
	);

	_ImageArea->DrawCircleFill(cx, cy, Diameter/2);
    }

    _ImageArea->SetColor(OldColorIndex);

    return 1;
}

int TKinokoCanvasMapPlotObject::DrawFrame(void)
{
    int Result = TKinokoCanvasColorScaledObject::DrawFrame();
    if (Result == 0) {
	return Result;
    }

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Black]);

    _ImageArea->DrawRectFill(
	_FrameOffsetX, _FrameOffsetY, 
	_FrameOffsetX + _FrameWidth, _FrameOffsetY + _FrameHeight
    );

    _ImageArea->SetColor(OldColorIndex);

    return Result;
}
