/* KinokoCanvasSchematicObject.cc */
/* Created by Enomoto Sanshiro on 24 September. */
/* Last updated by Enomoto Sanshiro on 24 September. */


#include <string>
#include <iostream>
#include <cstdio>
#include <cmath>
#include <algorithm>
#include "KinokoCanvasPictureObject.hh"
#include "KinokoCanvasColorScale.hh"
#include "KinokoCanvasSchematicObject.hh"

using namespace std;


TKinokoCanvasSchematicObject::TKinokoCanvasSchematicObject(TKinokoCanvas* Canvas)
: TKinokoCanvasPictureObject(Canvas)
{
}

TKinokoCanvasSchematicObject::~TKinokoCanvasSchematicObject()
{
    map<string, TKinokoCanvasSchematicScale*>::iterator i;
    for (i = _ScaleTable.begin(); i != _ScaleTable.end(); i++) {
	delete i->second;
    }

    map<int, TKinokoCanvasSchematicItem*>::iterator j;
    for (j = _ItemTable.begin(); j != _ItemTable.end(); j++) {
	delete j->second;
    }

    map<string, TKinokoCanvasSchematicItem*>::iterator k;
    for (
	k = _ItemPrototypeTable.begin(); 
	k != _ItemPrototypeTable.end(); 
	k++
    ){
	delete k->second;
    }
}

TKinokoShellObject* TKinokoCanvasSchematicObject::Clone(void)
{
    return new TKinokoCanvasSchematicObject(_Canvas);
}

void TKinokoCanvasSchematicObject::Initialize(void) 
{
    TKinokoCanvasPictureObject::Initialize();

    _ItemPrototypeTable["text"] = new TKinokoCanvasSchematicTextItem();
    _ItemPrototypeTable["rectangle"] = new TKinokoCanvasSchematicRectangleItem();
    _ItemPrototypeTable["circle"] = new TKinokoCanvasSchematicCircleItem();
}

int TKinokoCanvasSchematicObject::ProcessCommand(const std::string& Command, std::istream& InputStream)
{
    if (Command == "createscale") {
	return ProcessCreateScaleCommand(InputStream);
    }
    else if (Command == "createitem") {
	return ProcessCreateItemCommand(InputStream);
    }
    else if (Command == "setvalue") {
	return ProcessSetValueCommand(InputStream);
    }

    return TKinokoCanvasPictureObject::ProcessCommand(Command, InputStream);
}

int TKinokoCanvasSchematicObject::ProcessCreateScaleCommand(std::istream& InputStream)
{
    double Min, Max;
    string Name, ScaleType;
    if (! (InputStream >> Name >> Min >> Max >> ScaleType)) {
	return 0;
    }

    TKinokoCanvasColorScale* ColorScale = 0;

    if (ScaleType == "rainbow") {
	string Background;
	InputStream >> Background;
	if (Background == "black") {
	    ColorScale = new TKinokoCanvasBlackBackgroundRainbowColorScale(_Canvas);
	}
	else if (Background == "white") {
	    ColorScale = new TKinokoCanvasWhiteBackgroundRainbowColorScale(_Canvas);
	}
	else if (Background == "gray") {
	    ColorScale = new TKinokoCanvasGrayBackgroundRainbowColorScale(_Canvas);
	}
	else {
	    ColorScale = new TKinokoCanvasRainbowColorScale(_Canvas);
	}
    }

    if (ColorScale == 0) {
	return 0;
    }

    ColorScale->Initialize();

    _ScaleTable[Name] = new TKinokoCanvasSchematicScale(
	Min, Max, ColorScale
    );

    return 1;
}

int TKinokoCanvasSchematicObject::ProcessCreateItemCommand(std::istream& InputStream)
{
    int ItemId;
    string ItemTypeName;
    TKinokoCanvasSchematicItem* Item = 0;
    if (InputStream >> ItemId >> ItemTypeName) {
	Item = _ItemPrototypeTable[ItemTypeName];
    }

    int OriginalColor = _ImageArea->SetColor(
	_ColorIndexList[Color_Foreground]
    );
    int OriginalBackgroundColor = _ImageArea->SetBackgroundColor(
	_ColorIndexList[Color_Background]
    );

    if (Item) {
	Item = Item->Clone();
	Item->Attach(this);
	Item->Construct(InputStream);
	_ItemTable[ItemId] = Item;
    }

    _ImageArea->SetColor(OriginalColor);
    _ImageArea->SetBackgroundColor(OriginalBackgroundColor);

    return (Item != 0) ? 1 : 0;
}

int TKinokoCanvasSchematicObject::ProcessSetValueCommand(std::istream& InputStream)
{
    int Result = 0;

    int OriginalColor = _ImageArea->SetColor(-1);
    string OriginalAdjustment = _ImageArea->SetTextAdjustment("tl");

    int ItemId;
    TKinokoCanvasSchematicItem* Item;
    while (InputStream >> ItemId) {
	Item = _ItemTable[ItemId];
	if (! Item) {
	    break;
	}
	if (! Item->SetValue(InputStream)) {
	    break;
	}
	Result++;
    }

    _ImageArea->SetColor(OriginalColor);
    _ImageArea->SetTextAdjustment(OriginalAdjustment);

    return Result;
}

int TKinokoCanvasSchematicObject::ProcessRangeSelect(int X0, int Y0, int X1, int Y1)
{
    int Result = 0;

    typedef map<int, TKinokoCanvasSchematicItem*>::iterator Iterator;
    for (Iterator i = _ItemTable.begin(); i != _ItemTable.end(); i++) {
	if (i->second->Includes(X0, Y0)) {
	    cerr << "Schematic: item #" << i->first << " clicked" << endl;
	    Result = 1;
	    break;
	}
    }

    return Result;
}

TKinokoCanvasImageArea* TKinokoCanvasSchematicObject::GetImageArea(void)
{
    return _ImageArea;
}

TKinokoCanvasSchematicScale* TKinokoCanvasSchematicObject::GetScale(const std::string& Name)
{
    return _ScaleTable[Name];
}



TKinokoCanvasSchematicScale::TKinokoCanvasSchematicScale(double Min, double Max, TKinokoCanvasColorScale* ColorScale)
: TKinokoCanvasColorScale(0, 0)
{
    _Min = min(Min, Max);
    _Max = max(Min, Max);
    _ColorScale = ColorScale;
}

TKinokoCanvasSchematicScale::~TKinokoCanvasSchematicScale()
{
    delete _ColorScale;
}

int TKinokoCanvasSchematicScale::ColorIndexOf(double Value)
{
    double Z = (Value - _Min) / (_Max - _Min);
    return _ColorScale->ColorIndexOf(min(max(Z, 0.0), 1.0));
}

void TKinokoCanvasSchematicScale::Initialize(void)
{
    _ColorScale->Initialize();
}



TKinokoCanvasSchematicItem::TKinokoCanvasSchematicItem(void)
{
    _SchematicObject = 0;
}

TKinokoCanvasSchematicItem::~TKinokoCanvasSchematicItem()
{
}

void TKinokoCanvasSchematicItem::Attach(TKinokoCanvasSchematicObject* SchematicObject)
{
    _SchematicObject = SchematicObject;

    _ImageArea = _SchematicObject->GetImageArea();
}



TKinokoCanvasSchematicTextItem::TKinokoCanvasSchematicTextItem(void)
{
    _Buffer = 0;
    _BufferSize = 0;

    _Color = -1;
}

TKinokoCanvasSchematicTextItem::~TKinokoCanvasSchematicTextItem()
{
    delete[] _Buffer;
}

TKinokoCanvasSchematicItem* TKinokoCanvasSchematicTextItem::Clone(void)
{
    return new TKinokoCanvasSchematicTextItem();
}

int TKinokoCanvasSchematicTextItem::Construct(std::istream& InputStream)
{
    int Result = 0;

    float X0, Y0;
    if (InputStream >> X0 >> Y0) {
	_X0 = CanvasXOf(X0);
	_Y0 = CanvasYOf(Y0);
	Result = 1;
    }

    getline(InputStream >> ws, _Format);
    if (_Format.empty()) {
	_Format = "%.0f";
    }

    delete[] _Buffer;
    _BufferSize = _Format.size() + 32;
    _Buffer = new char[_BufferSize + 1];

    _Color = _ImageArea->SetColor(-1);

    return Result;
}

int TKinokoCanvasSchematicTextItem::SetValue(std::istream& InputStream)
{
    int Result = 0;

    double Value;
    if (InputStream >> Value) {
	snprintf(_Buffer, _BufferSize, _Format.c_str(), Value);
	_ImageArea->SetColor(_Color);
	_ImageArea->DrawText(_X0, _Y0, _Buffer);
	_X1 = _X0 + _ImageArea->TextWidthOf(_Buffer);
	_Y1 = _Y0 + _ImageArea->TextHeightOf(_Buffer);
	Result = 1;
    }

    return Result;
}

bool TKinokoCanvasSchematicTextItem::Includes(int X, int Y)
{
    return ((X >= _X0) && (Y >= _Y0) && (X < _X1) && (Y < _Y1));
}



TKinokoCanvasSchematicRectangleItem::TKinokoCanvasSchematicRectangleItem(void)
{
}

TKinokoCanvasSchematicRectangleItem::~TKinokoCanvasSchematicRectangleItem()
{
}

TKinokoCanvasSchematicItem* TKinokoCanvasSchematicRectangleItem::Clone(void)
{
    return new TKinokoCanvasSchematicRectangleItem();
}

int TKinokoCanvasSchematicRectangleItem::Construct(std::istream& InputStream)
{
    int Result = 0;

    float X0, Y0, X1, Y1;
    string ScaleName;
    if (InputStream >> X0 >> Y0 >> X1 >> Y1 >> ScaleName) {
	_X0 = CanvasXOf(X0);
	_Y0 = CanvasYOf(Y0);
	_X1 = CanvasXOf(X1);
	_Y1 = CanvasYOf(Y1);

	_Scale = _SchematicObject->GetScale(ScaleName);
	Result = (_Scale) ? 1 : 0;
    }

    return Result;
}

int TKinokoCanvasSchematicRectangleItem::SetValue(std::istream& InputStream)
{
    int Result = 0;

    double Value;
    if (InputStream >> Value) {
	_ImageArea->SetColor(_Scale->ColorIndexOf(Value));
	_ImageArea->DrawRectFill(_X0, _Y0, _X1, _Y1);
	Result = 1;
    }

    return Result;
}

bool TKinokoCanvasSchematicRectangleItem::Includes(int X, int Y)
{
    return ((X >= _X0) && (Y >= _Y0) && (X < _X1) && (Y < _Y1));
}



TKinokoCanvasSchematicCircleItem::TKinokoCanvasSchematicCircleItem(void)
{
}

TKinokoCanvasSchematicCircleItem::~TKinokoCanvasSchematicCircleItem()
{
}

TKinokoCanvasSchematicItem* TKinokoCanvasSchematicCircleItem::Clone(void)
{
    return new TKinokoCanvasSchematicCircleItem();
}

int TKinokoCanvasSchematicCircleItem::Construct(std::istream& InputStream)
{
    int Result = 0;

    float X0, Y0, Radius;
    string ScaleName;
    if (InputStream >> X0 >> Y0 >> Radius >> ScaleName) {
	_X0 = CanvasXOf(X0);
	_Y0 = CanvasYOf(Y0);
	_Rx = fabs(CanvasXOf(X0 + Radius) - CanvasXOf(X0));
	_Ry = fabs(CanvasYOf(Y0 + Radius) - CanvasYOf(Y0));

	_Scale = _SchematicObject->GetScale(ScaleName);
	Result = (_Scale) ? 1 : 0;
    }

    return Result;
}

int TKinokoCanvasSchematicCircleItem::SetValue(std::istream& InputStream)
{
    int Result = 0;

    double Value;
    if (InputStream >> Value) {
	_ImageArea->SetColor(_Scale->ColorIndexOf(Value));
	_ImageArea->DrawEllipseFill(_X0, _Y0, _Rx, _Ry);
	Result = 1;
    }

    return Result;
}

bool TKinokoCanvasSchematicCircleItem::Includes(int X, int Y)
{
    return ((fabs(X - _X0) < _Rx) && (fabs(Y - _Y0) < _Ry));
}


#if 0
.create schematic s 0 0 100 100;
s createscale scale0 0 10 rainbow black;
s createitem 1 text 0.2 0.2 hello: %10.2f;
s createitem 2 box 0.5 0.5 0.7 0.7 scale0;
s setvalue 1 3.275;
s setvalue 2 2.789;
#endif
