/* KinokoCanvasTypesetter.cc */
/* Created by Enomoto Sanshiro on 16 November 2003. */
/* Last updated by Enomoto Sanshiro on 7 January 2010. */


#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <map>
#include <algorithm>
#include <cmath>
#include <cctype>
#include "KinokoShellConfig.hh"
#include "KinokoCanvasImageArea.hh"
#include "KinokoCanvasTypesetter.hh"

using namespace std;


struct TSymbolEntry { 
    const char* Name; 
    const char* CharCode;
};

#if USE_PANGO
// UNICODE //
static const TSymbolEntry SymbolList[] = {
    // small greek characters //
    { "alpha", "\u03b1" }, { "beta", "\u03b2" }, { "gamma", "\u03b3" }, 
    { "delta", "\u03b4" }, { "epsilon", "\u03b5" }, { "zeta", "\u03b6" }, 
    { "eta", "\u03b7" }, { "theta", "\u03b8" }, { "iota", "\u03b9" }, 
    { "kappa", "\u03ba" },  { "lambda", "\u03bb" }, { "mu", "\u03bc" }, 
    { "nu", "\u03bd" }, { "xi", "\u03be" }, { "omicron", "\u03bf" },  
    { "pi", "\u03c0" }, { "rho", "\u03c1" }, { "sigma", "\u03c3" },  
    { "tau", "\u03c4" }, { "upsilon", "\u03c5" }, { "phi", "\u03c6" },
    { "chi", "\u03c7" }, { "psi", "\u03c8" },{ "omega", "\u03c9" },  
    { "varphi", "\u03d5" }, { "varpi", "\u03c0" }, 
    
    // chapter greek characters //
    { "Alpha", "\u0391" }, { "Beta", "\u0392" }, { "Gamma", "\u0393" }, 
    { "Delta", "\u0394" }, { "Epsilon", "\u0395" }, { "Zeta", "\u0396" }, 
    { "Eta", "\u0397" }, { "Theta", "\u0398" }, { "Iota", "\u0399" }, 
    { "Kappa", "\u039a" },  { "Lambda", "\u039b" }, { "Mu", "\u039c" }, 
    { "Nu", "\u039d" }, { "Xi", "\u039e" }, { "Omicron", "\u039f" },  
    { "Pi", "\u03a0" }, { "Rho", "\u03a1" }, { "Sigma", "\u03a3" },  
    { "Tau", "\u03a4" }, { "Upsilon", "\u03a5" }, { "Phi", "\u03a6" },
    { "Chi", "\u03a7" }, { "Psi", "\u03a8" },{ "Omega", "\u03a9" },  
    { "Varphi", "\u03a5" }, { "Varpi", "\u03a0" }, 

    // math symbols //
    { "lt", "<" }, { "gt", ">" }, 
    { "leq", "\u2266" }, { "geq", "\u2267" }, 
    { "eq", "=" }, { "neq", "\u2260" }, { "equiv", "\u2261" },
    { "sim", "\u223c" }, { "approx", "\u2248" }, { "propto", "\u221d" },
    { "plus", "+" }, { "minus", "\u2212" }, { "pm", "\u00b1" },
    { "multiply", "\u00d7" }, { "divide", "\u00f7" }, { "cdot", "\u2219" },
    { "circlemultiply", "\u2297" }, { "circleplus", "\u2295" },
    { "leftarrow", "\u2190" }, { "Leftarrow", "\u21d0" }, 
    { "rightarrow", "\u2192" }, { "Rightarrow", "\u21d2" }, 
    { "leftrightarrow", "\u2194" },{ "Leftrightarrow", "\u21d4" },
    { "angle", "\u2220" }, { "degree", "\u00b0" }, { "perp", "\u27c2" }, 
    { "nabla", "\u2207" }, { "laplace", "\u2206" }, { "partial", "\u2202" }, 
    { "florin", "f" }, { "sqrt", "\u221a" }, { "infty", "\u221e" }, 
    { "sum", "\u2211" }, { "prod", "\u220f" }, { "int", "\u222b" },
    { "forall", "\u2200" }, { "exists", "\u2203" },
    { "intersection", "\u22c2" }, { "union", "\u22c3" }, 
    { "element", "\u22f2" }, { "notelement", "\u22f6" }, 
    { "logicalor", "\u22c1" }, { "logicaland", "\u22c0" },

    // aliases //
    { "plusminus", "\u00b1" }, { "times", "\u00d7" }, { "div", "\u00f7" },
    { "otimes", "\u2297" }, { "oplus", "\u2295" },
    { "le", "\u2266" }, { "ge", "\u2267" }, { "ne", "\u2260" },
    { "gradient", "\u2207" }, { "function", "f" }, { "inf", "\245" },
    { "prodcut", "\325" }, { "integral", "\362" },

    { "cap", "\u22c2" }, { "cup", "\u22c3" }, 
    { "in", "\u22f2" }, { "nin", "\u22f6" }, 
    { "vee", "\u22c1" }, { "wedge", "\u22c0" },

    // delimiter //
    { "", "\000" }
};
#else
// Adobe-Standard (Postscript) //
static const TSymbolEntry SymbolList[] = {
    // small greek characters //
    { "alpha", "a" }, { "beta", "b" }, { "chi", "c" }, { "delta", "d" }, 
    { "epsilon", "e" }, { "phi", "f" }, { "gamma", "g" }, { "eta", "h" }, 
    { "iota", "i" }, { "varphi", "j" }, { "kappa", "k" },  { "lambda", "l" }, 
    { "mu", "m" }, { "nu", "n" }, { "omicron", "o" },  { "pi", "p" }, 
    { "theta", "q" }, { "rho", "r" }, { "sigma", "s" },  { "tau", "t" }, 
    { "upsilon", "u" }, { "varpi", "v" }, { "omega", "w" },  { "xi", "x" }, 
    { "psi", "y" }, { "zeta", "z" }, 

    // chapter greek characters //
    { "Alpha", "A" }, { "Beta", "B" }, { "Chi", "C" }, { "Delta", "D" }, 
    { "Epsilon", "E" }, { "Phi", "F" }, { "Gamma", "G" }, { "Eta", "H" }, 
    { "Iota", "I" }, { "Varphi", "J" }, { "Kappa", "K" },  { "Lambda", "L" }, 
    { "Mu", "M" }, { "Nu", "N" }, { "Omicron", "O" },  { "Pi", "P" }, 
    { "Theta", "Q" }, { "Rho", "R" }, { "Sigma", "S" },  { "Tau", "T" }, 
    { "Upsilon", "U" }, { "Varpi", "V" }, { "Omega", "W" },  { "Xi", "X" }, 
    { "Psi", "Y" }, { "Zeta", "Z" }, 

    // math symbols //
    { "lt", "\074" }, { "gt", "\076" }, { "leq", "\243" }, { "geq", "\263" }, 
    { "eq", "\075" }, { "neq", "\271" }, { "equiv", "\272" },
    { "sim", "\176" }, { "approx", "\273" }, { "propto", "\265" },
    { "plus", "\053" }, { "minus", "\055" }, { "pm", "\261" },
    { "multiply", "\264" }, { "divide", "\270" }, { "cdot", "\327" },
    { "cirlcemultiply", "\304" }, { "circleplus", "\305" },
    { "leftarrow", "\254" }, { "Leftarrow", "\334" }, 
    { "rightarrow", "\256" }, { "Rightarrow", "\336" }, 
    { "leftrightarrow", "\253" },{ "Leftrightarrow", "\333" },
    { "angle", "\320" }, { "degree", "\260" }, { "perp", "\136" }, 
    { "nabla", "\321" }, { "partial", "\266" }, 
    { "florin", "\246" }, { "sqrt", "\326" }, { "infty", "\245" }, 
    { "sum", "\345" }, { "prod", "\325" }, { "int", "\362" },
    { "forall", "\042" }, { "exists", "\044" },
    { "intersection", "\307" }, { "union", "\310" }, 
    { "element", "\316" }, { "notelememnt", "\317" }, 
    { "logicalor", "\332" }, { "logicaland", "\331" },

    // aliases //
    { "plusminus", "\261" }, { "times", "\264" }, { "div", "\270" },
    { "otimes", "\304" }, { "oplus", "\305" },
    { "le", "\243" }, { "ge", "\263" }, { "ne", "\271" },
    { "gradient", "\321" }, { "function", "\246" }, { "inf", "\245" },
    { "prodcut", "\325" }, { "integral", "\362" },
    { "cap", "\307" }, { "cup", "\310" }, 
    { "in", "\316" }, { "nin", "\317" }, 
    { "vee", "\332" }, { "wedge", "\331" },

    // delimiter //
    { "", "\000" }
};
#endif

    
static map<string, const char*>* SymbolCodeTable = 0;


TKinokoCanvasMathTextComposer::TKinokoCanvasMathTextComposer(TKinokoCanvasImageArea* ImageArea)
{
    _ImageArea = ImageArea;

    if (SymbolCodeTable == 0) {
	int NumberOfSymbols = sizeof(SymbolList) / sizeof(*SymbolList);
	SymbolCodeTable = new map<string, const char*>;
	for (int i = 0; i < NumberOfSymbols; i++) {
	    (*SymbolCodeTable)[SymbolList[i].Name] = SymbolList[i].CharCode;
	}
    }
}

TKinokoCanvasMathTextComposer::TKinokoCanvasMathTextComposer(const TKinokoCanvasMathTextComposer& Composer)
{
    _ImageArea = Composer._ImageArea;

    for (unsigned i = 0; i < _NumberOfFontSizes; i++) {
	_FontIdList[i] = Composer._FontIdList[i];
	_SymbolFontIdList[i] = Composer._SymbolFontIdList[i];
	_TextHeightList[i] = Composer._TextHeightList[i];
    }
}

TKinokoCanvasMathTextComposer::~TKinokoCanvasMathTextComposer()
{
}

void TKinokoCanvasMathTextComposer::SetFont(const std::string& FontName, int FontSize)
{
    static const float NormalFontScale = 1.0;
    static const float SmallFontScale = 0.7;
    static const float TinyFontScale = 0.5;

    _FontIdList[FontSize_Normal] = _ImageArea->LoadFont(
	FontName, (int) (NormalFontScale * FontSize)
    );
    _FontIdList[FontSize_Small] = _ImageArea->LoadFont(
	FontName, (int) (SmallFontScale * FontSize)
    );
    _FontIdList[FontSize_Tiny] = _ImageArea->LoadFont(
	FontName, (int) (TinyFontScale * FontSize)
    );

    _SymbolFontIdList[FontSize_Normal] = _ImageArea->LoadSymbolFont(
	(int) (NormalFontScale * FontSize)
    );
    _SymbolFontIdList[FontSize_Small] = _ImageArea->LoadSymbolFont(
	(int) (SmallFontScale * FontSize)
    );
    _SymbolFontIdList[FontSize_Tiny] = _ImageArea->LoadSymbolFont(
	(int) (TinyFontScale * FontSize)
    );

    int OldFontId = _ImageArea->SetFont(_FontIdList[FontSize_Normal]);
    _TextHeightList[FontSize_Normal] = _ImageArea->TextHeightOf("0");

    _ImageArea->SetFont(_FontIdList[FontSize_Small]);
    _TextHeightList[FontSize_Small] = _ImageArea->TextHeightOf("0");

    _ImageArea->SetFont(_FontIdList[FontSize_Tiny]);
    _TextHeightList[FontSize_Tiny] = _ImageArea->TextHeightOf("0");

    _ImageArea->SetFont(OldFontId);
}

const TKinokoCanvasTextComposition& TKinokoCanvasMathTextComposer::Compose(const std::string& Text)
{
    _Composition.Initialize();

    _Text = Text.c_str();
    _TextLength = Text.size();

    _CurrentIndex = 0;
    _CurrentString = "";
    _CurrentX = _CurrentY = 0;
    _CurrentSubscriptionLevel = 0;
    _CurrentFontId = _FontIdList[FontSizeOf(_CurrentSubscriptionLevel)];

    Parse(_Composition);

    return _Composition;
}

const TKinokoCanvasTextComposition& TKinokoCanvasMathTextComposer::ComposeNumber(double Value)
{
    if ((fabs(Value) >= 1.0e+6) || (fabs(Value) < 1.0e-5)) {
	return Compose(PowerOf10TextOf(Value));
    }
    else {
	ostringstream os;
	os << Value;
	return Compose(os.str());
    }
}

string TKinokoCanvasMathTextComposer::PowerOf10TextOf(double Value)
{
    if (Value == 0) {
	return "0";
    }

    static const double Eps = 1.0e-6;

    int Power = (int) log10(fabs(Value));
    double Factor = Value / pow(10.0, Power);

    if (fabs(Factor) > 10.0 - Eps) {
	Power++;
	Factor = Value / pow(10.0, Power);
    }
    else if (fabs(Factor) < 1.0 - Eps) {
	Power--;
	Factor = Value / pow(10.0, Power);
    }

    ostringstream os;
    if (fabs(Factor - 1) < Eps) {
	os << "10^{" << Power << "}";
    }
    else if (fabs(Factor + 1) < Eps) {
	os << "-10^{" << Power << "}";
    }
    else {
	os << Factor << "#times10^{" << Power << "}";
    }

    return os.str();
}

int TKinokoCanvasMathTextComposer::FontSizeOf(int SubscriptionLevel)
{
    if (_CurrentSubscriptionLevel == 0) {
	return FontSize_Normal;
    }
    else if (_CurrentSubscriptionLevel == 1) {
	return FontSize_Small;
    }
    else {
	return FontSize_Tiny;
    }
}

void TKinokoCanvasMathTextComposer::Parse(TKinokoCanvasTextComposition& Composition)
{
    while (_CurrentIndex < _TextLength) {
	char Ch = _Text[_CurrentIndex++];

	if (Ch == '{') {
	    Parse(Composition);
	}
	else if (Ch == '}') {
	    break;
	}
	else if (Ch == '^') {
	    ParseSuperscription(Composition);
	}
	else if (Ch == '_') {
	    ParseSubscription(Composition);
	}
	else if (Ch == '#') {
	    ParseSymbol(Composition);
	}
	else {
	    _CurrentString += Ch;
	}
    }

    ComposeThis(Composition);
}

void TKinokoCanvasMathTextComposer::ComposeThis(TKinokoCanvasTextComposition& Composition)
{
    if (_CurrentString.empty()) {
	return;
    }

    int OldFontId = _ImageArea->SetFont(_CurrentFontId);
    float ThisWidth = _ImageArea->TextWidthOf(_CurrentString);
    float ThisHeight = _ImageArea->TextHeightOf(_CurrentString);
    float ThisAscent = _ImageArea->TextAscentOf(_CurrentString);

    Composition.Add(
	_CurrentString, _CurrentFontId, _CurrentX, _CurrentY, 
	ThisWidth, ThisHeight, ThisAscent
    );

    _CurrentX += ThisWidth;
    _CurrentString = "";
    _ImageArea->SetFont(OldFontId);
}

void TKinokoCanvasMathTextComposer::ParseSubscription(TKinokoCanvasTextComposition& Composition)
{
    char Ch = _Text[_CurrentIndex++];
    if (Ch == '\0') {
	return;
    }

    ComposeThis(Composition);

    _CurrentY += 0.5 * _TextHeightList[FontSizeOf(_CurrentSubscriptionLevel)];
    _CurrentSubscriptionLevel++;
    _CurrentFontId = _FontIdList[FontSizeOf(_CurrentSubscriptionLevel)];

    if (Ch == '{') {
	Parse(Composition);
    }
    else {
	_CurrentString += Ch;
	ComposeThis(Composition);
    }

    _CurrentSubscriptionLevel--;
    _CurrentY -= 0.5 * _TextHeightList[FontSizeOf(_CurrentSubscriptionLevel)];
    _CurrentFontId = _FontIdList[FontSizeOf(_CurrentSubscriptionLevel)];
}

void TKinokoCanvasMathTextComposer::ParseSuperscription(TKinokoCanvasTextComposition& Composition)
{
    char Ch = _Text[_CurrentIndex++];
    if (Ch == '\0') {
	return;
    }

    ComposeThis(Composition);

    _CurrentY -= 0.7 * _TextHeightList[FontSizeOf(_CurrentSubscriptionLevel)];
    _CurrentSubscriptionLevel++;
    _CurrentFontId = _FontIdList[FontSizeOf(_CurrentSubscriptionLevel)];

    if (Ch == '{') {
	Parse(Composition);
    }
    else {
	_CurrentString += Ch;
	ComposeThis(Composition);
    }

    _CurrentSubscriptionLevel--;
    _CurrentY += 0.7* _TextHeightList[FontSizeOf(_CurrentSubscriptionLevel)];
    _CurrentFontId = _FontIdList[FontSizeOf(_CurrentSubscriptionLevel)];
}

void TKinokoCanvasMathTextComposer::ParseSymbol(TKinokoCanvasTextComposition& Composition)
{
    char Ch = _Text[_CurrentIndex];
    if (! isalpha(Ch)) {
	_CurrentString += Ch;
	_CurrentIndex++;
	return;
    }

    ComposeThis(Composition);

    string SymbolName;
    while (isalpha(Ch = _Text[_CurrentIndex])) {
	SymbolName += Ch;
	_CurrentIndex++;
    }
    
    const char* SymbolCode = (*SymbolCodeTable)[SymbolName];
    if (SymbolCode != 0) {
	_CurrentString = SymbolCode;
	_CurrentFontId = _SymbolFontIdList[FontSizeOf(_CurrentSubscriptionLevel)];
	ComposeThis(Composition);
	_CurrentFontId = _FontIdList[FontSizeOf(_CurrentSubscriptionLevel)];
    }
    else {
	_CurrentString = SymbolName;
	ComposeThis(Composition);
    }
}



TKinokoCanvasTextComposition::TKinokoCanvasTextComposition(void)
{
}

TKinokoCanvasTextComposition::~TKinokoCanvasTextComposition()
{
}

void TKinokoCanvasTextComposition::Initialize(void)
{
    _StringList.erase(_StringList.begin(), _StringList.end());
    _FontIdList.erase(_FontIdList.begin(), _FontIdList.end());
    _PositionList.erase(_PositionList.begin(), _PositionList.end());
    _WidthList.erase(_WidthList.begin(), _WidthList.end());
    _HeightList.erase(_HeightList.begin(), _HeightList.end());
    _AscentList.erase(_AscentList.begin(), _AscentList.end());
    
    _Width = 0;
    _Ascent = 0;
    _Descent = 0;
}

void TKinokoCanvasTextComposition::Add(const std::string& String, int FontId, float X, float Y, float Width, float Height, float Ascent)
{
    _StringList.push_back(String);
    _FontIdList.push_back(FontId);
    _PositionList.push_back(make_pair(X, Y));
    _WidthList.push_back(Width);
    _HeightList.push_back(Height);
    _AscentList.push_back(Ascent);

    _Width = max(_Width, X + Width);
    _Ascent = max(_Ascent, -Y + Ascent);
    _Descent = max(_Descent, Y + (Height - Ascent));
}



TKinokoCanvasTypesetter::TKinokoCanvasTypesetter(TKinokoCanvasImageArea* ImageArea)
{
    _ImageArea = ImageArea;

    _HorizontalTextAdjustment = TextAdjustment_Left;
    _VerticalTextAdjustment = TextAdjustment_Bottom;
    _TextAdjustment = "bl";

    _TextOrientation = 0;
}

TKinokoCanvasTypesetter::~TKinokoCanvasTypesetter()
{
}

void TKinokoCanvasTypesetter::Typeset(float x, float y, const TKinokoCanvasTextComposition& Composition, const std::string& TextAdjustment)
{
    if (! TextAdjustment.empty()) {
	SetTextAdjustment(TextAdjustment);
    }

    double Width, Height;
    int OldTextOrientation = 0;
    if (_TextOrientation == 0) {
	Width = Composition.Width();
	Height = Composition.Ascent();
    }
    else if (_TextOrientation == 90) {
	Width = Composition.Ascent();
	Height = Composition.Width();
	OldTextOrientation = _ImageArea->SetTextOrientation(_TextOrientation);
    }
    else if (_TextOrientation == 180) {
	Width = Composition.Width();
	Height = Composition.Ascent();
	OldTextOrientation = _ImageArea->SetTextOrientation(_TextOrientation);
    }
    else if (_TextOrientation == 270) {
	Width = Composition.Ascent();
	Height = Composition.Width();
	OldTextOrientation = _ImageArea->SetTextOrientation(_TextOrientation);
    }
    else {
	//...
    }

    if (_VerticalTextAdjustment == TextAdjustment_Center) {
	y += 0.5 * Height;
    }
    else if (_VerticalTextAdjustment == TextAdjustment_Top) {
	y += 1.0 * Height;
    }
    if (_HorizontalTextAdjustment == TextAdjustment_Center) {
	x -= 0.5 * Width;
    }
    else if (_HorizontalTextAdjustment == TextAdjustment_Right) {
	x -= 1.0 * Width;
    }

    int OldFontId = _ImageArea->SetFont(0);
    string OldTextAdjustment = _ImageArea->SetTextAdjustment("bl");

    for (unsigned i = 0; i < Composition.NumberOfElements(); i++) {
	double x0, y0;
	if (_TextOrientation == 0) {
	    x0 = x + Composition.PositionXOf(i);
	    y0 = y + Composition.PositionYOf(i);
	}
	else if (_TextOrientation == 90) {
	    x0 = x + Composition.Ascent() + Composition.PositionYOf(i) - Composition.AscentOf(i);
	    y0 = y - Composition.PositionXOf(i);
	}
	else if (_TextOrientation == 180) {
	    x0 = x + Composition.Width() - Composition.PositionXOf(i) - Composition.WidthOf(i);
	    y0 = y - Composition.Ascent() - Composition.PositionYOf(i) + Composition.AscentOf(i);
	}
	else if (_TextOrientation == 270) {
	    x0 = x - Composition.PositionYOf(i);
	    y0 = y - Composition.Width() + Composition.PositionXOf(i) + Composition.WidthOf(i);
	}
	else {
	    //...
	}

	_ImageArea->SetFont(Composition.FontIdOf(i));
	_ImageArea->DrawText(x0, y0, Composition.StringOf(i));
    }

    _ImageArea->SetFont(OldFontId);
    _ImageArea->SetTextAdjustment(OldTextAdjustment);

    if (_TextOrientation != OldTextOrientation) {
	_ImageArea->SetTextOrientation(OldTextOrientation);
    }
}

string TKinokoCanvasTypesetter::SetTextAdjustment(const string& Adjustment)
{
    string OldTextAdjustment = _TextAdjustment;
    _TextAdjustment = Adjustment;

    _HorizontalTextAdjustment = TextAdjustment_Center;
    _VerticalTextAdjustment = TextAdjustment_Center;

    if (Adjustment.find_first_of('t') != string::npos) {
	_VerticalTextAdjustment = TextAdjustment_Top;
    }
    if (Adjustment.find_first_of('b') != string::npos) {
	_VerticalTextAdjustment = TextAdjustment_Bottom;
    }
    if (Adjustment.find_first_of('l') != string::npos) {
	_HorizontalTextAdjustment = TextAdjustment_Left;
    }
    if (Adjustment.find_first_of('r') != string::npos) {
	_HorizontalTextAdjustment = TextAdjustment_Right;
    }

    return OldTextAdjustment;
}

int TKinokoCanvasTypesetter::SetTextOrientation(int Orientation)
{
    int OldOrientation = _TextOrientation;
    _TextOrientation = Orientation;

    return OldOrientation;
}
