/* KinokoControlScript.cc */
/* Created by Enomoto Sanshiro on 25 October 2002. */
/* Last updated by Enomoto Sanshiro on 12 August 2007. */


#include <string>
#include <vector>
#include <map>
#include <climits>
#include "MushMisc.hh"
#include "ParaParser.hh"
#include "ParaExpression.hh"
#include "KinokoControl.hh"
#include "KinokoControlScript.hh"
#include "KinokoControlViewletMessenger.hh"

using namespace std;


TKinokoControlScript::TKinokoControlScript(TKinokoControl* Control)
{
    _Control = Control;
}

TKinokoControlScript::~TKinokoControlScript()
{
    list<pair<long, TKinokoControlDelayedTask*> >::iterator Task;
    for (
	Task = _DelayedTaskList.begin(); 
	Task != _DelayedTaskList.end();
	Task++
    ){
	Task->second->Destroy();
	delete Task->second;
    }
}

TParaTokenTable* TKinokoControlScript::CreateTokenTable(void)
{
    TParaTokenTable* TokenTable = TParaStandardParser::CreateTokenTable();

    TokenTable->AddKeyword("exit");

    return TokenTable;
}

TParaOperatorTable* TKinokoControlScript::CreateOperatorTable(void)
{
    TParaOperatorTable* OperatorTable;
    OperatorTable = TParaStandardParser::CreateOperatorTable();

    OperatorTable->AddElementaryOperator(
	new TKinokoControlWidgetAccessOperator(_Control)
    );

    return OperatorTable;
}

TParaStatementTable* TKinokoControlScript::CreateStatementTable(void)
{
    TParaStatementTable* StatementTable;
    StatementTable = TParaStandardParser::CreateStatementTable();

    StatementTable->AddStatement(
	new TKinokoControlInvokeStatement(_Control)
    );
    StatementTable->AddStatement(
	new TKinokoControlExitStatement(_Control)
    );
    StatementTable->AddStatement(
	new TKinokoControlAfterStatement(this, _Control)
    );

    return StatementTable;
}

TParaPackage* TKinokoControlScript::CreatePackage(void)
{
    TParaPackage* Package = TParaStandardParser::CreatePackage();
    Package->AddEntry(new TKinokoControlTaskEntry(this));

    return Package;
}

TParaObjectPrototypeTable* TKinokoControlScript::CreateObjectPrototypeTable(void)
{
    TParaObjectPrototypeTable* ObjectPrototypeTable;
    ObjectPrototypeTable = TParaStandardParser::CreateObjectPrototypeTable();

    return ObjectPrototypeTable;
}

TParaBuiltinFunctionTable* TKinokoControlScript::CreateBuiltinFunctionTable(void)
{
    TParaBuiltinFunctionTable* BuiltinFunctionTable;
    BuiltinFunctionTable = TParaStandardParser::CreateBuiltinFunctionTable();

    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKinokoControlPanelMessenger(_Control)
    );

    return BuiltinFunctionTable;
}

void TKinokoControlScript::AddEventTask(const string& Name, TParaPackageEntry* Task)
{
    _EventTaskTable.insert(make_pair(Name, Task));
}

void TKinokoControlScript::AddTransitionTask(const string& State, const string& NewState, TParaPackageEntry* Task)
{
    string Key = "[" + State + "][" + NewState + "]";
    _TransitionTaskTable.insert(make_pair(Key, Task));
}

void TKinokoControlScript::AddInvocationTask(const string& Name, TParaPackageEntry* Task)
{
    _InvocationTaskTable.insert(make_pair(Name, Task));
}

void TKinokoControlScript::AddPeriodicTask(long Interval, TParaPackageEntry* Task)
{
    _PeriodicTaskTable.push_back(make_pair(Interval, Task));
    _NextExecutionTimeTable.push_back(0);
}

void TKinokoControlScript::AddDelayedTask(long ExecutionTime, TKinokoControlDelayedTask* Task)
{
    _DelayedTaskList.push_back(make_pair(ExecutionTime, Task));
}

int TKinokoControlScript::ExecuteEventTask(const string& Name, vector<TParaValue*> ArgumentList) throw(TScriptException)
{
    int Result = 0;

    typedef multimap<string, TParaPackageEntry*>::iterator TTableIterator;
    typedef pair<TTableIterator, TTableIterator> TTableRange;

    for (
	TTableRange Range = _EventTaskTable.equal_range(Name); 
	Range.first != Range.second; 
	Range.first++
    ){
	TParaPackageEntry* Task = Range.first->second;
	Task->Execute(ArgumentList, _SymbolTable);
	Result++;
    }

    return Result;
}

int TKinokoControlScript::ExecuteTransitionTask(const string& CurrentState, const string& NextState) throw(TScriptException)
{
    int Result = 0;
    vector<TParaValue*> ArgumentList;

    string KeyList[4];
    KeyList[0] = "[" + CurrentState + "][" + NextState + "]";
    KeyList[1] = "[" + CurrentState + "][]";
    KeyList[2] = "[][" + NextState + "]";
    KeyList[3] = "[][]";

    typedef multimap<string, TParaPackageEntry*>::iterator TTableIterator;
    typedef pair<TTableIterator, TTableIterator> TTableRange;

    for (unsigned i = 0; i < sizeof(KeyList)/sizeof(KeyList[0]); i++) {
	for (
	    TTableRange Range = _TransitionTaskTable.equal_range(KeyList[i]);
	    Range.first != Range.second; 
	    Range.first++
	){
	    TParaPackageEntry* Task = Range.first->second;
	    Task->Execute(ArgumentList, _SymbolTable);
	    Result++;
	}
    }

    return Result;
}

int TKinokoControlScript::ExecuteInvocationTask(const string& Name) throw(TScriptException)
{
    int Result = 0;
    vector<TParaValue*> ArgumentList;

    string KeyList[2];
    KeyList[0] = Name;
    KeyList[1] = "";

    typedef multimap<string, TParaPackageEntry*>::iterator TTableIterator;
    typedef pair<TTableIterator, TTableIterator> TTableRange;

    for (unsigned i = 0; i < sizeof(KeyList)/sizeof(KeyList[0]); i++) {
	for (
	    TTableRange Range = _InvocationTaskTable.equal_range(KeyList[i]);
	    Range.first != Range.second; 
	    Range.first++
	){
	    TParaPackageEntry* Task = Range.first->second;
	    Task->Execute(ArgumentList, _SymbolTable);
	    Result++;
	}
    }

    return Result;
}

int TKinokoControlScript::ExecuteScheduledTask(long PresentTime) throw(TScriptException)
{
    int Result = 0;

    for (unsigned i = 0; i < _PeriodicTaskTable.size(); i++) {
	long NextTime = _NextExecutionTimeTable[i];
	if (NextTime == 0) {
	    NextTime = PresentTime;
	}

	if (NextTime <= PresentTime) {
	    TParaPackageEntry* Task = _PeriodicTaskTable[i].second;
	    vector<TParaValue*> ArgumentList;
	    try {
		Task->Execute(ArgumentList, _SymbolTable);
	    }
	    catch (TScriptException &e) {
		_NextExecutionTimeTable[i] = LONG_MAX;
		throw;
	    }

	    long Interval = _PeriodicTaskTable[i].first;
	    NextTime += Interval * ((PresentTime - NextTime) / Interval + 1);
	    _NextExecutionTimeTable[i] = NextTime;

	    Result++;
	}
    }

    list<pair<long, TKinokoControlDelayedTask*> >::iterator Task, ThisTask;
    Task = _DelayedTaskList.begin(); 
    while (Task != _DelayedTaskList.end()) {
	ThisTask = Task;
	Task++;  // Be careful not to mess up the iterator //
	if (ThisTask->first > PresentTime) {
	    continue;
	}

	try {
	    ThisTask->second->Execute(_SymbolTable);
	    Result++;
	}
	catch (TScriptException &e) {
	    ThisTask->second->Destroy();
	    delete ThisTask->second;
	    _DelayedTaskList.erase(ThisTask);
	    throw;
	}
	ThisTask->second->Destroy();
	delete ThisTask->second;
	_DelayedTaskList.erase(ThisTask);
    }

    return Result;
}



TKinokoControlPanelMessenger::TKinokoControlPanelMessenger(TKinokoControl* Control)
: TParaObjectPrototype("ControlPanel")
{
    _Control = Control;
}

TKinokoControlPanelMessenger::~TKinokoControlPanelMessenger()
{
    map<TKinokoControlWidget*, TKinokoControlWidgetMessenger*>::iterator i;
    for (
	i = _ControlWidgetMessengerPool.begin();
	i != _ControlWidgetMessengerPool.end();
	i++
    ){
	i->second->Destruct();
	delete i->second;
    }
}

TParaObjectPrototype* TKinokoControlPanelMessenger::Clone(void)
{
    return new TKinokoControlPanelMessenger(_Control);
}

void TKinokoControlPanelMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
}

void TKinokoControlPanelMessenger::Destruct(void) throw(TScriptException)
{
}

int TKinokoControlPanelMessenger::MethodIdOf(const string& MethodName)
{
    if (MethodName == "lookupWidget") {
        return MethodId_LookupWidget;
    }
    else if (MethodName == "currentState") {
        return MethodId_CurrentState;
    }
    else if (MethodName == "clearWidgetValues") {
        return MethodId_ClearWidgetValues;
    }
    else if (MethodName == "loadWidgetValues") {
        return MethodId_LoadWidgetValues;
    }
    else if (MethodName == "saveWidgetValues") {
        return MethodId_SaveWidgetValues;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKinokoControlPanelMessenger::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_LookupWidget:
        Result = LookupWidget(ArgumentList, ReturnValue);
	break;
      case MethodId_CurrentState:
        Result = CurrentState(ArgumentList, ReturnValue);
	break;
      case MethodId_ClearWidgetValues:
        Result = ClearWidgetValues(ArgumentList, ReturnValue);
	break;
      case MethodId_LoadWidgetValues:
        Result = LoadWidgetValues(ArgumentList, ReturnValue);
	break;
      case MethodId_SaveWidgetValues:
        Result = SaveWidgetValues(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TKinokoControlPanelMessenger::LookupWidget(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsString())) {
	throw TScriptException(
	    "ControlPanel::lookupWidget(string widget_name)", 
	    "invalid argument[s]"
	);
    }

    string WidgetName = ArgumentList[0]->AsString();
    TKinokoControlWidget* ControlWidget = _Control->LookupWidget(WidgetName);
    
    if (ControlWidget == 0) {
	throw TScriptException(
	    "ControlPanel::lookupWidget(string widget_name)", 
	    "unable to find widget: " + WidgetName
	);
    }

    TKinokoControlWidgetMessenger* ControlWidgetMessenger = (
	_ControlWidgetMessengerPool[ControlWidget]
    );
    if (ControlWidgetMessenger == 0) {
	while (1) {
	    TKinokoControlCanvasViewlet* CanvasWidget = (
		dynamic_cast<TKinokoControlCanvasViewlet*>(ControlWidget)
	    );
	    if (CanvasWidget) {
		ControlWidgetMessenger = (
		    new TKinokoControlCanvasViewletMessenger(CanvasWidget)
		);
		break;
	    }

	    TKinokoControlPlotViewlet* PlotWidget = (
		dynamic_cast<TKinokoControlPlotViewlet*>(ControlWidget)
	    );
	    if (PlotWidget) {
		ControlWidgetMessenger = (
		    new TKinokoControlPlotViewletMessenger(PlotWidget)
		);
		break;
	    }

	    ControlWidgetMessenger = (
		new TKinokoControlWidgetMessenger(ControlWidget)
	    );
	    break;
	}

	_ControlWidgetMessengerPool[ControlWidget] = ControlWidgetMessenger;
	ControlWidgetMessenger->_ReferenceCount++;  // not to be deleted
    }

    ReturnValue = TParaValue(ControlWidgetMessenger);

    return 1;
}

int TKinokoControlPanelMessenger::ClearWidgetValues(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _Control->ClearWidgetValues();

    return 1;
}

int TKinokoControlPanelMessenger::CurrentState(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue(_Control->CurrentState());

    return 1;
}

int TKinokoControlPanelMessenger::LoadWidgetValues(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsString())) {
	throw TScriptException(
	    "ControlPanel::loadWidgetValues(string file_name)", 
	    "invalid argument[s]"
	);
    }

    string FileName = ArgumentList[0]->AsString();
    _Control->LoadWidgetValues(FileName);

    return 1;
}

int TKinokoControlPanelMessenger::SaveWidgetValues(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((ArgumentList.size() < 1) || (! ArgumentList[0]->IsString())) {
	throw TScriptException(
	    "ControlPanel::saveWidgetValues(string file_name)", 
	    "invalid argument[s]"
	);
    }

    string FileName = ArgumentList[0]->AsString();
    _Control->SaveWidgetValues(FileName);

    return 1;
}



TKinokoControlWidgetMessenger::TKinokoControlWidgetMessenger(TKinokoControlWidget* ControlWidget)
: TParaObjectPrototype("ControlWidget")
{
    _ControlWidget = ControlWidget;
}

TKinokoControlWidgetMessenger::~TKinokoControlWidgetMessenger()
{
}

TParaObjectPrototype* TKinokoControlWidgetMessenger::Clone(void)
{
    // object duplication is prohibited.
    return 0;
}

void TKinokoControlWidgetMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
}

void TKinokoControlWidgetMessenger::Destruct(void) throw(TScriptException)
{
}

int TKinokoControlWidgetMessenger::MethodIdOf(const string& MethodName)
{
    if (MethodName == "setValue") {
        return MethodId_SetValue;
    }
    else if (MethodName == "getValue") {
        return MethodId_GetValue;
    }
    else if (MethodName == "setAttribute") {
        return MethodId_SetAttribute;
    }
    else if (MethodName == "getAttribute") {
        return MethodId_GetAttribute;
    }
    else if (MethodName == "enable") {
        return MethodId_Enable;
    }
    else if (MethodName == "disable") {
        return MethodId_Disable;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKinokoControlWidgetMessenger::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_SetValue:
        Result = SetValue(ArgumentList, ReturnValue);
	break;
      case MethodId_GetValue:
        Result = GetValue(ArgumentList, ReturnValue);
	break;
      case MethodId_SetAttribute:
        Result = SetAttribute(ArgumentList, ReturnValue);
	break;
      case MethodId_GetAttribute:
        Result = GetAttribute(ArgumentList, ReturnValue);
	break;
      case MethodId_Enable:
        Result = Enable(ArgumentList, ReturnValue);
	break;
      case MethodId_Disable:
        Result = Disable(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TKinokoControlWidgetMessenger::SetValue(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "ControlWidget::setValue(string value)", "too few argument[s]"
	);
    }

    string Value = ArgumentList[0]->AsString();

    for (unsigned i = 1; i < ArgumentList.size(); i++) {
	Value += " " + ArgumentList[i]->AsString();
    }

    _ControlWidget->SetValue(Value);

    return 1;
}

int TKinokoControlWidgetMessenger::GetValue(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue(_ControlWidget->Value());

    return 1;
}

int TKinokoControlWidgetMessenger::SetAttribute(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "ControlWidget::setAttribute(string name, string value)", 
	    "too few argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();
    bool Result = _ControlWidget->SetAttribute(Name, Value);

    ReturnValue = TParaValue(Result);

    return 1;
}

int TKinokoControlWidgetMessenger::GetAttribute(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "ControlWidget::setAttribute(string name)", 
	    "too few argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value;
    bool Result = _ControlWidget->GetAttribute(Name, Value);

    if (Result) {
	ReturnValue = TParaValue(Value);
    }
    else {
	ReturnValue = TParaValue();
    }

    return 1;
}

int TKinokoControlWidgetMessenger::Enable(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _ControlWidget->Enable();

    return 1;
}

int TKinokoControlWidgetMessenger::Disable(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _ControlWidget->Disable();

    return 1;
}



TKinokoControlWidgetAccessOperator::TKinokoControlWidgetAccessOperator(TKinokoControl* Control)
{
    _Control = Control;
    _ControlWidgetMessenger = 0;
}

TKinokoControlWidgetAccessOperator::~TKinokoControlWidgetAccessOperator()
{
    if (_ControlWidgetMessenger) {
	_ControlWidgetMessenger->Destruct();
	delete _ControlWidgetMessenger;
    }
}

TParaOperator* TKinokoControlWidgetAccessOperator::Clone(void) const
{
    return new TKinokoControlWidgetAccessOperator(_Control);
}

string TKinokoControlWidgetAccessOperator::Symbol(void) const
{
    return string("<");
}

string TKinokoControlWidgetAccessOperator::Name(void) const
{
    return string("WidgetAccess");
}

void TKinokoControlWidgetAccessOperator::Parse(TParaTokenizer* Tokenizer, TParaExpressionParser* ExpressionParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("<");
    _WidgetName = Tokenizer->Next().RemoveQuotation('"').AsString();
    Tokenizer->Next().MustBe(">");
}

TParaValue& TKinokoControlWidgetAccessOperator::Evaluate(TParaValue& Left, TParaValue& Right, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    if (_ControlWidgetMessenger == 0) {
	TKinokoControlWidget* ControlWidget = _Control->LookupWidget(_WidgetName);
	if (ControlWidget == 0) {
	    throw TScriptException(
		"ControlPanel::lookupWidget(string widget_name)", 
		"unable to find widget: " + _WidgetName
	    );
	}

	while (1) {
	    TKinokoControlCanvasViewlet* CanvasWidget = (
		dynamic_cast<TKinokoControlCanvasViewlet*>(ControlWidget)
	    );
	    if (CanvasWidget) {
		_ControlWidgetMessenger = (
		    new TKinokoControlCanvasViewletMessenger(CanvasWidget)
		);
		break;
	    }

	    TKinokoControlPlotViewlet* PlotWidget = (
		dynamic_cast<TKinokoControlPlotViewlet*>(ControlWidget)
	    );
	    if (PlotWidget) {
		_ControlWidgetMessenger = (
		    new TKinokoControlPlotViewletMessenger(PlotWidget)
		);
		break;
	    }

	    _ControlWidgetMessenger = (
		new TKinokoControlWidgetMessenger(ControlWidget)
	    );
	    break;
	}
    }

    _ControlWidgetMessenger->_ReferenceCount++;  // not to be deleted

    Result = TParaValue(_ControlWidgetMessenger);

    return Result;
}



TKinokoControlInvokeStatement::TKinokoControlInvokeStatement(TKinokoControl* Control)
{
    _Control = Control;
}

TKinokoControlInvokeStatement::~TKinokoControlInvokeStatement()
{
    for (unsigned i = 0; i < _ArgumentExpressionList.size(); i++) {
	delete _ArgumentExpressionList[i];
    }
}

TParaStatement* TKinokoControlInvokeStatement::Clone(void)
{
    return new TKinokoControlInvokeStatement(_Control);
}

string TKinokoControlInvokeStatement::FirstToken(void) const
{
    return "invoke";
}

void TKinokoControlInvokeStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe("invoke");

    _MethodName = Tokenizer->Next().AsString();
    if (Tokenizer->LookAhead().Is("(")) {
	_ArgumentExpressionList = ExpressionParser->ParseExpressionList(
	    Tokenizer, SymbolTable, "(", ")", ","
	);
    }

    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKinokoControlInvokeStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    vector<string> ArgumentList;
    for (unsigned i = 0; i < _ArgumentExpressionList.size(); i++) {
	ArgumentList.push_back(
	    _ArgumentExpressionList[i]->Evaluate(SymbolTable).AsString()
	);
    }

    _Control->Invoke(_MethodName, ArgumentList);

    return TExecResult();
}



TKinokoControlExitStatement::TKinokoControlExitStatement(TKinokoControl* Control)
{
    _Control = Control;
}

TKinokoControlExitStatement::~TKinokoControlExitStatement()
{
}

TParaStatement* TKinokoControlExitStatement::Clone(void)
{
    return new TKinokoControlExitStatement(_Control);
}

string TKinokoControlExitStatement::FirstToken(void) const
{
    return "exit";
}

void TKinokoControlExitStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("exit");
    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKinokoControlExitStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _Control->TryToQuit();

    TExecResult Result;
    Result.ExecStatus = TParaStatement::esReturn;

    return Result;
}



TKinokoControlTaskEntry::TKinokoControlTaskEntry(TKinokoControlScript* Script)
: TParaPackageEntry("task")
{
    _Script = Script;
    _Statement = 0;
}

TKinokoControlTaskEntry::~TKinokoControlTaskEntry()
{
    delete _Statement;
}

TParaPackageEntry* TKinokoControlTaskEntry::Clone(void)
{
    return new TKinokoControlTaskEntry(_Script);
}

bool TKinokoControlTaskEntry::HasEntryWordsOf(TParaTokenizer* Tokenizer)
{
    return Tokenizer->LookAhead().Is("on");
}

void TKinokoControlTaskEntry::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("on");
    if (Tokenizer->LookAhead().Is("every")) {
	Tokenizer->Next().MustBe("every");
	Tokenizer->Next().MustBe("(");
	long Interval = Tokenizer->Next().AsLong();

	TParaToken Token = Tokenizer->Next();
	if (Token.Is("sec")) {
	    Interval *= 1;
	}
	else if (Token.Is("min")) {
	    Interval *= 60;
	}
	else if (Token.Is("hour")) {
	    Interval *= 60 * 60;
	}
	else if (Token.Is("day")) {
	    Interval *= 60 * 60 * 24;
	}
	else {
	    Token.ThrowUnexpected("sec, min, hour or day");
	}

	Tokenizer->Next().MustBe(")");
	
	_Script->AddPeriodicTask(Interval, this);
    }
    else if (Tokenizer->LookAhead().Is("transition")) {
	string State, NewState;
	Tokenizer->Next().MustBe("transition");
	Tokenizer->Next().MustBe("(");
	if (Tokenizer->LookAhead().Is("from")) {
	    Tokenizer->Next().MustBe("from");
	    State = Tokenizer->Next().RemoveQuotation('"').AsString();
	}
	if (Tokenizer->LookAhead().Is("to")) {
	    Tokenizer->Next().MustBe("to");
	    NewState = Tokenizer->Next().RemoveQuotation('"').AsString();
	}
	Tokenizer->Next().MustBe(")");

	_Script->AddTransitionTask(State, NewState, this);
    }
    else if (Tokenizer->LookAhead().Is("invocation")) {
	string Name;
	Tokenizer->Next().MustBe("invocation");
	Tokenizer->Next().MustBe("(");
	if (Tokenizer->LookAhead().IsNot(")")) {
	    Name = Tokenizer->Next().RemoveQuotation('"').AsString();
	}
	Tokenizer->Next().MustBe(")");

	_Script->AddInvocationTask(Name, this);
    }
    else {
	string Name = Tokenizer->Next().AsString();
	Tokenizer->Next().MustBe("(");
	Tokenizer->Next().MustBe(")");

	_Script->AddEventTask(Name, this);
    }

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaValue TKinokoControlTaskEntry::Execute(const vector<TParaValue*>& ArgumentList, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _Statement->Execute(SymbolTable).ReturnValue;
}



TKinokoControlAfterStatement::TKinokoControlAfterStatement(TKinokoControlScript* Script, TKinokoControl* Control)
{
    _Script = Script;
    _Control = Control;

    _TimeToExecutionExpression = 0;
    _DelayedTask = 0;

    _TimeUnit = 1;
}

TKinokoControlAfterStatement::~TKinokoControlAfterStatement()
{
    delete _DelayedTask;
    delete _TimeToExecutionExpression;

    // note: _ArgumentExpressionList will be deleted in TParaFunctionCallNode
}

TParaStatement* TKinokoControlAfterStatement::Clone(void)
{
    return new TKinokoControlAfterStatement(_Script, _Control);
}

string TKinokoControlAfterStatement::FirstToken(void) const
{
    return string("after");
}

void TKinokoControlAfterStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe("after");
    Tokenizer->Next().MustBe("(");
    
    _TimeToExecutionExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);
    TParaToken Token = Tokenizer->Next();
    if (Token.Is("sec")) {
	_TimeUnit = 1;
    }
    else if (Token.Is("min")) {
	_TimeUnit = 60;
    }
    else if (Token.Is("hour")) {
	_TimeUnit = 60 * 60;
    }
    else if (Token.Is("day")) {
	_TimeUnit = 60 * 60 * 24;
    }
    else {
	Token.ThrowUnexpected("sec, min, hour or day");
    }

    Tokenizer->Next().MustBe(")");
    
    if (Tokenizer->LookAhead().Is("call")) {
	Tokenizer->Next().MustBe("call");
	string FunctionName = Tokenizer->Next().AsString();
	long FunctionId = SymbolTable->NameToId(FunctionName);

	_ArgumentExpressionList = ExpressionParser->ParseExpressionList(
	    Tokenizer, SymbolTable, "(", ")", ","
	);

	_DelayedTask = new TKinokoControlDelayedFunctionCall(
	    FunctionId, FunctionName, _ArgumentExpressionList
	);
	_DelayedTask->SetLineNumber(Tokenizer->LineNumber());
    }
    else if (Tokenizer->LookAhead().Is("invoke")) {
	Tokenizer->Next().MustBe("invoke");
	string MethodName = Tokenizer->Next().AsString();
	long FunctionId = SymbolTable->NameToId(MethodName);

	_ArgumentExpressionList = ExpressionParser->ParseExpressionList(
	    Tokenizer, SymbolTable, "(", ")", ","
	);

	_DelayedTask = new TKinokoControlDelayedMethodInvocation(
	    FunctionId, MethodName, _ArgumentExpressionList, _Control
	);
	_DelayedTask->SetLineNumber(Tokenizer->LineNumber());
    }
    else {
	Tokenizer->Next().ThrowUnexpected();
    }

    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKinokoControlAfterStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long TimeToExecution = _TimeToExecutionExpression->Evaluate(SymbolTable).AsLong();    
    
    _DelayedTask->EvaluateArguments(SymbolTable);

    _Script->AddDelayedTask(
	TMushDateTime::SecSinceEpoch() + TimeToExecution * _TimeUnit,
	_DelayedTask->Duplicate()
    );

    return TExecResult();
}



TKinokoControlDelayedTask::TKinokoControlDelayedTask(long FunctionId, std::vector<TParaExpression*>& ArgumentExpressionList)
: TParaFunctionCallNode(FunctionId, ArgumentExpressionList)
{
}

TKinokoControlDelayedTask::~TKinokoControlDelayedTask(void)
{
}

void TKinokoControlDelayedTask::Destroy(void)
{
    for (unsigned i = 0; i < _ArgumentList.size(); i++) {
	delete _ArgumentList[i];
    }
}

void TKinokoControlDelayedTask::DuplicateArguments(vector<TParaValue*>& ArgumentList)
{
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	_ArgumentList.push_back(new TParaValue(*ArgumentList[i]));
    }
}



TKinokoControlDelayedFunctionCall::TKinokoControlDelayedFunctionCall(long FunctionId, const string& FunctionName, vector<TParaExpression*>& ArgumentExpressionList)
: TKinokoControlDelayedTask(FunctionId, ArgumentExpressionList)
{
    _FunctionName = FunctionName;
}

TKinokoControlDelayedFunctionCall::~TKinokoControlDelayedFunctionCall(void)
{
}

TKinokoControlDelayedTask* TKinokoControlDelayedFunctionCall::Duplicate(void)
{
    TKinokoControlDelayedTask* Task;
    vector<TParaExpression*> ArgumentExpressionList;
    
    Task = new TKinokoControlDelayedFunctionCall(
	_FunctionId, _FunctionName, ArgumentExpressionList
    );

    Task->DuplicateArguments(_ArgumentList);

    return Task;
}

void TKinokoControlDelayedFunctionCall::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaFunctionCallNode::EvaluateFunction(SymbolTable);
}

void TKinokoControlDelayedFunctionCall::DumpThis(std::ostream &os) const
{
    os << "call{" << _FunctionName << "()}";
}



TKinokoControlDelayedMethodInvocation::TKinokoControlDelayedMethodInvocation(long FunctionId, const string& MethodName, vector<TParaExpression*>& ArgumentExpressionList, TKinokoControl* Control)
: TKinokoControlDelayedTask(FunctionId, ArgumentExpressionList)
{
    _MethodName = MethodName;
    _Control = Control;
    _ArgumentExpressionList = ArgumentExpressionList;
}

TKinokoControlDelayedMethodInvocation::~TKinokoControlDelayedMethodInvocation(void)
{
}

TKinokoControlDelayedTask* TKinokoControlDelayedMethodInvocation::Duplicate(void)
{
    TKinokoControlDelayedTask* Task;

    Task = new TKinokoControlDelayedMethodInvocation(
	_FunctionId, _MethodName, _ArgumentExpressionList, _Control
    );

    Task->DuplicateArguments(_ArgumentList);

    return Task;
}

void TKinokoControlDelayedMethodInvocation::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    vector<string> ArgumentList;
    for (unsigned i = 0; i < _ArgumentExpressionList.size(); i++) {
	ArgumentList.push_back(
	    _ArgumentExpressionList[i]->Evaluate(SymbolTable).AsString()
	);
    }

    _Control->Invoke(_MethodName, ArgumentList);
}

void TKinokoControlDelayedMethodInvocation::DumpThis(std::ostream &os) const
{
    os << "invoke{" << _MethodName << "()}";
}
