/* KinokoControlViewlet.cc */
/* Created by Enomoto Sanshiro on 9 August 2007. */
/* Last updated by Enomoto Sanshiro on 11 August 2007. */


#include <iostream>
#include <sstream>
#include <string>
#include <algorithm>
#include "KinokoCanvas.hh"
#include "KinokoControlWidget.hh"
#include "KinokoControlViewlet.hh"

using namespace std;
using namespace kuma;


TKinokoControlViewletCanvas::TKinokoControlViewletCanvas(TKinokoControlImagePort* ImagePort)
: TKinokoCanvas(0)
{
    _ImagePort = ImagePort;
}

TKinokoControlViewletCanvas::~TKinokoControlViewletCanvas()
{
}

void TKinokoControlViewletCanvas::Start(void)
{
    AddPage(_ImagePort->GetImageArea());
    SelectPage(0);
}

void TKinokoControlViewletCanvas::Quit(void)
{
}

TKinokoShellPopupWindow* TKinokoControlViewletCanvas::CreatePopupWindow(void)
{
    return 0;
}

TKinokoShellFileSelectDialog* TKinokoControlViewletCanvas::CreateFileSelectDialog(void)
{
    return 0;
}

int TKinokoControlViewletCanvas::OpenContextMenu(TKinokoCanvasObject* CanvasObject)
{
    return 0;
}



TKinokoControlViewlet::TKinokoControlViewlet(const std::string& Name, TKinokoControlImagePort* ImagePort)
: TKinokoControlWidget("Viewlet", Name)
{
    _ImagePort = ImagePort;
    _Canvas = new TKinokoControlViewletCanvas(_ImagePort);
}

TKinokoControlViewlet::~TKinokoControlViewlet()
{
    delete _Canvas;
    delete _ImagePort;
}

void TKinokoControlViewlet::Enable(void)
{
    TKinokoControlWidget::Enable();
}

void TKinokoControlViewlet::Disable(void)
{
    TKinokoControlWidget::Disable();
}

std::string TKinokoControlViewlet::Value(void)
{
    return "[Viewlet]";
}

void TKinokoControlViewlet::SetValue(const std::string& Value)
{
    _Canvas->ProcessCommandLine(Value.c_str());
}

void TKinokoControlViewlet::Initialize(void)
{
    _ImagePort->Initialize();
    _Canvas->Start();
}

void TKinokoControlViewlet::Redraw(void)
{
    _ImagePort->Redraw();
}

void TKinokoControlViewlet::SetColor(const std::string& ColorName)
{
    _Color = ColorName;
}

void TKinokoControlViewlet::SetForegroundColor(const std::string& ColorName)
{
    _ForegroundColor = ColorName;
}

void TKinokoControlViewlet::SetBackgroundColor(const std::string& ColorName)
{
    _BackgroundColor = ColorName;
}



TKinokoControlCanvasViewlet::TKinokoControlCanvasViewlet(const std::string& Name, TKinokoControlImagePort* ImagePort)
: TKinokoControlViewlet(Name, ImagePort)
{
}

TKinokoControlCanvasViewlet::~TKinokoControlCanvasViewlet()
{
}

void TKinokoControlCanvasViewlet::Initialize(void)
{
    TKinokoControlViewlet::Initialize();

    _Canvas->ProcessCommandLine(".create picture picture 0 0 100 100");
    _Canvas->ProcessCommandLine("picture setmargin 0 0 0 0");

    float Width = _ImagePort->GetImageArea()->Width();
    float Height = _ImagePort->GetImageArea()->Height();

    ostringstream Frame;
    Frame << "picture frame 0 0 " << Width << " " << Height;
    _Canvas->ProcessCommandLine(Frame.str().c_str());

    _Canvas->ProcessCommandLine("picture clear");
}

void TKinokoControlCanvasViewlet::SetValue(const std::string& Value)
{
    _Canvas->ProcessCommandLine(("picture " + Value).c_str());
}

void TKinokoControlCanvasViewlet::Clear(void)
{
    _Canvas->ProcessCommandLine("picture clear");
}

void TKinokoControlCanvasViewlet::DrawLine(float x0, float y0, float x1, float y1)
{
    //... TODO
}

void TKinokoControlCanvasViewlet::DrawCircle(float x, float y, float radius)
{
    //... TODO
}

void TKinokoControlCanvasViewlet::DrawText(float x, float y, const std::string& Text)
{
    //... TODO
}



TKinokoControlPlotViewlet::TKinokoControlPlotViewlet(const std::string& Name, TKinokoControlImagePort* ImagePort)
: TKinokoControlViewlet(Name, ImagePort)
{
    _IsLineUsed = true;
    _IsMarkerUsed = true;

    _IsRangeDecided = false;
    _IsYScaleLog = false;
}

TKinokoControlPlotViewlet::TKinokoControlPlotViewlet(const std::string& Name, TKinokoControlImagePort* ImagePort, float XMin, float XMax, float YMin, float YMax)
: TKinokoControlViewlet(Name, ImagePort)
{
    _IsLineUsed = true;
    _IsMarkerUsed = true;

    _IsRangeDecided = true;
    _IsYScaleLog = false;

    _XMin = min(XMin, XMax);
    _XMax = max(XMin, XMax);
    _YMin = min(YMin, YMax);
    _YMax = max(YMin, YMax);
}

TKinokoControlPlotViewlet::~TKinokoControlPlotViewlet()
{
}

void TKinokoControlPlotViewlet::Initialize(void)
{
    TKinokoControlViewlet::Initialize();

    _Canvas->ProcessCommandLine(".create plot plot 0 0 100 100");
    _Canvas->ProcessCommandLine("plot setmargin 1 3 5 2");
    _Canvas->ProcessCommandLine("plot set contentbackground white");
    _Canvas->ProcessCommandLine("plot clear");

    if (_IsRangeDecided) {
	ostringstream Command;
	Command << "plot frame ";
	Command << _XMin << " " << _XMax << " ";
	Command << _YMin << " " << _YMax << " ";
	_Canvas->ProcessCommandLine(Command.str().c_str());
    }

    if (! _Color.empty()) {
	// color for plotting //
	_Canvas->ProcessCommandLine(("plot set color " + _Color).c_str());
    }
    if (! _ForegroundColor.empty()) {
	// color for frame, label, etc. //
	_Canvas->ProcessCommandLine(("plot set foreground " + _ForegroundColor).c_str());
    }
    if (! _BackgroundColor.empty()) {
	_Canvas->ProcessCommandLine(("plot set contentbackground " + _BackgroundColor).c_str());
    }
}

void TKinokoControlPlotViewlet::SetValue(const std::string& Value)
{
    _Canvas->ProcessCommandLine(("plot " + Value).c_str());
}

void TKinokoControlPlotViewlet::Clear(void)
{
    _Canvas->ProcessCommandLine("plot clear");

    if (_IsRangeDecided) {
	_Canvas->ProcessCommandLine("plot drawframe");
	_Canvas->ProcessCommandLine("plot drawxgrid");
	_Canvas->ProcessCommandLine("plot drawygrid");
    }
}

void TKinokoControlPlotViewlet::SetRange(float XMin, float XMax, float YMin, float YMax)
{
    _XMin = min(XMin, XMax);
    _XMax = max(XMin, XMax);
    _YMin = min(YMin, YMax);
    _YMax = max(YMin, YMax);
    _IsRangeDecided = true;

    _Canvas->ProcessCommandLine("plot clear");

    ostringstream Command;
    Command << "plot frame ";
    Command << _XMin << " " << _XMax << " " << _YMin << " " << _YMax << " ";
    _Canvas->ProcessCommandLine(Command.str().c_str());

    _Canvas->ProcessCommandLine("plot drawframe");
    _Canvas->ProcessCommandLine("plot drawxgrid");
    _Canvas->ProcessCommandLine("plot drawygrid");
}

void TKinokoControlPlotViewlet::SetYScaleLog(void)
{
    _Canvas->ProcessCommandLine("plot set yscale log");
    Clear();
}

void TKinokoControlPlotViewlet::SetTimeTick(long Epoch, const std::string& Format, std::string Unit, int Step)
{
    if (Unit.empty()) {
	Unit = "-";
    }

    ostringstream Command;
    Command << "plot setxscaletimetick" << " ";
    Command << Epoch << " " << Unit << " " << Step << " " << Format;
    _Canvas->ProcessCommandLine(Command.str().c_str());

    Clear();
}

void TKinokoControlPlotViewlet::ResetRange(void)
{
    _IsRangeDecided = false;
    _Canvas->ProcessCommandLine("plot clear");
}

void TKinokoControlPlotViewlet::GuessRange(TKumaVector& XList, TKumaVector& YList)
{
    _XMin = _XMax = XList[0];
    _YMin = _YMax = YList[0];

    for (int i = 1; i < XList.Size(); i++) {
	_XMin = min((float) XList[i], _XMin);
	_XMax = max((float) XList[i], _XMax);
	_YMin = min((float) YList[i], _YMin);
	_YMax = max((float) YList[i], _YMax);
    }

    double XWidth = (_XMax - _XMin) / 2;
    double XCenter = (_XMax + _XMin) / 2;
    double YWidth = (_YMax - _YMin) / 2;
    double YCenter = (_YMax + _YMin) / 2;
    
    if (fabs(_XMin) < (_XMax - _XMin) / 100.0) {
	_XMin = 0;
    }
    else {
	_XMin = XCenter - 1.1 * (XWidth > 0 ? XWidth : 1);
    }
    _XMax = XCenter + 1.1 * (XWidth > 0 ? XWidth : 1);
    
    if (fabs(_YMin) < (_YMax - _YMin) / 100.0) {
	_YMin = 0;
    }
    else {
	_YMin = YCenter - 1.3 * (YWidth > 0 ? YWidth : 1);
    }
    _YMax = YCenter + 1.3 * (YWidth > 0 ? YWidth : 1);
    
    if (_IsYScaleLog && (_YMin <= 0)) {
	_YMin = 0.5;
    }

    SetRange(_XMin, _XMax, _YMin, _YMax);
}

void TKinokoControlPlotViewlet::SetLineWidth(int LineWidth)
{
    ostringstream Command;
    Command << "plot set linewidth " << LineWidth;
    _Canvas->ProcessCommandLine(Command.str().c_str());

    _IsLineUsed = (LineWidth > 0);
}

void TKinokoControlPlotViewlet::SetMarker(const std::string& MarkerName, int MarkerSize)
{
    if (! MarkerName.empty()) {
	_Canvas->ProcessCommandLine(("plot set marker " + MarkerName).c_str());
    }

    if (MarkerSize > 0) {
	ostringstream Command;
	Command << "plot set markersize " << MarkerSize;
	_Canvas->ProcessCommandLine(Command.str().c_str());
    }

    _IsMarkerUsed = (! MarkerName.empty()) && (MarkerSize > 0);
}

void TKinokoControlPlotViewlet::Draw(TKumaVector& XList, TKumaVector& YList)
{
    if (! _IsRangeDecided) {
	GuessRange(XList, YList);
    }

    ostringstream Command;
    Command << "plot ";

    if (_IsMarkerUsed && _IsLineUsed) {
	Command << "plot ";
    }
    else if (_IsMarkerUsed) {
	Command << "pointsplot ";
    }
    else {
	Command << "linesplot ";
    }

    for (int i = 0; i < XList.Size(); i++) {
	Command << XList[i] << " " << YList[i] << " ";
    }

    _Canvas->ProcessCommandLine(Command.str().c_str());
}
