/* KinokoMonitor.cc */
/* Created by Enomoto Sanshiro on 16 May 2004. */
/* Last updated by Enomoto Sanshiro on 31 October 2005. */


#include <string>
#include <sstream>
#include "MushXml.hh"
#include "MushFileSystem.hh"
#include "MushChainedFile.hh"
#include "MushMisc.hh"
#include "KinokoMonitor.hh"

using namespace std;
using namespace mush;



TKinokoMonitorComponentCondition::TKinokoMonitorComponentCondition(const std::string& Name)
{
    _Name = Name;

    _CheckTime = 0;
    _State = "UNKNOWN";
    _HeartBeatCount = 0;
}

TKinokoMonitorComponentCondition::~TKinokoMonitorComponentCondition()
{
}

void TKinokoMonitorComponentCondition::Update(const std::string& CheckTime, const std::string& State, const std::string& HeartBeatCount)
{
    {
	istringstream is(CheckTime);
	is >> _CheckTime;
    }

    _State = State;

    {
	istringstream is(HeartBeatCount);
	is >> _HeartBeatCount;
    }
}



TKinokoMonitorFileCondition::TKinokoMonitorFileCondition(const std::string& Name)
{
    _Name = Name;
    _CheckTime = TMushDateTime::SecSinceEpoch();
    _GrowingSpeed = 0;

    TMushChainedFileAttribute File(_Name);
    if (File.Exists()) {
	_FileSize = File.Size();
    }
    else {
	_FileSize = 0;
    }

    ostringstream os;
    os << _FileSize;
    _FileSizeString = os.str();
}

TKinokoMonitorFileCondition::~TKinokoMonitorFileCondition()
{
}

void TKinokoMonitorFileCondition::Update(void)
{
    long CheckTime = TMushDateTime::SecSinceEpoch();
    long FileSize;

    TMushChainedFileAttribute File(_Name);
    if (File.Exists()) {
	FileSize = File.Size();
    }
    else {
	FileSize = 0;
    }

    if (CheckTime > _CheckTime) {
	_GrowingSpeed = (FileSize - _FileSize) / (CheckTime - _CheckTime);
    }

    if (FileSize != _FileSize) {
	ostringstream os;
	os << FileSize;
	_FileSizeString = os.str();
    }

    _FileSize = FileSize;
    _CheckTime = CheckTime;
}



TKinokoMonitor::TKinokoMonitor(void)
{
    _Parser = new sax::Parser();
    _Parser->setDocumentHandler(this);

    _UpdateTime = TMushDateTime::SecSinceEpoch();
    _FileTime = _UpdateTime;
}

TKinokoMonitor::~TKinokoMonitor()
{
    delete _Parser;

    map<string, TKinokoMonitorComponentCondition*>::iterator i;
    for (i = _ComponentList.begin(); i != _ComponentList.end(); i++) {
	delete i->second;
    }

    map<string, TKinokoMonitorFileCondition*>::iterator j;
    for (j = _FileList.begin(); j != _FileList.end(); j++) {
	delete j->second;
    }
}

void TKinokoMonitor::Update(const std::string& RegistryFileName)
{
    try {
	_State = State_Init;
	_Parser->parse(RegistryFileName);

	_UpdateTime = TMushDateTime::SecSinceEpoch();
	_FileTime = TMushFileAttribute(RegistryFileName).LastModificationTime();
    }
    catch (TSystemCallException &e) {
	cout << "ERROR: " << e << ": retrying..." << endl;
	;
    }

    map<string, TKinokoMonitorFileCondition*>::iterator i;
    for (i = _FileList.begin(); i != _FileList.end(); i++) {
	i->second->Update();
    }
}

void TKinokoMonitor::startElement(const std::string& Name, const sax::AttributeList& AttributeList)
{
    if (_State == State_Init) {
	if (Name == "kcom-registry") {
	    _State = State_Root;
	}
	else {
	    //...
	    cerr << "ERROR: bad XML root node: kcom-registry is expected" << endl;
	}
    }
    else if (_State == State_Root) {
	if (Name == "__properties") {
	    _State = State_ComponentList;
	}
	else if (Name == "monitor") {
	    _State = State_Monitor;
	}
	else {
	    _State = State_Others;
	    _Depth = 0;
	}
    }
    else if (_State == State_ComponentList) {
	_State = State_PropertyList;
    }
    else if (_State == State_PropertyList) {
	_State = State_Property;
    }
    else if (_State == State_Monitor) {
	_State = State_MonitorElement;
    }

    if (_State == State_Others) {
	_Depth++;
    }
}

void TKinokoMonitor::endElement(const std::string& Name) 
{
    if (_State == State_Property) {
	_CurrentPropertyList[Name] = _CurrentValue;

	if ((Name == "data_file_name") && ! _CurrentValue.empty()) {
	    const string& DataFileName = _CurrentValue;
	    if (_FileList.count(DataFileName) == 0) {
		_FileList[DataFileName] = new TKinokoMonitorFileCondition(
		    DataFileName
		);
		_FileNameList.push_back(DataFileName);
	    }
	}

	_CurrentValue.erase();
	_State = State_PropertyList;
    }
    else if (_State == State_PropertyList) {
	TKinokoMonitorComponentCondition* Component;
	if (_ComponentList.count(Name) == 0) {
	    Component = new TKinokoMonitorComponentCondition(Name);
	    _ComponentList[Name] = Component;
	    _ComponentNameList.push_back(Name);
	}
	else {
	    Component = _ComponentList[Name];
	}
	Component->Update(
	    _CurrentPropertyList["condition_check_time"],
	    _CurrentPropertyList["state"],
	    _CurrentPropertyList["heart_beat_count"]
	);
	_State = State_ComponentList;
    }
    else if (_State == State_ComponentList) {
	_State = State_Root;
    }

    else if (_State == State_MonitorElement) {
	if ((Name == "data_file") && ! _CurrentValue.empty()) {
	    const string& DataFileName = _CurrentValue;
	    if (_FileList.count(DataFileName) == 0) {
		_FileList[DataFileName] = new TKinokoMonitorFileCondition(
		    DataFileName
		);
		_FileNameList.push_back(DataFileName);
	    }
	}

	_CurrentValue.erase();
	_State = State_Monitor;
    }

    else if (_State == State_Others) {
	_Depth--;
	if (_Depth == 0) {
	    _State = State_Root;
	}
    }
}

void TKinokoMonitor::characters(const std::string& text)
{
    if (_State == State_Property) {
	_CurrentValue += text;
    }
}
