/* KinokoCanvasGtk.cc */
/* Created by Enomoto Sanshiro on 9 July 2000. */
/* Last updated by Enomoto Sanshiro on 19 August 2008. */


#include <string>
#include <cstdlib>
#include <gtk/gtk.h>
#include "KinokoShellConfig.hh"
#include "KinokoCanvas.hh"
#include "KinokoCanvasImageAreaGtk.hh"
#include "KinokoCanvasImageAreaGtkPango.hh"
#include "KinokoCanvasGtk.hh"
#include "KinokoShellFileSelectDialogGtk.hh"
#include "KinokoShellAboutDialogGtk.hh"
#include "KinokoShellPopupWindowGtk.hh"
#include "KinokoShellConfig.hh"

using namespace std;


static TKinokoCanvasGtk* g_Canvas = 0;
static gint g_InputTag;
static bool g_IsButtonBeingPressed = false;
static bool g_IsInputPostponed = false;
static int g_ButtonPressPage;

static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data);
static gint expose_event_cb(GtkWidget* Widget, GdkEventExpose* Event, gpointer Data);
static gint button_press_event_cb(GtkWidget* Widget, GdkEventButton* Event, gpointer Data);
static gint button_release_event_cb(GtkWidget* Widget, GdkEventButton* Event, gpointer Data);
static gboolean mouse_move_event_cb(GtkWidget *widget, GdkEventMotion *Event, gpointer Data);
static void context_menu_selected_cb(GtkWidget* Widget, gpointer Data);
static void context_menu_close_cb(GtkWidget* Widget, gpointer Data);
static void input_event_cb(gpointer Data, gint File, GdkInputCondition Condition);
static void menu_save_image_cb(void);
static void menu_quit_cb(void);
static void menu_about_cb(void);


static GtkItemFactoryEntry MenuItemList[] = {
    {"/_File",               NULL,         0,                  0, "<Branch>"},
    {"/File/_Save Image...", NULL,         menu_save_image_cb, 0, "<Item>"},
    {"/File/",               NULL,         0,                  0, "<Separator>"},
    {"/File/e_Xit",          "<control>X", menu_quit_cb,       0, "<Item>"},
    {"/_Help",               NULL,         0,                  0, "<Branch>"},
    {"/Help/_About...",      NULL,         menu_about_cb,      0, "<Item>"}
};
static gint NumberOfMenuItems = sizeof(MenuItemList) / sizeof(MenuItemList[0]);



TKinokoCanvasGtk::TKinokoCanvasGtk(TKinokoShellConnector* ShellConnector, int argc, char** argv, int Width, int Height, int Left, int Top, bool EnableEps)
: TKinokoCanvas(ShellConnector)
{
    _Width = Width;
    _Height = Height;
    _Left = Left;
    _Top = Top;
    _IsEpsEnabled = EnableEps;

    _RootWindow = 0;

    _CurrentContextMenuObject = 0;
    _CurrentContextMenu = 0;

    gtk_init(&argc, &argv);
}

TKinokoCanvasGtk::~TKinokoCanvasGtk()
{
    for (unsigned i = 0; i < _ImageAreaList.size(); i++) {
	delete _ImageAreaList[i];
    }
}

TKinokoShellPopupWindow* TKinokoCanvasGtk::CreatePopupWindow(void)
{
    return new TKinokoShellPopupWindowGtk();
}

TKinokoShellFileSelectDialog* TKinokoCanvasGtk::CreateFileSelectDialog(void)
{
    return new TKinokoShellFileSelectDialogGtk();
}

void TKinokoCanvasGtk::Start(void)
{
    if (g_Canvas == 0) {
	g_Canvas = this;
	BuildRootWindow();
	
	g_InputTag = gdk_input_add(
	    _ShellConnector->FileDescriptor(),
	    (GdkInputCondition) (GDK_INPUT_READ | GDK_INPUT_EXCEPTION), 
	    input_event_cb, NULL
	);
	
	CreatePage();
    }

    g_IsButtonBeingPressed = false;
    g_IsInputPostponed = false;

    gtk_main();
}

void TKinokoCanvasGtk::Quit(void)
{
    gtk_main_quit();
}

void TKinokoCanvasGtk::Detach(void)
{
    gdk_input_remove(g_InputTag);
    _ShellConnector->Disconnect();
}

void TKinokoCanvasGtk::BuildRootWindow(void)
{
    _RootWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    // Menu and Menubar //
    GtkAccelGroup* AccelGroup = gtk_accel_group_new();
    GtkItemFactory* ItemFactory = gtk_item_factory_new(
	GTK_TYPE_MENU_BAR, "<menu>", AccelGroup
    );
    gtk_item_factory_create_items(
	ItemFactory, NumberOfMenuItems, MenuItemList, NULL
    );
    gtk_window_add_accel_group(GTK_WINDOW(_RootWindow), AccelGroup);
    GtkWidget* Menu = gtk_item_factory_get_widget(ItemFactory, "<menu>");
    GtkWidget* MenuHandleBox = gtk_handle_box_new();
    gtk_container_add(GTK_CONTAINER(MenuHandleBox), Menu);

    // Notebook //
    _Notebook = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(_Notebook), GTK_POS_TOP);
    gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(_Notebook), TRUE);
    gtk_notebook_set_scrollable(GTK_NOTEBOOK(_Notebook), TRUE);

    // Packing //
    GtkWidget* VBox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(_RootWindow), VBox);
    gtk_box_pack_start(GTK_BOX(VBox), MenuHandleBox, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(VBox), _Notebook, TRUE, TRUE, 0);

    // Event Handler Connection //
    gtk_signal_connect(
	GTK_OBJECT(_RootWindow), "delete_event", 
	GTK_SIGNAL_FUNC(delete_event_cb), NULL
    );

    // Show //
    gtk_widget_show_all(_RootWindow);
    if ((_Top >= 0) && (_Left >= 0)) {
	// moves the window after show() because most window managers 
	// ignore requests for initial window positions.
	gtk_window_move(GTK_WINDOW(_RootWindow), _Left, _Top);
    }
}

int TKinokoCanvasGtk::Reset(void)
{
    g_IsButtonBeingPressed = false;
    g_IsInputPostponed = false;

    int Result = TKinokoCanvas::Reset();

    for (unsigned i = 0; i < _ImageAreaList.size(); i++) {
	delete _ImageAreaList[i];
    }
    _ImageAreaList.erase(_ImageAreaList.begin(), _ImageAreaList.end());

    int NumberOfPages = gtk_notebook_get_n_pages(GTK_NOTEBOOK(_Notebook));
    for (int i = 0; i < NumberOfPages; i++) {
	gtk_notebook_remove_page(GTK_NOTEBOOK(_Notebook), -1);
    }

    CreatePage();

    return Result;
}

int TKinokoCanvasGtk::SetTitle(const string& Title)
{
    gtk_window_set_title(GTK_WINDOW(_RootWindow), Title.c_str());
    
    return 1;
}

int TKinokoCanvasGtk::SetPageTitle(const string& Title)
{
    GtkWidget* PageWidget = gtk_notebook_get_nth_page(
	GTK_NOTEBOOK(_Notebook), _CurrentPageNumber
    );
    if (PageWidget != NULL) {
	gtk_notebook_set_tab_label_text(
	    GTK_NOTEBOOK(_Notebook), PageWidget, Title.c_str()
	);
    }

    return 1;
}

int TKinokoCanvasGtk::CreatePage(const string& PageName)
{
    string ThisPageName = PageName;
    if (ThisPageName.empty()) {
	ThisPageName = "untitled";
    }

    if (_ImageAreaList.empty()) {
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(_Notebook), FALSE);
    }
    else {
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(_Notebook), TRUE);
    }

    GtkWidget* DrawingAreaWidget = gtk_drawing_area_new();
    gtk_drawing_area_size(
	GTK_DRAWING_AREA(DrawingAreaWidget), _Width, _Height
    );
    gtk_widget_set_events(
	DrawingAreaWidget, 
	GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	| GDK_BUTTON_MOTION_MASK
    );

    GtkWidget* PageLabel = gtk_label_new(ThisPageName.c_str());
    GtkWidget* PageBox = gtk_hbox_new(FALSE, 0);
    gtk_notebook_append_page(GTK_NOTEBOOK(_Notebook), PageBox, PageLabel); 
    gtk_box_pack_start(GTK_BOX(PageBox), DrawingAreaWidget, TRUE, TRUE, 0);
    gtk_widget_show_all(PageBox);
    gtk_widget_show_all(PageLabel);

    gtk_signal_connect(
	GTK_OBJECT(DrawingAreaWidget), "expose_event", 
	GTK_SIGNAL_FUNC(expose_event_cb), _Notebook
    );
    gtk_signal_connect(
	GTK_OBJECT(DrawingAreaWidget), "button_press_event", 
	GTK_SIGNAL_FUNC(button_press_event_cb), _Notebook
    );
    gtk_signal_connect(
	GTK_OBJECT(DrawingAreaWidget), "button_release_event", 
	GTK_SIGNAL_FUNC(button_release_event_cb), NULL
    );
    gtk_signal_connect(
	GTK_OBJECT(DrawingAreaWidget), "motion_notify_event", 
	GTK_SIGNAL_FUNC(mouse_move_event_cb), NULL
    );

    TKinokoCanvasImageArea* ImageArea = (
#if USE_PANGO
	new TKinokoCanvasImageAreaGtkPango(DrawingAreaWidget, _Width, _Height)
#else
	new TKinokoCanvasImageAreaGtk(DrawingAreaWidget, _Width, _Height)
#endif
    );
    _ImageAreaList.push_back(ImageArea);
    AddPage(ImageArea, _IsEpsEnabled);

    gtk_notebook_set_current_page(GTK_NOTEBOOK(_Notebook), 0);

    return TKinokoCanvas::CreatePage(ThisPageName);
}

int TKinokoCanvasGtk::OpenContextMenu(TKinokoCanvasObject* CanvasObject)
{
    string ObjectName = CanvasObject->Name();
    vector<string> ActionList = CanvasObject->ActionList();
    if (ActionList.empty()) {
	return 0;
    }

    GtkWidget* Menu = gtk_menu_new();
    gtk_menu_set_title(GTK_MENU(Menu), ObjectName.c_str());

    for (unsigned i = 0; i < ActionList.size(); i++) {
	string Label = ActionList[i];
	GtkWidget* MenuItem = gtk_menu_item_new_with_label(Label.c_str());
	gtk_menu_append(GTK_MENU(Menu), MenuItem);
	gtk_widget_show(MenuItem);

	char* ActionName = strdup(Label.c_str());
	_CurrentContextMenuActionNameList.push_back(ActionName);

	gtk_signal_connect(
	    GTK_OBJECT(MenuItem), 
	    "activate", GTK_SIGNAL_FUNC(context_menu_selected_cb), ActionName
	);
    }
    
    _CurrentContextMenuObject = CanvasObject;
    _CurrentContextMenu = Menu;

    gtk_widget_ref(Menu);
    gtk_signal_connect(
	GTK_OBJECT(Menu),
	"selection-done", GTK_SIGNAL_FUNC(context_menu_close_cb), NULL
    );

    gtk_menu_popup(
	GTK_MENU(Menu), 
	NULL, NULL, NULL, NULL, 3, gtk_get_current_event_time()
    );

    return 1;
}

int TKinokoCanvasGtk::CloseContextMenu(void)
{
    if (_CurrentContextMenu != 0) {
	gtk_widget_unref(_CurrentContextMenu);
	_CurrentContextMenu = 0;
    }
    while (! _CurrentContextMenuActionNameList.empty()) {
	free(_CurrentContextMenuActionNameList.back());
	_CurrentContextMenuActionNameList.pop_back();
    }

    return 1;
}

int TKinokoCanvasGtk::ProcessContextMenuAction(const char* ActionName)
{
    int Result = 0;

    if (_CurrentContextMenuObject != 0) {
	Result = _CurrentContextMenuObject->ProcessAction(ActionName);
	_CurrentContextMenuObject = 0;
    }

    return Result;
}

int TKinokoCanvasGtk::SavePageImageTo(const std::string& FileName)
{
    SelectPage(gtk_notebook_get_current_page(GTK_NOTEBOOK(_Notebook)));
    SaveImageTo(FileName);

    return 1;
}

static void context_menu_selected_cb(GtkWidget* Widget, gpointer Data)
{
    if (g_Canvas) {
	g_Canvas->ProcessContextMenuAction((char*) Data);
    }
}

static void context_menu_close_cb(GtkWidget* Widget, gpointer Data)
{
    if (g_Canvas) {
	g_Canvas->CloseContextMenu();
    }
}

static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data)
{
    bool Result = g_Canvas->TryToQuit();
    return Result ? FALSE : TRUE;
}

static gint expose_event_cb(GtkWidget* Widget, GdkEventExpose* Event, gpointer Data)
{
    if (g_Canvas) {
	int PageNumber = gtk_notebook_get_current_page(GTK_NOTEBOOK(Data));
	g_Canvas->Redraw(PageNumber);
    }

    return FALSE;
}

static gint button_press_event_cb(GtkWidget* Widget, GdkEventButton* Event, gpointer Data)
{
    g_ButtonPressPage = gtk_notebook_get_current_page(GTK_NOTEBOOK(Data));

    if (g_Canvas) {
	g_Canvas->OnButtonPress(
	    g_ButtonPressPage, Event->button, (int) Event->x, (int) Event->y
	);
    }

    if (Event->button == 1) {
	g_IsButtonBeingPressed = true;
	g_IsInputPostponed = false;
    }

    return TRUE;
}

static gint button_release_event_cb(GtkWidget* Widget, GdkEventButton* Event, gpointer Data)
{
    if (g_Canvas) {
	g_Canvas->OnButtonRelease(
	    g_ButtonPressPage, Event->button, (int) Event->x, (int) Event->y
	);
    }

    g_IsButtonBeingPressed = false;
    if (g_IsInputPostponed) {
	int ProcessedLength = g_Canvas->ProcessInput();
	if (ProcessedLength <= 0) {
	    gdk_input_remove(g_InputTag);
	}
    }

    return TRUE;
}

static gint mouse_move_event_cb(GtkWidget* Widget, GdkEventMotion* Event, gpointer Data)
{
    if (g_Canvas) {
	g_Canvas->OnMouseMove(
	    g_ButtonPressPage, (int) Event->x, (int) Event->y
	);
    }

    return TRUE;
}

static void input_event_cb(gpointer Data, gint File, GdkInputCondition Condition)
{
    if (g_Canvas == 0) {
	return;
    }

    int ProcessedLength = 0;
    if (Condition == GDK_INPUT_READ) {
	if (g_IsButtonBeingPressed) {
	    g_IsInputPostponed = true;
	    return;
	}
	ProcessedLength = g_Canvas->ProcessInput();
    }
    
    if ((Condition == GDK_INPUT_EXCEPTION) || (ProcessedLength <= 0)) {
	gdk_input_remove(g_InputTag);
    }
}

static void menu_save_image_cb(void)
{
    TKinokoShellFileSelectDialogGtk FileSelectDialog("Save Image");

    string FileName = FileSelectDialog.Open();
    if (! FileName.empty()) {
	g_Canvas->SavePageImageTo(FileName);
    }
}

static void menu_quit_cb(void)
{
    g_Canvas->TryToQuit();
}

static void menu_about_cb(void)
{
    TKinokoShellAboutDialogGtk AboutDialog;

    AboutDialog.SetTitle("Kinoko Canvas " KINOKO_SHELL_VERSION);
    AboutDialog.SetCopyright(KINOKO_SHELL_COPYING);
    AboutDialog.AddAuthor(KINOKO_SHELL_AUTHOR);
    AboutDialog.AddComment("Remotely accessible image and plot display");

    AboutDialog.Open();
}

