/* kinoko-browser-gtk.cc */
/* Created by Enomoto Sanshiro on 1 August 2003. */
/* Last updated by Enomoto Sanshiro on 1 August 2003. */


#include <string>
#include <vector>
#include <deque>
#include <map>
#include <cstring>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include "MushXml.hh"
#include "KinokoShellAboutDialogGtk.hh"
#include "KinokoShellFileSelectDialogGtk.hh"
#include "KinokoShellConfig.hh"


using namespace std;


class TKinokoXmlElement {
  public:
    TKinokoXmlElement(const std::string& Name, const TMushSaxAttributeList& AttributeList);
    virtual ~TKinokoXmlElement();
    virtual void AddText(const std::string& Text);
    virtual void AddComment(const std::string& Comment);
    virtual const std::string& Name(void);
    virtual const TMushSaxAttributeList& AttributeList(void);
    virtual const std::string& Text(void);
    virtual const std::string& Comment(void);
  protected:
    std::string _Name;
    TMushSaxAttributeList _AttributeList;
    std::string _Text;
    std::string _Comment;
};


TKinokoXmlElement::TKinokoXmlElement(const std::string& Name, const TMushSaxAttributeList& AttributeList)
    : _Name(Name), _AttributeList(AttributeList)
{
}

TKinokoXmlElement::~TKinokoXmlElement()
{
}

void TKinokoXmlElement::AddText(const std::string& Text)
{
    _Text += Text;
}

void TKinokoXmlElement::AddComment(const std::string& Text)
{
    _Comment += Text;
}

const std::string& TKinokoXmlElement::Name(void)
{
    return _Name;
}

const TMushSaxAttributeList& TKinokoXmlElement::AttributeList(void)
{
    return _AttributeList;
}

const std::string& TKinokoXmlElement::Text(void)
{
    return _Text;
}

const std::string& TKinokoXmlElement::Comment(void)
{
    return _Comment;
}


///////////////////////


class TKinokoBrowserGtk: public TMushSaxDocumentHandler {
  public:
    TKinokoBrowserGtk(int argc, char** argv);
    virtual ~TKinokoBrowserGtk();
    virtual void Update(const std::string& FileName);
    virtual void Start(void);
    virtual void Quit(void);
  public:
    virtual void startElement(const std::string& name, const TMushSaxAttributeList& attribute_list);
    virtual void endElement(const std::string& name);
    virtual void characters(const std::string& text);
    virtual void comment(const std::string& text);
  protected:
    void CreateRootWindow(int Width, int Height);
  protected:
    TMushSaxParser* _Parser;
    std::string _InputFileName;
    TKinokoXmlElement* _CurrentElement;
    deque<GtkCTreeNode*> _TreeNodeStack;
  public:
    GtkWidget *_RootWindow;
    GtkWidget *_ElementCTree;
    GtkWidget *_AttributeCList;
    GtkWidget *_ContentText;
    GtkWidget *_Statusbar;
    map<void*, TKinokoXmlElement*> _NodeElementTable;
};


TKinokoBrowserGtk::TKinokoBrowserGtk(int argc, char** argv)
{
    _InputFileName = argv[1];
    _Parser = new TMushSaxParser();
    _Parser->setDocumentHandler(this);

    gtk_init(&argc, &argv);
}

TKinokoBrowserGtk::~TKinokoBrowserGtk()
{
    delete _Parser;
}

void TKinokoBrowserGtk::Update(const std::string& FileName)
{
    try {
	_Parser->parse(FileName);
    }
    catch (TSystemCallException &e) {
	cerr << "ERROR: " << e << endl;
    }
}

void TKinokoBrowserGtk::Start(void)
{
    CreateRootWindow(800, 600);
    Update(_InputFileName);
    gtk_widget_show_all(_RootWindow);
    gtk_main ();
}

void TKinokoBrowserGtk::Quit(void)
{
    gtk_main_quit();
}

void TKinokoBrowserGtk::startElement(const std::string& Name, const TMushSaxAttributeList& AttributeList)
{
    _CurrentElement = new TKinokoXmlElement(Name, AttributeList);

    GtkCTreeNode* Parent;
    if (_TreeNodeStack.empty()) {
	Parent = NULL;
    }
    else {
	Parent = _TreeNodeStack.back();
    }

    //...
    char* NameBuffer = new char[Name.size() + 1];
    strcpy(NameBuffer, Name.c_str());
    
    GtkCTreeNode* Node = gtk_ctree_insert_node(
	GTK_CTREE(_ElementCTree), Parent, NULL,
	&NameBuffer, 8, NULL, NULL, NULL, NULL,
	false, true
    );

    _TreeNodeStack.push_back(Node);
    _NodeElementTable[Node] = _CurrentElement;
}

void TKinokoBrowserGtk::endElement(const std::string& Name) 
{
    _CurrentElement = _NodeElementTable[_TreeNodeStack.back()];
    _TreeNodeStack.pop_back();
}

void TKinokoBrowserGtk::characters(const std::string& Text)
{
    _CurrentElement->AddText(Text);
}

void TKinokoBrowserGtk::comment(const std::string& Text)
{
    _CurrentElement->AddComment(Text);
}

/////////////////////


static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data);
static void tree_select_row_cb(GtkCTree* Tree, GList* Node, gint column, gpointer Data);
static void menu_exit_cb(void);
static void menu_about_cb(void);


static GtkItemFactoryEntry MenuItemList[] = {
    {"/_File",          NULL,         0,             0, "<Branch>"},
    {"/File/e_Xit",     "<control>X", menu_exit_cb,  0, "<Item>"},
    {"/_Help",          NULL,         0,             0, "<Branch>"},
    {"/Help/_About...", NULL,         menu_about_cb, 0, "<Item>"}
};
static gint NumberOfMenuItems = sizeof(MenuItemList) / sizeof(MenuItemList[0]);


void TKinokoBrowserGtk::CreateRootWindow(int Width, int Height)
{
    // RootWindow //
    _RootWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_object_set_data(GTK_OBJECT(_RootWindow), "root_window", _RootWindow);
    gtk_window_set_title(GTK_WINDOW(_RootWindow), "kinoko-browser");
    gtk_window_set_default_size(GTK_WINDOW(_RootWindow), Width, Height);
    
    GtkWidget* RootVBox = gtk_vbox_new(FALSE, 0);
    gtk_widget_ref(RootVBox);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "RootVBox", RootVBox,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(RootVBox);
    gtk_container_add(GTK_CONTAINER(_RootWindow), RootVBox);
    
    // Menu and Menubar //
    GtkAccelGroup* AccelGroup = gtk_accel_group_new();
    GtkItemFactory* ItemFactory = gtk_item_factory_new(
	GTK_TYPE_MENU_BAR, "<menu>", AccelGroup
    );
    gtk_item_factory_create_items(
	ItemFactory, NumberOfMenuItems, MenuItemList, NULL
    );
    gtk_window_add_accel_group(GTK_WINDOW(_RootWindow), AccelGroup);
    GtkWidget* MenuBar = gtk_item_factory_get_widget(ItemFactory, "<menu>");
    GtkWidget* MenuHandleBox = gtk_handle_box_new();
    gtk_container_add(GTK_CONTAINER(MenuHandleBox), MenuBar);
    gtk_box_pack_start(GTK_BOX(RootVBox), MenuHandleBox, FALSE, FALSE, 0);

    // Element-Content Paned //
    GtkWidget* ElementContentPaned = gtk_hpaned_new();
    gtk_widget_ref(ElementContentPaned);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ElementContentPaned", ElementContentPaned,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(ElementContentPaned);
    gtk_box_pack_start(GTK_BOX(RootVBox), ElementContentPaned, TRUE, TRUE, 0);
    gtk_paned_set_position(GTK_PANED(ElementContentPaned), Width / 3);
    
    // Element Tree //
    GtkWidget* ElementWindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_ref(ElementWindow);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ElementWindow", ElementWindow,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(ElementWindow);
    gtk_paned_pack1(
	GTK_PANED(ElementContentPaned), ElementWindow, FALSE, TRUE
    );
    
    _ElementCTree = gtk_ctree_new(1, 0);
    gtk_widget_ref(_ElementCTree);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ElementCTree", _ElementCTree,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(_ElementCTree);
    gtk_container_add(GTK_CONTAINER(ElementWindow), _ElementCTree);
    gtk_clist_set_column_width(GTK_CLIST(_ElementCTree), 0, 80);
    gtk_clist_column_titles_hide(GTK_CLIST(_ElementCTree));
    
    GtkWidget* ElementLabel = gtk_label_new("ElementName");
    gtk_widget_ref(ElementLabel);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ElementLabel", ElementLabel,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(ElementLabel);
    gtk_clist_set_column_widget(
	GTK_CLIST(_ElementCTree), 0, ElementLabel
    );
    
    // Attribute-Content Paned //
    GtkWidget* AttributeContentPaned = gtk_vpaned_new();
    gtk_widget_ref(AttributeContentPaned);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "AttributeContentPaned", AttributeContentPaned,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(AttributeContentPaned);
    gtk_paned_pack2(
	GTK_PANED(ElementContentPaned), AttributeContentPaned, TRUE, TRUE
    );
    gtk_paned_set_position(GTK_PANED(AttributeContentPaned), Height/2);
    
    // Attribute List //
    GtkWidget* AttributeWindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_ref(AttributeWindow);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "AttributeWindow", AttributeWindow,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(AttributeWindow);
    gtk_paned_pack1(
	GTK_PANED(AttributeContentPaned), AttributeWindow, FALSE, TRUE
    );
    
    _AttributeCList = gtk_clist_new(2);
    gtk_widget_ref(_AttributeCList);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "AttributeCList", _AttributeCList,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(_AttributeCList);
    gtk_container_add(GTK_CONTAINER(AttributeWindow), _AttributeCList);
    gtk_clist_set_column_width(GTK_CLIST(_AttributeCList), 0, 80);
    gtk_clist_set_column_width(GTK_CLIST(_AttributeCList), 1, 80);
    gtk_clist_column_titles_show(GTK_CLIST(_AttributeCList));
    
    GtkWidget* AttributeNameLabel = gtk_label_new("Name");
    gtk_widget_ref(AttributeNameLabel);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "AttributeNameLabel", AttributeNameLabel,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(AttributeNameLabel);
    gtk_clist_set_column_widget(
	GTK_CLIST(_AttributeCList), 0, AttributeNameLabel
    );
    
    GtkWidget* AttributeValueLabel = gtk_label_new("Value");
    gtk_widget_ref(AttributeValueLabel);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), 
	"AttributeValueLabel", AttributeValueLabel,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(AttributeValueLabel);
    gtk_clist_set_column_widget(
	GTK_CLIST(_AttributeCList), 1, AttributeValueLabel
    );
    
    // Contents Box //
    GtkWidget* ContentWindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_ref(ContentWindow);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ContentWindow", ContentWindow,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(ContentWindow);
    gtk_paned_pack2(
	GTK_PANED(AttributeContentPaned), ContentWindow, TRUE, TRUE
    );
    gtk_scrolled_window_set_policy(
	GTK_SCROLLED_WINDOW(ContentWindow), 
	GTK_POLICY_NEVER, GTK_POLICY_ALWAYS
    );
    
    _ContentText = gtk_text_view_new();
    gtk_widget_ref(_ContentText);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ContentText", _ContentText,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(_ContentText);
    gtk_container_add(GTK_CONTAINER(ContentWindow), _ContentText);
    
    _Statusbar = gtk_statusbar_new();
    gtk_widget_ref(_Statusbar);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "StatusBar", _Statusbar,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(_Statusbar);
    gtk_box_pack_start(GTK_BOX(RootVBox), _Statusbar, FALSE, FALSE, 0);
    
    // Signal Connection //
    gtk_signal_connect(
	GTK_OBJECT(_RootWindow), "delete_event", 
	GTK_SIGNAL_FUNC(delete_event_cb), NULL
    );

    gtk_signal_connect(
	GTK_OBJECT(_ElementCTree), "tree_select_row",
	GTK_SIGNAL_FUNC(tree_select_row_cb), this
    );
}

static void menu_exit_cb(void)
{
    gtk_main_quit();
}

static void menu_about_cb(void)
{
    TKinokoShellAboutDialogGtk AboutDialog;

    AboutDialog.SetTitle("Kinoko Browser " KINOKO_SHELL_VERSION);
    AboutDialog.SetCopyright(KINOKO_SHELL_COPYING);
    AboutDialog.AddAuthor(KINOKO_SHELL_AUTHOR);
    AboutDialog.AddComment("A simple XML file viewer");

    AboutDialog.Open();
}

static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data)
{
    gtk_main_quit();
    return TRUE;
}

static void tree_select_row_cb(GtkCTree* Tree, GList* Node, gint column, gpointer Data)
{
    TKinokoBrowserGtk* Browser = (TKinokoBrowserGtk*) Data;
    TKinokoXmlElement* Element = Browser->_NodeElementTable[Node];

    if (Element) {
	gtk_clist_clear(GTK_CLIST(Browser->_AttributeCList));

	const TMushSaxAttributeList& AttributeList = Element->AttributeList();
	for (unsigned i = 0; i < AttributeList.getLength(); i++) {
	    //...
	    char** Buffer = new char*[2];
	    Buffer[0] = new char[AttributeList.getName(i).size() + 1];
	    Buffer[1] = new char[AttributeList.getValue(i).size() + 1];
	    strcpy(Buffer[0], AttributeList.getName(i).c_str());
	    strcpy(Buffer[1], AttributeList.getValue(i).c_str());
	    
	    gtk_clist_append(GTK_CLIST(Browser->_AttributeCList), Buffer);
	}
	gtk_widget_show_all(Browser->_AttributeCList);

	string Text = Element->Text();

	GtkTextBuffer* TextBuffer = gtk_text_view_get_buffer(
	    GTK_TEXT_VIEW(Browser->_ContentText)
	);
	gtk_text_buffer_set_text(TextBuffer, Text.data(), Text.size());
    }
}



int main (int argc, char *argv[])
{
    if (argc < 2) {
	cerr << "Usage: " << argv[0] << " file_name" << endl;
	return -1;
    }

    TKinokoBrowserGtk Browser(argc, argv);
    Browser.Start();

    return 0;
}
