// kinoterm.cc //


#include <iostream>
#include <cstring>
#include <cerrno>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <poll.h>

using namespace std;


const char* DeviceFile = (
// "/dev/ttyS0"    /* Serial Port #1 */
// "/dev/ttyS1"    /* Serial Port #2 */
   "/dev/ttyUSB0"  /* USB-Serial #1 */
);
const speed_t BaudRate = (
// B1200     /* 1200 Baud */
// B2400     /* 2400 Baud */
// B4800     /* 4800 Baud */
   B9600     /* 9600 Baud */
// B19200    /* 19200 Baud */
// B38400    /* 38400 Baud; may be altered with setserial(8) */
// B57600    /* 57600 Baud */
);
const tcflag_t InputFlags = (0
// | ICRNL   /* Replace CR with NL on input*/
   | IGNCR   /* Ignore CR on input*/
// | IXON    /* Enable XON/XOFF on output */
// | IXOFF   /* Enable XON/XOFF on input */
);
const tcflag_t OutputFlags = (0
);
const tcflag_t ControlFlags = (0
   | CS8     /* 8bit */
// | CS7     /* 7bit */
// | PARENB  /* Enable parity */
// | PARODD  /* Odd parity, rather than Even */
// | CSTOPB  /* Two stop bits, rather than one */
   | CREAD   /* Enable receiver */
);
const tcflag_t LocalFlags = (0
// | ECHO    /* Echo input characters */
);


static struct termios _OldTerminal, _Terminal;
static int _Serial;


int Initialize(void) 
{
    _Serial = open(DeviceFile, O_RDWR | O_NDELAY);
    if (_Serial < 0) {
	cerr << "ERROR: unable to open " << DeviceFile << ": ";
	cerr << strerror(errno) << endl;
	return -1;
    }

    if (tcgetattr(_Serial, &_OldTerminal) < 0) {
	cerr << "ERROR: unable to get terminal attributes: ";
	cerr << strerror(errno) << endl;
	return -1;
    }
    
    memset(&_Terminal, 0, sizeof(_Terminal));

    _Terminal.c_iflag = InputFlags;
    _Terminal.c_oflag = OutputFlags;
    _Terminal.c_cflag = BaudRate | ControlFlags;
    _Terminal.c_lflag = LocalFlags;
    
    _Terminal.c_cc[VMIN] = 1;
    _Terminal.c_cc[VTIME] = 0;

    tcflush(_Serial, TCIFLUSH);

    if (tcsetattr(_Serial, TCSANOW, &_Terminal) < 0) {
	cerr << "ERROR: unable to set terminal attributes: ";
	cerr << strerror(errno) << endl;
	return -1;
    }

    return _Serial;
}


void Finalize(void)
{
    tcsetattr(_Serial, TCSANOW, &_OldTerminal);
    close(_Serial);
}


int main(void)
{
    if (Initialize() < 0) {
	return -1;
    }

    pollfd PollFd[2];
    PollFd[0].fd = _Serial;
    PollFd[0].events = POLLIN;
    PollFd[1].fd = STDIN_FILENO;
    PollFd[1].events = POLLIN;

    static char Buffer[65536];
    char* Data;
    size_t DataSize, Size;
    bool IsStopRequested = false;

    while (! IsStopRequested) {
	// wait for inputs //
	if (poll(PollFd, 2, 1) <= 0) {
	    continue;
	}

	// process serial port input //
	if (PollFd[0].revents == POLLIN) {
	    DataSize = read(_Serial, Buffer, sizeof(Buffer)-1);
	    if (DataSize > 0) {
		Buffer[DataSize] = '\0';
		cout << Buffer << flush;
	    }
	    else {
		if (DataSize < 0) {
		    cerr << "ERROR: unable to read from serial port: ";
		    cerr << strerror(errno) << endl;
		}
		IsStopRequested = true;
	    }
	}
	if (PollFd[0].revents == POLLHUP) {
	    IsStopRequested = true;
	}

	// process console input //
	if (PollFd[1].revents == POLLIN) {
	    if (! cin.getline(Buffer, sizeof(Buffer)-1)) {
		Buffer[0] = '\0';
		IsStopRequested = true;
	    }
	    strcat(Buffer, "\x0d"); // append CR
	    Data = Buffer;
	    DataSize = strlen(Buffer);
	    while (DataSize > 0) {
		if ((Size = write(_Serial, Data, DataSize)) < 0) {
		    cerr << "ERROR: unable to write to serial port: ";
		    cerr << strerror(errno) << endl;
		    IsStopRequested = true;
		    break;
		}
		Data += Size;
		DataSize -= Size;
	    }
	}
	if (PollFd[1].revents == POLLHUP) {
	    IsStopRequested = true;
	}
    }

    Finalize();

    return 0;
}
