/* camdrv.c */
/* CAMAC device driver for CC/7700-PCI on Linux 2.0.x */
/* Created by Enomoto Sanshiro on 11 April 1999. */
/* Last updated by Enomoto Sanshiro on 14 April 1999. */


#define __KERNEL__
#define MODULE

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/config.h>
#include <linux/bios32.h>
#include <linux/pci.h>
#include <linux/errno.h>
#include <asm/io.h>
#include "camdrv.h"


#define CC7700_VENDOR_ID 0x0001
#define CC7700_DEVICE_ID 0xcc77

#ifndef CAMDRV_MAJOR
#  define CAMDRV_MAJOR 0
#endif

static int camdrv_major = CAMDRV_MAJOR;
static const char* camdrv_name = "camdrv";
static unsigned ioport;
static char irq;
static unsigned crate_addr = 0;
struct wait_queue* camdrv_wait_queue = NULL;
static int interrupt_count;
static int dev_id;

static int camdrv_open(struct inode* inode, struct file* filep);
static void camdrv_release(struct inode* inode, struct file* filep);
static int camdrv_read(struct inode* inode, struct file* filep, char* buf, int count);
static int camdrv_write(struct inode* inode, struct file* filep, const char* buf, int count);
static int camdrv_ioctl(struct inode *inode, struct file *filep, unsigned int cmd, unsigned long arg);
static void camdrv_interrupt(int irq, void* dev_id, struct pt_regs* regs);

static int initialize(void);
static int clear(void);
static int inhibit(void);
static int release_inhibit(void);
static int camac_action(unsigned naf, unsigned* data);
static int read_lam(unsigned* data);
static int enable_interrupt(void);
static int disable_interrupt(void);
static int wait_lam(unsigned time_out, unsigned* data);
static int detect_pci_device(unsigned vendor, unsigned device, unsigned* ioport, unsigned char* irq);


static struct file_operations camdrv_fops = {
    NULL,              /* camdrv_lseek */
    camdrv_read,       /* camdrv_read */
    camdrv_write,      /* camdrv_write */
    NULL,              /* camdrv_readdir */
    NULL,              /* camdrv_select */
    camdrv_ioctl,      /* camdrv_ioctl */
    NULL,              /* camdrv_mmap */
    camdrv_open,       /* camdrv_open */
    camdrv_release,    /* camdrv_close */
};

int init_module(void)
{
    int result;

    result = detect_pci_device(CC7700_VENDOR_ID, CC7700_DEVICE_ID, &ioport, &irq);
    if (result) {
        printk(KERN_WARNING "%s: unable to find CC/7700-PCI.\n", camdrv_name);
	return result;
    }
    printk(KERN_INFO "%s: CC/7700-PCI was detected at ioport 0x%04x on irq %d.\n", camdrv_name, ioport, irq);

    result = register_chrdev(camdrv_major, camdrv_name, &camdrv_fops);
    if (result < 0) {
        printk(KERN_WARNING "%s: can't get major %d\n", camdrv_name, camdrv_major);
        return result;
    }
    if (camdrv_major == 0) {
        camdrv_major = result;
    }

    printk(KERN_INFO "%s: at 0x%04x on irq %d (major = %d).\n", camdrv_name, ioport, irq, camdrv_major);
    return 0;
}

void cleanup_module(void)
{
    unregister_chrdev(camdrv_major, camdrv_name);
    printk(KERN_INFO "%s: removed.\n", camdrv_name);
}

static int camdrv_open(struct inode* inode, struct file* filep)
{
    int result;

    if (MOD_IN_USE) {
        return -EBUSY;
    }

    result = request_irq(irq, camdrv_interrupt, SA_INTERRUPT | SA_SHIRQ, camdrv_name, &dev_id);
    if (result < 0) {
        printk(KERN_WARNING "%s: unable to request IRQ.\n", camdrv_name);
        return result;
    }

    MOD_INC_USE_COUNT;

    return 0;
}

static void camdrv_release(struct inode* inode, struct file* filep)
{
    disable_interrupt();
    free_irq(irq, &dev_id);

    MOD_DEC_USE_COUNT;
}

static int camdrv_read(struct inode* inode, struct file* filep, char* buf, int count)
{
    return -ENODEV;
}

static int camdrv_write(struct inode* inode, struct file* filep, const char* buf, int count)
{
    return -ENODEV;
}

static int camdrv_ioctl(struct inode* inode, struct file *filep, unsigned int cmd, unsigned long arg)
{
    unsigned naf = 0, data = 0, timeout = 0;
    unsigned *user_naf_ptr, *user_data_ptr;
    int result = -EINVAL;

    user_naf_ptr = (unsigned *) arg;
    user_data_ptr = (unsigned *) arg + 1;

    if (_IOC_TYPE(cmd) != CAMDRV_IOC_MAGIC) {
        return -EINVAL;
    }

    if (_IOC_DIR(cmd) & (_IOC_READ | _IOC_WRITE)) {
        result = verify_area(VERIFY_WRITE, (void *) arg, 2);
	if (result) {
	    return result;
	}
	naf = get_user(user_naf_ptr);
	data = get_user(user_data_ptr);
	timeout = naf;
    }

    switch (cmd) {
      case CAMDRV_IOC_INITIALIZE:
        result = initialize();
        break;
      case CAMDRV_IOC_CLEAR:
        result = clear();
        break;
      case CAMDRV_IOC_INHIBIT:
        result = inhibit();
        break;
      case CAMDRV_IOC_RELEASE_INHIBIT:
        result = release_inhibit();
        break;
      case CAMDRV_IOC_ENABLE_INTERRUPT:
        result = enable_interrupt();
        break;
      case CAMDRV_IOC_DISABLE_INTERRUPT:
        result = disable_interrupt();
        break;
      case CAMDRV_IOC_CAMAC_ACTION:
        result = camac_action(naf, &data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_READ_LAM:
        result = read_lam(&data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_WAIT_LAM:
        result = wait_lam(timeout, &data);
        put_user(data, user_data_ptr);
        break;
      default:
	result = -EINVAL;
    }

    return result;
}

static void camdrv_interrupt(int irq, void* dev_id, struct pt_regs* regs)
{
    unsigned lam_pattern;
    read_lam(&lam_pattern);

    if (lam_pattern == 0) {
        return;
    }

    disable_interrupt();
    interrupt_count++;
    wake_up_interruptible(&camdrv_wait_queue);
}



/* CAMAC service functions */

enum cc7700_regs {
    regCONTROL = 0x00,
    regSTATUS = 0x00,
    regCOMMAND = 0x04,
    regLAM = 0x04,
    regDATA = 0x08,
    regGO = 0x0C
};

enum cc7700_ctrlbits {
    ctrlCLEAR = 0x01,
    ctrlINITIALIZE = 0x02,
    ctrlINHIBIT = 0x04,
    ctrlENABLEINTERRUPT = 0x08,
    ctrlRESET = 0x40,
    ctrlLAMINTERNAL = 0x80
};

enum cc7700_statbits {
    statNOQ = 0x01,
    statNOX = 0x02,
    statINHIBIT = 0x04,
    statENABLEINTERRUPT = 0x08,
    statDONE = 0x10,
    statONLINE = 0x20,
    statLAMSUM = 0x40,
    statLAMINTERNAL = 0x80
};


#define CONTROL_KEPT_BIT_MASK (0x04 | 0x08 | 0x80)
#define STATUS_BIT_MASK 0x00ff

static int initialize(void)
{
    outl(ctrlINITIALIZE, ioport + regCONTROL);

    return 0;
}

static int clear(void)
{
    unsigned reg;

    reg = inl(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlCLEAR;
    
    outl(reg, ioport + regCONTROL);

    return 0;
}

static int inhibit(void)
{
    unsigned reg;

    reg = inl(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlINHIBIT;
    
    outl(reg, ioport + regCONTROL);

    return 0;
}

static int release_inhibit(void)
{
    unsigned reg;

    reg = inl(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg &= ~ctrlINHIBIT;
    
    outl(reg, ioport + regCONTROL);

    return 0;
}

static int enable_interrupt(void)
{
    unsigned reg;

    interrupt_count = 0;
    reg = inl(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg |= ctrlENABLEINTERRUPT;
    
    outl(reg, ioport + regCONTROL);

    return 0;
}

static int disable_interrupt(void)
{
    unsigned reg;

    reg = inl(ioport + regCONTROL) & CONTROL_KEPT_BIT_MASK;
    reg &= ~ctrlENABLEINTERRUPT;
    
    outl(reg, ioport + regCONTROL);

    return 0;
}

static int camac_action(unsigned naf, unsigned* data)
{
    unsigned command;
    unsigned data_mask = 0x00ffffff;
    unsigned long start_jiffies;
    int status;
    
    /* Hardware Problem of CC/7700 ??? */
    /* subtract 1 from station number */
    naf -= (1 << 9);
    
    command = (crate_addr << 16) | naf;
    outl(command, ioport + regCOMMAND);
    outl(*data, ioport + regDATA);

    outl(0, ioport + regGO);

    start_jiffies = jiffies;
    while (! ((status = inl(ioport + regSTATUS)) & statDONE)) {
        schedule();
	if (jiffies - start_jiffies > HZ) {
	    *data = 0;
	    return -ETIMEDOUT;
	}
    }

    *data = inl(ioport + regDATA) & data_mask;

    return status & STATUS_BIT_MASK;
}

static int read_lam(unsigned* data)
{
    unsigned data_mask = 0x00ffffff;

    *data = inl(ioport + regLAM) & data_mask;
    return *data;
}

static int wait_lam(unsigned timeout, unsigned* data)
{
    unsigned long flags;
    timeout *= HZ;

    save_flags(flags);
    cli();
    if (interrupt_count == 0) {
        current->timeout = jiffies + timeout;
	interruptible_sleep_on(&camdrv_wait_queue);
	restore_flags(flags);

	if (interrupt_count == 0) {
            return -ETIMEDOUT;
	}
    }
    else {
        disable_interrupt();
        restore_flags(flags);
    }

    interrupt_count = 0;

    return read_lam(data);
}

#ifndef CONFIG_PCI
#error "Kernel doesn't support PCI BIOS."
#endif

static int detect_pci_device(unsigned vendor, unsigned device, unsigned* ioport, unsigned char* irq)
{
    int index, result;
    unsigned address_mask;
    unsigned char bus, function;
    unsigned long flags;

    if (! pcibios_present()) {
        printk(KERN_WARNING "%s: unable to find PCI bios.\n", camdrv_name);
        return -ENODEV;
    }

    index = 0;
    result = pcibios_find_device(vendor, device, index, &bus, &function);
    if (result != PCIBIOS_SUCCESSFUL) {
        if (result == PCIBIOS_DEVICE_NOT_FOUND) {
            return -ENODEV;
	}
	else {
            printk(KERN_WARNING "%s: %s\n", camdrv_name, pcibios_strerror(result));
	    return -EIO;
	}
    }

    pcibios_read_config_dword(bus, function, PCI_BASE_ADDRESS_0, ioport);
    save_flags(flags);
    cli();
    pcibios_write_config_dword(bus, function, PCI_BASE_ADDRESS_0, ~0);
    pcibios_read_config_dword(bus, function, PCI_BASE_ADDRESS_0, &address_mask);
    pcibios_write_config_dword(bus, function, PCI_BASE_ADDRESS_0, *ioport);
    restore_flags(flags);
    *ioport &= (address_mask & PCI_BASE_ADDRESS_IO_MASK);

    pcibios_read_config_byte(bus, function, PCI_INTERRUPT_LINE, irq);

    return result;
}
