/* camdrv.c */
/* CAMAC device driver for CC/7000-ISA on Linux 2.6.x */
/* Created by Enomoto Sanshiro on 11 April 1999. */
/* Last updated by Enomoto Sanshiro on 27 October 2005. */


#if defined(USE_MODVERSIONS) && USE_MODVERSIONS
#  define MODVERSIONS
#  include <config/modversions.h>
#endif


#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/poll.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include "camdrv.h"


MODULE_LICENSE("GPL");
MODULE_AUTHOR("Enomoto Sanshiro");
MODULE_DESCRIPTION("Camac Driver for TOYO-Corp. CC/7000-ISA Controller");


#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 18))
#define USE_OLD_IRQ 1
#endif


#ifndef CAMDRV_MAJOR
#  define CAMDRV_MAJOR 0
#endif
static int camdrv_major = CAMDRV_MAJOR;
module_param(camdrv_major, int, S_IRUGO);

static const char* camdrv_name = "camdrv";
static unsigned long ioport = 0;
static int irq = 0;
module_param(ioport, ulong, S_IRUGO);
module_param(irq, int, S_IRUGO);

static int open_count;
static unsigned crate_address = 0;
static wait_queue_head_t camdrv_wait_queue;
static int interrupt_count;
static int dev_id;

static int camdrv_open(struct inode* inode, struct file* filep);
static int camdrv_release(struct inode* inode, struct file* filep);
static int camdrv_ioctl(struct inode *inode, struct file *filep, unsigned int cmd, unsigned long arg);
static unsigned int camdrv_poll(struct file* filep, poll_table* table);

#ifdef USE_OLD_IRQ
static irqreturn_t camdrv_interrupt(int irq, void* dev_id, struct pt_regs* regs);
#else
static irqreturn_t camdrv_interrupt(int irq, void* dev_id);
#endif

static int set_crate(void);
static int initialize(void);
static int clear(void);
static int inhibit(void);
static int release_inhibit(void);
static int camac_action(unsigned naf, unsigned* data);
static int read_lam(unsigned* data);
static int enable_interrupt(void);
static int disable_interrupt(void);
static int wait_lam(unsigned time_out, unsigned* data);


static struct file_operations camdrv_fops = {
    .open = camdrv_open,
    .release = camdrv_release,
    .ioctl = camdrv_ioctl,
    .poll = camdrv_poll,
};


int __init init_module(void)
{
    int result;

    if (ioport == 0) {
        printk(KERN_WARNING "%s: I/O port address (ioport) is not specified.\n", camdrv_name);
        return -EINVAL;
    }
    if (irq == 0) {
        printk(KERN_WARNING "%s: IRQ number (irq) is not specified.\n", camdrv_name);
        return -EINVAL;
    }

    result = register_chrdev(camdrv_major, camdrv_name, &camdrv_fops);
    if (result < 0) {
        printk(KERN_WARNING "%s: unable to get major %d\n", camdrv_name, camdrv_major);
        return result;
    }
    if (camdrv_major == 0) {
        camdrv_major = result;
    }
    init_waitqueue_head(&camdrv_wait_queue);

    printk(KERN_INFO "%s: at 0x%04lx on irq %d (major = %d).\n", camdrv_name, ioport, irq, camdrv_major);

    open_count = 0;

    return 0;
}

void __exit cleanup_module(void)
{
    unregister_chrdev(camdrv_major, camdrv_name);
    printk(KERN_INFO "%s: removed.\n", camdrv_name);
}

static int camdrv_open(struct inode* inode, struct file* filep)
{
    int result;

    if (open_count) {
        return -EBUSY;
    }

#ifdef USE_OLD_IRQ
    result = request_irq(
	irq, camdrv_interrupt, 
	SA_INTERRUPT | SA_SHIRQ, camdrv_name, &dev_id
    );
#else
    result = request_irq(
	irq, camdrv_interrupt, 
	IRQF_DISABLED | IRQF_SHARED, camdrv_name, &dev_id
    );
#endif
    if (result < 0) {
        printk(KERN_WARNING "%s: unable to request IRQ.\n", camdrv_name);
        return result;
    }

    try_module_get(THIS_MODULE);
    open_count++;

    return 0;
}

static int camdrv_release(struct inode* inode, struct file* filep)
{
    disable_interrupt();
    free_irq(irq, &dev_id);

    module_put(THIS_MODULE);
    open_count--;

    return 0;
}

static int camdrv_ioctl(struct inode* inode, struct file *filep, unsigned int cmd, unsigned long arg)
{
    unsigned parameter = 0, data = 0;
    unsigned *user_parameter_ptr, *user_data_ptr;
    int result = -EINVAL;

    user_parameter_ptr = (unsigned *) arg;
    user_data_ptr = (unsigned *) arg + 1;

    if (_IOC_TYPE(cmd) != CAMDRV_IOC_MAGIC) {
        return -EINVAL;
    }

    if (_IOC_DIR(cmd) & (_IOC_READ | _IOC_WRITE)) {
	if (get_user(parameter, user_parameter_ptr) < 0) {
	    return -EFAULT;
	}
	if (get_user(data, user_data_ptr) < 0) {
	    return -EFAULT;
	}
    }

    switch (cmd) {
      case CAMDRV_IOC_INITIALIZE:
        result = initialize();
        break;
      case CAMDRV_IOC_CLEAR:
        result = clear();
        break;
      case CAMDRV_IOC_INHIBIT:
        result = inhibit();
        break;
      case CAMDRV_IOC_RELEASE_INHIBIT:
        result = release_inhibit();
        break;
      case CAMDRV_IOC_ENABLE_INTERRUPT:
        result = enable_interrupt();
        break;
      case CAMDRV_IOC_DISABLE_INTERRUPT:
        result = disable_interrupt();
        break;
      case CAMDRV_IOC_CAMAC_ACTION:
        result = camac_action(parameter, &data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_READ_LAM:
        result = read_lam(&data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_WAIT_LAM:
        result = wait_lam(parameter, &data);
        put_user(data, user_data_ptr);
        break;
      case CAMDRV_IOC_SET_CRATE:
        crate_address = parameter;
        result = set_crate();
        break;
      default:
	result = -EINVAL;
    }

    return result;
}

static unsigned int camdrv_poll(struct file* filep, poll_table* table)
{
    unsigned int mask = 0;

    poll_wait(filep, &camdrv_wait_queue, table);

    if (interrupt_count > 0) {
	mask |= POLLIN | POLLRDNORM;
    }

    return mask;
}

#ifdef USE_OLD_IRQ
static irqreturn_t camdrv_interrupt(int irq, void* dev_id, struct pt_regs* regs)
#else
static irqreturn_t camdrv_interrupt(int irq, void* dev_id)
#endif
{
    unsigned lam_pattern;
    read_lam(&lam_pattern);

    if (lam_pattern == 0) {
        return IRQ_NONE;
    }

    disable_interrupt();
    interrupt_count++;
    wake_up_interruptible(&camdrv_wait_queue);

    return IRQ_HANDLED;
}


/* CAMAC service functions */

enum cc7000_regs {
    regCONTROL = 0x06,
    regSTATUS = 0x06,
    regCRATE_ADDRESS = 0x01,
    regCOMMAND_NA = 0x04,
    regLAM_LOW = 0x04,
    regLAM_HIGH = 0x06,
    regDATA_LOW = 0x00,
    regDATA_HIGH = 0x02,
    regGO = 0x07
};

enum cc7000_ctrlbits {
    ctrlCLEAR = 0x0100,
    ctrlINITIALIZE = 0x0200,
    ctrlINHIBIT = 0x0400,
    ctrlENABLEINTERRUPT = 0x0800,
    ctrlLAMINTERNAL = 0x8000
};

enum cc7000_statbits {
    statQ = 0x0100,
    statX = 0x0200,
    statINHIBIT = 0x0400,
    statENABLEINTERRUPT = 0x0800,
    statDONE = 0x1000,
    statONLINE = 0x2000,
    statLAMSUM = 0x4000,
    statLAMINTERNAL = 0x8000
};

#define CONTROL_UNTOUCHED_BIT_MASK (0x0400 | 0x0800 | 0x8000)
#define DATA_HIGH_MASK 0x00ff
#define LAM_HIGH_MASK 0x00ff

static int set_crate(void)
{
    outb_p(crate_address, ioport + regCRATE_ADDRESS);
    return 0;
}

static int initialize(void)
{
    outb_p(crate_address, ioport + regCRATE_ADDRESS);
    wmb();
    outw_p(ctrlINITIALIZE, ioport + regCONTROL);

    return 0;
}

static int clear(void)
{
    unsigned short reg;

    reg = inw_p(ioport + regCONTROL) & CONTROL_UNTOUCHED_BIT_MASK;
    reg |= ctrlCLEAR;
    
    outw_p(reg, ioport + regCONTROL);

    return 0;
}

static int inhibit(void)
{
    unsigned short reg;

    reg = inw_p(ioport + regCONTROL) & CONTROL_UNTOUCHED_BIT_MASK;
    reg |= ctrlINHIBIT;
    
    outw_p(reg, ioport + regCONTROL);

    return 0;
}

static int release_inhibit(void)
{
    unsigned short reg;

    reg = inw_p(ioport + regCONTROL) & CONTROL_UNTOUCHED_BIT_MASK;
    reg &= ~ctrlINHIBIT;
    
    outw_p(reg, ioport + regCONTROL);

    return 0;
}

static int enable_interrupt(void)
{
    unsigned short reg;

    interrupt_count = 0;
    reg = inw_p(ioport + regCONTROL) & CONTROL_UNTOUCHED_BIT_MASK;
    reg |= ctrlENABLEINTERRUPT;
    
    outw_p(reg, ioport + regCONTROL);

    return 0;
}

static int disable_interrupt(void)
{
    unsigned short reg;

    reg = inw_p(ioport + regCONTROL) & CONTROL_UNTOUCHED_BIT_MASK;
    reg &= ~ctrlENABLEINTERRUPT;
    
    outw_p(reg, ioport + regCONTROL);

    return 0;
}

static int camac_action(unsigned naf, unsigned* data)
{
    unsigned long start_jiffies;
    int status;
    unsigned short control_reg;

    unsigned short n = (naf >> 9) & 0x1f;
    unsigned short a = (naf >> 5) & 0x0f;
    unsigned short f = (naf >> 0) & 0x1f;
    unsigned short na = (a << 8) | n;

    unsigned short data_l = (*data >> 0) & 0xffff;
    unsigned short data_h = (*data >> 16) & 0x00ff;

    outw_p(na, ioport + regCOMMAND_NA);
    control_reg = inw_p(ioport + regCONTROL) & CONTROL_UNTOUCHED_BIT_MASK;
    control_reg |= f;
    outw_p(control_reg, ioport + regCONTROL);

    outw_p(data_l, ioport + regDATA_LOW);
    outw_p(data_h, ioport + regDATA_HIGH);

    wmb();
    outb_p(0, ioport + regGO);

    start_jiffies = jiffies;
    while (! ((status = inw_p(ioport + regSTATUS)) & statDONE)) {
        schedule();
	if (jiffies - start_jiffies > HZ) {
	    *data = 0;
	    return -ETIMEDOUT;
	}
    }

    data_l = inw_p(ioport + regDATA_LOW);
    data_h = inw_p(ioport + regDATA_HIGH) & DATA_HIGH_MASK;

    *data = (data_h << 16) | data_l;

    return (status >> 8);
}

static int read_lam(unsigned* data)
{
    unsigned short lam_l, lam_h;

    lam_l = inw_p(ioport + regLAM_LOW);
    lam_h = inw_p(ioport + regLAM_HIGH) & LAM_HIGH_MASK;

    *data = (lam_h << 16) | lam_l;

    return *data;
}

static int wait_lam(unsigned timeout, unsigned* data)
{
    long remaining_time;
    timeout *= HZ;

    remaining_time = wait_event_interruptible_timeout(
        camdrv_wait_queue, 
	(interrupt_count > 0),
	timeout
    );
    if (remaining_time == 0) {
        return -ETIMEDOUT;
    }
    if (interrupt_count == 0) {
        return -ERESTARTSYS;
    }

    interrupt_count = 0;

    return read_lam(data);
}
