/* MoguraEvnetPiece.cc */
/* Created by Enomoto Sanshiro on 13 August 2009. */
/* Last updated by Enomoto Sanshiro on 11 March 2010. */

#include <iostream>
#include <map>
#include "MoguraEventPiece.hh"

using namespace std;

//#define DEBUG(x) x
#define DEBUG(x)


enum TFrameHeaderBits {
    shiftLaunchOffset = 0,
    maskLaunchOffset = 0x07fff,
    bitLaunchOffsetSign = 0x0001 << 15,
    bitBL = 0x0001 << 16,
    bitBM = 0x0001 << 17,
    bitBH = 0x0001 << 18,
    bitBP = 0x0001 << 19,
    bitAL = 0x0001 << 20,
    bitAM = 0x0001 << 21,
    bitAH = 0x0001 << 22,
    bitAP = 0x0001 << 23,
    bitDataType = 0x0001 << 24,
    bitHitA = 0x0001 << 25,
    bitHitB = 0x0001 << 26,
    maskDataFlag = 0x00ff0000,
    maskHitFlag = 0x06000000,
    shiftFrameFormatType = 27,
    maskFrameFormatType = 0x03,
    shiftFefNumber = 29,
    maskFefNumber = 0x07,
    _NumberOfFrameHeaderBits
};

enum TFrameTrailerBits {
    shiftEventId = 0,
    maskEventId = 0x00ff,
    shiftCompletionFlag = 8,
    maskCompletionFlag = 0x00ff,
    shiftTimestamp = 16,
    maskTimestamp = 0x00ffff,
    _NumberOfFrameTrailerBits
};


static const int NumberOfDataBlocks = 8;
static const char DataGainNameList[] = {
    'P', 'H', 'M', 'L', 'P', 'H', 'M', 'L'
};
static const char DataChannelNameList[] = {
    'A', 'A', 'A', 'A', 'B', 'B', 'B', 'B', 
};
static const int DataChannelList[] = {
    0, 0, 0, 0, 1, 1, 1, 1
};
static const int DataWordLengthList[] = {
    10, 2, 2, 2, 10, 2, 2, 2
};
static const int SamplesPerTickList[] = {
    20, 4, 4, 4, 20, 4, 4, 4
};
static const int DataFlagBitList[] = {
    Bit(23), Bit(22), Bit(21), Bit(20), Bit(19), Bit(18), Bit(17), Bit(16)
};

static const int MoGURA_NumberOfFefs = 6;
static const int MoGURA_MaximumFrameTrainLength = 128;
static const int MoGURA_MinimumFrameSize = 4*3;
static const int MoGURA_MaximumFrameSize = 4*35;



TMoguraEventPieceScanner::TMoguraEventPieceScanner(void)
{
}

TMoguraEventPieceScanner::~TMoguraEventPieceScanner()
{
}

int TMoguraEventPieceScanner::MaximumPieceSize(void)
{
    return MoGURA_MaximumFrameTrainLength * MoGURA_MaximumFrameSize + 4;
}

int TMoguraEventPieceScanner::Scan(TStreamKey StreamKey, TKinokoEventPiece& EventPiece)
{
    int CardNumber = StreamKey & 0xffff;

    if (EventPiece[0] == 0) {
	return ScanTestPattern(CardNumber, EventPiece);
    }

    // continued-corrupted before test pattern //
    if (_LastPieceTypeTable[CardNumber] == PieceType_Corrupted) {
	return ScanCorrupted(CardNumber, EventPiece);
    }

    EventPiece.PieceType = PieceType_Normal;
    _LastPieceTypeTable[CardNumber] = EventPiece.PieceType;

    int NumberOfFrames = (EventPiece[0] >> 16) & 0xffff;
    int FrameTrainSize = ((EventPiece[0] >> 0) & 0xffff) * 4;
    int PieceSize = FrameTrainSize + 4;
    DEBUG(cout << "CARD: " << CardNumber << endl);
    DEBUG(cout << "SIZE: ");
    DEBUG(cout << NumberOfFrames << " frames, ");
    DEBUG(cout << FrameTrainSize << " bytes" << endl);

    if (NumberOfFrames < 1) {
	DEBUG(cout << "FRAME CORRUPTION: incorrect number of frames" << endl);
	return ScanCorrupted(CardNumber, EventPiece);
    }
    if (NumberOfFrames > MoGURA_MaximumFrameTrainLength) {
	DEBUG(cout << "FRAME CORRUPTION: too large number of frames" << endl);
	return ScanCorrupted(CardNumber, EventPiece);
    }
    if (FrameTrainSize < MoGURA_MinimumFrameSize * NumberOfFrames) {
	DEBUG(cout << "FRAME CORRUPTION: too small frame train size" << endl);
	return ScanCorrupted(CardNumber, EventPiece);
    }
    if (FrameTrainSize > MoGURA_MaximumFrameSize * NumberOfFrames) {
	DEBUG(cout << "FRAME CORRUPTION: too large frame train size" << endl);
	return ScanCorrupted(CardNumber, EventPiece);
    }
    if (EventPiece.Size() < PieceSize) {
	return 0;
    }

    return ScanFirstDataFrame(CardNumber, EventPiece);
}

int TMoguraEventPieceScanner::ScanFirstDataFrame(int CardNumber, TKinokoEventPiece& EventPiece)
{
    int Index = 0;

    // Frame Train Header //
    DEBUG(int NumberOfFrames = (EventPiece[0] >> 16) & 0xffff);
    int FrameTrainSize = ((EventPiece[0] >> 0) & 0xffff) * 4;
    int PieceSize = FrameTrainSize + 4;
    DEBUG(cout << "=== FRAME TRAIN HEADER === ");
    DEBUG(cout << NumberOfFrames << " frames, ");
    DEBUG(cout << 4*FrameTrainSize << " bytes" << endl);
    Index++;

    // Frame Header 1 //
    DEBUG(int LaunchOffset = (
	(EventPiece[Index] >> shiftLaunchOffset) & maskLaunchOffset
    ));
    DEBUG(if (EventPiece[Index] & bitLaunchOffsetSign) {
	LaunchOffset *= -1;
    })
    int FefNumber = (
	(EventPiece[Index] >> shiftFefNumber) & maskFefNumber
    );
    int FrameFormatType = (
	(EventPiece[Index] >> shiftFrameFormatType) & maskFrameFormatType
    );
    int DataType = (EventPiece[Index] & bitDataType) ? 1: 0;
    int DataFlag = EventPiece[Index] & maskDataFlag;
    DEBUG(int HitFlag = EventPiece[Index] & maskHitFlag);
    Index++;

    // Frame Header 2 //
    long long TimestampH;
    int EventCount;
    if (FrameFormatType == 0) {
	EventCount = EventPiece[Index];
	TimestampH = EventPiece[Index];
    }
    else if (FrameFormatType == 1) {
	EventCount = 0;
	TimestampH = EventPiece[Index];
    }
    else {
	DEBUG(cout << "FRAME CORRUPTION: unknown frame type" << endl);
	return ScanCorrupted(CardNumber, EventPiece);
    }
    Index++;

    DEBUG(cout << "== Frame Header == ");
    DEBUG(cout << "FEF: " << FefNumber);
    DEBUG(cout << ", ");
    DEBUG(cout << "Hit: ");
    DEBUG(cout << ((HitFlag & bitHitA) ? 'A' : '-'));
    DEBUG(cout << ((HitFlag & bitHitB) ? 'B' : '-'));
    DEBUG(cout << ", ");
    DEBUG(cout << "Data: ");
    DEBUG(cout << ((DataFlag & bitAP) ? 'P' : '-'));
    DEBUG(cout << ((DataFlag & bitAH) ? 'H' : '-'));
    DEBUG(cout << ((DataFlag & bitAM) ? 'M' : '-'));
    DEBUG(cout << ((DataFlag & bitAL) ? 'L' : '-'));
    DEBUG(cout << " ");
    DEBUG(cout << ((DataFlag & bitBP) ? 'P' : '-'));
    DEBUG(cout << ((DataFlag & bitBH) ? 'H' : '-'));
    DEBUG(cout << ((DataFlag & bitBM) ? 'M' : '-'));
    DEBUG(cout << ((DataFlag & bitBL) ? 'L' : '-'));
    DEBUG(cout << ", ");
    DEBUG(cout << "IsWaveform: " << DataType << ", ");
    DEBUG(cout << "FormatType: " << FrameFormatType << ", ");
    DEBUG(cout << "LaunchOffset: " << LaunchOffset << ", ");
    DEBUG(cout << "EventCounter: " << EventCount << ", ");
    DEBUG(cout << "TimestampH: " << TimestampH << endl);
    
    // Data Block //
    int DataBlockLength;
    if (DataType != 0) {
	// Waveform Data Block //
	DataBlockLength = 0;
	for (int i = 0; i < NumberOfDataBlocks; i++) {
	    if (DataFlag & DataFlagBitList[i]) {
		DataBlockLength += DataWordLengthList[i];
	    }
	}
    }
    else if (DataFlag != 0) {
	// TQ Data Block (FIFO Almost Full Mode) //
	DataBlockLength = 0;
	for (int i = 0; i < NumberOfDataBlocks; i++) {
	    if (DataFlag & DataFlagBitList[i]) {
		DataBlockLength += 2;
	    }
	}
    }
    else {
	// HIT Data Block (FIFO Full Mode) //
	DataBlockLength = 0;
    }
    Index += DataBlockLength;

    if (PieceSize < 4*Index) {
	DEBUG(cout << "FRAME CORRUPTION: inconsistent frame size" << endl);
	return ScanCorrupted(CardNumber, EventPiece);
    }
    
    // FrameTrailer //
    DEBUG(int EventId = (
	(EventPiece[Index] >> shiftEventId) & maskEventId
    ));
    long long TimestampL = (
	(EventPiece[Index] >> shiftTimestamp) & maskTimestamp
    );
    int CompletionFlag = (
	(EventPiece[Index] >> shiftCompletionFlag) & maskCompletionFlag
    );
    DEBUG(cout << "== Frame Trailer == ");
    DEBUG(cout << "EventId: " << EventId << ", ");
    DEBUG(cout << "TimestampL: " << TimestampL << ", ");
    DEBUG(cout << "TrailerFlag: 0x" << hex << CompletionFlag << endl);
    
    if ((CompletionFlag != 0xaa) && (CompletionFlag != 0x55)) {
	DEBUG(cout << "FRAME CORRUPTION: invalid completion flag" << endl);
	return ScanCorrupted(CardNumber, EventPiece);
    }

    long long Timestamp = ((TimestampH << 16) | TimestampL);
    DEBUG(cout << dec << FefNumber << " ");
    DEBUG(cout << hex << Timestamp);
    DEBUG(cout << dec << endl);

    EventPiece.ChannelKey = FefNumber + MoGURA_NumberOfFefs * CardNumber;
    EventPiece.EventKey = Timestamp;

    EventPiece.PieceFlag = 0;
    if ((DataType == 0) && (DataFlag == 0)) {
	// hit-missed packet //
	EventPiece.PieceFlag = (
	    PieceFlag_PossiblyEarlyArrival | PieceFlag_LastOfTheEvent
	);
    }

    return PieceSize;
}

int TMoguraEventPieceScanner::ScanTestPattern(int CardNumber, TKinokoEventPiece& EventPiece)
{
    EventPiece.PieceType = PieceType_Padding;
    _LastPieceTypeTable[CardNumber] = EventPiece.PieceType;

    int PieceSize = 0;
    for (unsigned Index = 0; Index < (unsigned) EventPiece.Size()/4; Index++) {
	if (EventPiece[Index] == 0xffffffff) {
	    // test pattern completed //
	    PieceSize = 4*(Index+1);
	    break;
	}

	if ((EventPiece[Index] & 0xffff) != Index) {
	    // test pattern corrupted //
	    return ScanCorrupted(CardNumber, EventPiece);
	}
    }

    return PieceSize;
}

int TMoguraEventPieceScanner::ScanCorrupted(int CardNumber, TKinokoEventPiece& EventPiece)
{
    EventPiece.PieceType = PieceType_Corrupted;
    _LastPieceTypeTable[CardNumber] = EventPiece.PieceType;

    // search for test pattern for recovery //

    // Note we start from Index==1 because (EventPiece[0] == 0) cases are
    // handled by ScanTestPattern() and this is to avoid infinite loop
    // by corrupted test pattern.

    int Index;
    for (Index = 1; Index < EventPiece.Size()/4; Index++) {
	if (EventPiece[Index] == 0) {
	    // test pattern (candidate) found //
	    break;
	}
    }
    // Keep going while Data == 0 to avoid zero-value-block break out //
    for (; Index < EventPiece.Size()/4 - 1; Index++) {
	if (EventPiece[Index+1] != 0) {
	    break;
	}
    }
    int PieceSize = 4*Index;
    
    return PieceSize;
}
