/* dmabench.cc */
/* Created by Enomoto Sanshiro on 10 October 1998. */
/* Last updated by Enomoto Sanshiro on 27 December 1998. */


#include <cstdio>
#include <iostream>
#include "RoomVmeAccess.hh"
#include "RoomDeviceFactory.hh"
#include "MushMisc.hh"

using namespace std;


static const off_t DmaOffset = 0x000000;
static const size_t DmaBufferSize = 0x0ffffc;
static const int PacketSize = 10 * 1024;
static const int NumberOfPackets = DmaBufferSize / PacketSize;


class TDmaBenchmarker {
  public:
    TDmaBenchmarker(TRoomVmeModule* Module, off_t DmaOffset, size_t BufferSize);
    virtual ~TDmaBenchmarker();
    virtual void Run(int PacketSize, int NumberOfPackets) throw(THardwareException);
  protected:
    TRoomVmeModule* _Module;
    off_t _DmaOffset;
    size_t _BufferSize;
    char* _Buffer;
};


TDmaBenchmarker::TDmaBenchmarker(TRoomVmeModule* Module, off_t DmaOffset, size_t BufferSize)
{
    _Module = Module;
    _DmaOffset = DmaOffset;
    _BufferSize = BufferSize;
    _Buffer = new char[_BufferSize];
}

TDmaBenchmarker::~TDmaBenchmarker()
{
    delete[] _Buffer;
}

void TDmaBenchmarker::Run(int PacketSize, int NumberOfPackets) throw(THardwareException)
{
    TMushDateTime StartTime, StopTime;
    
    for (int i = 1; i < NumberOfPackets; i++) {
	size_t DataSize = PacketSize * i;

	StartTime.Update();
#ifdef DMA
	_Module->DmaRead(_DmaOffset, _Buffer, DataSize);
#else
	_Module->PioRead(_DmaOffset, _Buffer, DataSize);
#endif
	StopTime.Update();

	double Time = (
            (StopTime.AsLong() - StartTime.AsLong()) + 
            (StopTime.USec() - StartTime.USec()) * 1e-6
        );
	double Speed = DataSize / Time;
	
	cout << (DataSize / 1024) << "\t";
	cout << (Time * 1.0e+3) << "\t";
        cout << (Speed / 1024.0 / 1024.0)  << "\t";
	cout << endl; 
    }
}


int main(int argc, char** argv)
{
    int ModuleAddress;
    if (
	(argc < 2) ||
	(sscanf(argv[1], "%x", &ModuleAddress) != 1)
    ){
	cerr << "Usage: " << argv[0] << " ADDRESS" << endl;
	return -1;
    }

    TRoomVmeCrate* VmeCrate = 0;
    TRoomVmeController* VmeController = 0;
    TRoomVmeModule* VmeMemoryModule = 0;
    try{
        TRoomDeviceFactory* DeviceFactory = TRoomDeviceFactory::GetInstance();
	VmeCrate = DeviceFactory->CreateVmeCrate(); 
	VmeController = DeviceFactory->CreateVmeController("SBS-620");
	VmeMemoryModule = DeviceFactory->CreateVmeModule("Generic-Memory");
	
	VmeCrate->InstallController(VmeController);
	VmeCrate->Install(VmeMemoryModule, ModuleAddress);

	TDmaBenchmarker Benchmarker(VmeMemoryModule, DmaOffset, DmaBufferSize);
	Benchmarker.Run(PacketSize, NumberOfPackets);
    }
    catch (THardwareException &e) {
	cerr << "ERROR: " << e << endl;
    }

    delete VmeMemoryModule;
    delete VmeController;
    delete VmeCrate;

    return 0;
}
