/* KamePlainTextRepository.hh */
/* Created by Enomoto Sanshiro on 1 July 2009. */
/* Last updated by Enomoto Sanshiro on 1 July 2009. */


#ifndef __KamePlainTextRepository_hh__
#define __KamePlainTextRepository_hh__


#include <string>
#include <map>
#include <cstdio>
#include "KameNtuple.hh"
#include "KameRepository.hh"


namespace kame {


class TKamePlainTextFieldParser;
class TKamePlainTextHeaderProcessor;

    
class TKamePlainTextRepository: public TKameRepository {
  public:
    TKamePlainTextRepository(void);
    TKamePlainTextRepository(const std::string& RepositoryName);
    virtual ~TKamePlainTextRepository();
    virtual void EnableCompression(void);
    virtual void SetDelimiter(const std::string& Delimiter);
    virtual void SetQuote(const std::string& Quote);
    virtual void SetHeaderDelimiter(const std::string& Delimiter);
    virtual void SetHeaderQuote(const std::string& Quote);
    virtual void SetLineTerminator(char LineTerminator);
    virtual void SetHeaderProcessor(TKamePlainTextHeaderProcessor* HeaderProcessor);
  public:
    static char UnixLineTerminator(void) { return '\x0a'; }
    static char MacLineTerminator(void) { return '\x0d'; }
    static char DosLineTerminator(void) { return '\x0a'; }
  public:
    virtual void SaveNtuple(const TKameNtuple& Ntuple, const std::string& Name) throw(TKameException);
    virtual void LoadNtuple(TKameNtuple& Ntuple, const std::string& Name, int Revision=-1) throw(TKameException);
  protected:
    virtual std::string FileNameOf(const std::string& ObjectName, int Revision = -1);
    virtual std::istream* OpenInputFile(const std::string& ObjectName, int Revision = -1) throw(TKameException);
    virtual std::ostream* OpenOutputFile(const std::string& ObjectName, int Revision = -1) throw(TKameException);
    virtual void CloseInputFile(std::istream* File) throw(TKameException);
    virtual void CloseOutputFile(std::ostream* File) throw(TKameException);
  protected:
    std::string _RepositoryName;
    bool _IsCompressionEnabled;
    std::string _Delimiter, _Quote;
    std::string _HeaderDelimiter, _HeaderQuote;
    char _LineTerminator;
    bool _IsRepositoryCreated;
    std::map<std::string, int> _RevisionTable;
    std::map<std::ios*, std::FILE*> _CommandPipeTable;
    std::map<std::ios*, std::ios*> _MyFileTable;
    TKamePlainTextHeaderProcessor* _HeaderProcessor;
    TKamePlainTextHeaderProcessor* _MyHeaderProcessor;
};


class TKamePlainTextLineTokenizer {
  public:
    TKamePlainTextLineTokenizer(void);
    virtual ~TKamePlainTextLineTokenizer();
    virtual void SetDelimiter(const std::string& Delimiter);
    virtual void SetQuote(const std::string& Quote);
    virtual void TokenizeLine(const std::string& Line, std::vector<std::string>& ElementList) const throw(TKameException);
  protected:
    char _Delimiter, _Quote;
};


class TKamePlainTextFieldParser: public TKamePlainTextLineTokenizer {
  public:
    TKamePlainTextFieldParser(void);
    virtual ~TKamePlainTextFieldParser();
    virtual void SetFieldTypeList(const std::vector<TKameVariant>& FieldTypeList);
    virtual unsigned ProcessLine(TKameTable::TRow Row, std::string& Line, unsigned ColumnOffset) throw(TKameException);
  protected:
    virtual unsigned ProcessSimpleLine(TKameTable::TRow Row, std::istream& LineStream, unsigned ColumnOffset) throw(TKameException);
    virtual unsigned ProcessTokenizedLine(TKameTable::TRow Row, std::vector<std::string>& ElementList, unsigned ColumnOffset) throw(TKameException);
  protected:
    enum TFieldType {
	FieldType_Int,
	FieldType_Float,
	FieldType_String,
	_NumberOfFieldTypes
    };
    std::vector<TFieldType> _FieldTypeList;
    bool _IsAllFloat;
};


class TKamePlainTextHeaderProcessor: public TKamePlainTextLineTokenizer {
  public:
    TKamePlainTextHeaderProcessor(void);
    virtual ~TKamePlainTextHeaderProcessor(void);
    virtual unsigned NumberOfItems();
    virtual bool ProcessLine(const std::string& Line);
    virtual std::string ItemNameOf(unsigned ItemIndex);
    virtual TKameVariant ItemValueOf(unsigned ItemIndex);
    virtual TKameVariant ItemValueOf(const std::string& ItemName);
    virtual std::vector<TKameVariant> ItemValueListOf(unsigned ItemIndex);
    virtual std::vector<TKameVariant> ItemValueListOf(const std::string& ItemName);
  protected:
    std::vector<std::pair<std::string, TKameVariant> > _ItemList;
    std::map<std::string, int> _ItemTable;
};


}
#endif
