/* KameRepository.cc */
/* Created by Enomoto Sanshiro on 9 July 2009. */
/* Last updated by Enomoto Sanshiro on 9 July 2009. */


#include <string>
#include "KameDefs.hh"
#include "KameNtuple.hh"
#include "KameGraph.hh"
#include "KameHistogram.hh"
#include "KameHistogram2d.hh"
#include "KameTrend.hh"
#include "KameHistory.hh"
#include "KameRepository.hh"


using namespace std;
using namespace kame;



void TKameRepository::SaveGraph(const TKameGraph& Graph, const std::string& Name) throw(TKameException)
{
    TKameNtuple Ntuple;

    Ntuple.ImportProperty(Graph);
    Ntuple.Property("Type") = "Graph";

    Ntuple.SetColumnNameType(0, "X", "float");
    Ntuple.SetColumnNameType(1, "Y", "float");
    if (Graph.HasErrors()) {
	Ntuple.SetColumnNameType(2, "XError", "float");
	Ntuple.SetColumnNameType(3, "YError", "float");
    }

    unsigned NumberOfPoints = Graph.NumberOfPoints();
    const double* XValueList = Graph.XValueList();
    const double* YValueList = Graph.YValueList();
    const double* XErrorList = Graph.XErrorList();
    const double* YErrorList = Graph.YErrorList();

    unsigned SegmentIndex = 0;
    unsigned NextSegmentBoundary = Graph.SegmentSizeOf(SegmentIndex);

    for (unsigned Index = 0; Index < NumberOfPoints; Index++) {
	if (Index == NextSegmentBoundary) {
	    NextSegmentBoundary += Graph.SegmentSizeOf(++SegmentIndex);
	    Ntuple.BreakSegment();
	}

	Ntuple[Index][0] = XValueList[Index];
	Ntuple[Index][1] = YValueList[Index];
	if (Graph.HasErrors()) {
	    Ntuple[Index][2] = XErrorList[Index];
	    Ntuple[Index][3] = YErrorList[Index];
	}
    }

    this->SaveNtuple(Ntuple, Name);
}

void TKameRepository::LoadGraph(TKameGraph& Graph, const std::string& Name, int Revision) throw(TKameException)
{
    TKameNtuple Ntuple;
    this->LoadNtuple(Ntuple, Name, Revision);

    if (Ntuple.Property("Type") != "Graph") {
	throw TKameException(
	    "TKameRepository::LoadGraph()",
	    "Graph object is expected: " + Name +
	    " (" + string(Ntuple.Property("Type")) + ")"
	);
    }
    Graph.ImportProperty(Ntuple);

    int NumberOfRows = Ntuple.NumberOfRows();
    bool HasErrors = (Ntuple.NumberOfColumns() >= 3);
    unsigned CurrentSegmentIndex = 0;
    unsigned CurrentSegmentSize = 0;

    for (int i = 0; i < NumberOfRows; i++) {
	if (Ntuple.SegmentIndexOf(i) != CurrentSegmentIndex) {
	    Graph.BreakSegment();
	    CurrentSegmentIndex = Ntuple.SegmentIndexOf(i);
	    CurrentSegmentSize = 0;
	}

	if (HasErrors) {
	    Graph.Fill(
		Ntuple[i][0], Ntuple[i][1], Ntuple[i][2], Ntuple[i][3]
	    );
	}
	else {
	    Graph.Fill(Ntuple[i][0], Ntuple[i][1]);
	}
    }
    
    if (CurrentSegmentSize > 0) {
	Graph.BreakSegment();
    }
}

void TKameRepository::SaveHistogram(const TKameHistogram& Histogram, const std::string& Name) throw(TKameException)
{
    TKameNtuple Ntuple;

    Ntuple.ImportProperty(Histogram);
    Ntuple.Property("Type") = "Histogram";

    Ntuple.Property("NumberOfBins") = Histogram.Scale().NumberOfBins();
    Ntuple.Property("Min") = Histogram.Scale().Min();
    Ntuple.Property("Max") = Histogram.Scale().Max();
    Ntuple.Property("Overflow") = Histogram.OverflowCounts();
    Ntuple.Property("Underflow") = Histogram.UnderflowCounts();
    Ntuple.Property("AxisTitle") = Histogram.Scale().Title();

    long AccumulationTime = Histogram.AccumulationTime();
    if (AccumulationTime >= 0) {
	Ntuple.Property("AccumulationTime") = AccumulationTime;
    }

    Ntuple.SetColumnNameType(0, "BinCenter", "float");
    Ntuple.SetColumnNameType(1, "Counts", "float");
    if (Histogram.HasErrors()) {
	Ntuple.SetColumnNameType(2, "Error", "float");
    }

    for (int Index = 0; Index < Histogram.Scale().NumberOfBins(); Index++) {
	Ntuple[Index][0] = Histogram.Scale().BinCenterOf(Index);
	Ntuple[Index][1] = Histogram.CountsOf(Index);
	if (Histogram.HasErrors()) {
	    Ntuple[Index][2] = Histogram.ErrorOf(Index);
	}
    }

    this->SaveNtuple(Ntuple, Name);
}

void TKameRepository::LoadHistogram(TKameHistogram& Histogram, const std::string& Name, int Revision) throw(TKameException)
{
    TKameNtuple Ntuple;
    this->LoadNtuple(Ntuple, Name, Revision);

    if (Ntuple.Property("Type") != "Histogram") {
	throw TKameException(
	    "TKameRepository::LoadHistogram()",
	    "Histogram object is expected: " + Name +
	    " (" + string(Ntuple.Property("Type")) + ")"
	);
    }
    Histogram.ImportProperty(Ntuple);

    int NumberOfBins = Ntuple.Property("NumberOfBins");
    double Min = Ntuple.Property("Min");
    double Max = Ntuple.Property("Max");
    Histogram.SetScale(TKameHistogramScale(NumberOfBins, Min, Max));
    
    if (! Ntuple.Property("AxisTitle").IsVoid()) {
	Histogram.Scale().SetTitle(Ntuple.Property("AxisTitle"));
    }
    if (! Ntuple.Property("AccumulationTime").IsVoid()) {
	Histogram.SetAccumulationTime(Ntuple.Property("AccumulationTime"));
    }

    int NumberOfRows = Ntuple.NumberOfRows();
    bool HasErrors = (Ntuple.NumberOfColumns() >= 3);
    for (int i = 0; i < NumberOfRows; i++) {
	if (HasErrors) {
	    Histogram.Fill(Ntuple[i][0], Ntuple[i][1], Ntuple[i][2]);
	}
	else {
	    Histogram.Fill(Ntuple[i][0], Ntuple[i][1]);
	}
    }
}

void TKameRepository::SaveHistogram2d(const TKameHistogram2d& Histogram, const std::string& Name) throw(TKameException)
{
    TKameNtuple Ntuple;

    Ntuple.ImportProperty(Histogram);
    Ntuple.Property("Type") = "Histogram2d";

    Ntuple.Property("NumberOfXBins") = Histogram.XScale().NumberOfBins();
    Ntuple.Property("XMin") = Histogram.XScale().Min();
    Ntuple.Property("XMax") = Histogram.XScale().Max();
    Ntuple.Property("NumberOfYBins") = Histogram.YScale().NumberOfBins();
    Ntuple.Property("YMin") = Histogram.YScale().Min();
    Ntuple.Property("YMax") = Histogram.YScale().Max();
    Ntuple.Property("XAxisTitle") = Histogram.XScale().Title();
    Ntuple.Property("YAxisTitle") = Histogram.YScale().Title();

    Ntuple.SetColumnNameType(0, "XBinCenter", "float");
    Ntuple.SetColumnNameType(1, "YBinCenter", "float");
    Ntuple.SetColumnNameType(2, "Counts", "float");

    int XIndex, YIndex, RowIndex = 0;
    for (YIndex = 0; YIndex < Histogram.YScale().NumberOfBins(); YIndex++) {
	for (XIndex = 0; XIndex < Histogram.XScale().NumberOfBins(); XIndex++) {
	    Ntuple[RowIndex][0] = Histogram.XScale().BinCenterOf(XIndex);
	    Ntuple[RowIndex][1] = Histogram.YScale().BinCenterOf(YIndex);
	    Ntuple[RowIndex][2] = Histogram.CountsOf(XIndex, YIndex);
	    RowIndex++;
	}
    }

    this->SaveNtuple(Ntuple, Name);
}

void TKameRepository::LoadHistogram2d(TKameHistogram2d& Histogram, const std::string& Name, int Revision) throw(TKameException)
{
    TKameNtuple Ntuple;
    this->LoadNtuple(Ntuple, Name, Revision);

    if (Ntuple.Property("Type") != "Histogram2d") {
	throw TKameException(
	    "TKameRepository::LoadHistogram2d()",
	    "Histogram2d object is expected: " + Name +
	    " (" + string(Ntuple.Property("Type")) + ")"
	);
    }
    Histogram.ImportProperty(Ntuple);

    int NumberOfXBins = Ntuple.Property("NumberOfXBins");
    double XMin = Ntuple.Property("XMin");
    double XMax = Ntuple.Property("XMax");
    int NumberOfYBins = Ntuple.Property("NumberOfYBins");
    double YMin = Ntuple.Property("YMin");
    double YMax = Ntuple.Property("YMax");
    Histogram.SetScale(
	TKameHistogramScale(NumberOfXBins, XMin, XMax),
	TKameHistogramScale(NumberOfYBins, YMin, YMax)
    );
    
    if (! Ntuple.Property("XAxisTitle").IsVoid()) {
	Histogram.XScale().SetTitle(Ntuple.Property("XAxisTitle"));
    }
    if (! Ntuple.Property("YAxisTitle").IsVoid()) {
	Histogram.YScale().SetTitle(Ntuple.Property("YAxisTitle"));
    }

    int NumberOfRows = Ntuple.NumberOfRows();
    for (int i = 0; i < NumberOfRows; i++) {
	Histogram.Fill(Ntuple[i][0], Ntuple[i][1], Ntuple[i][2]);
    }
}

void TKameRepository::SaveTrend(const TKameTrend& Trend, const std::string& Name) throw(TKameException)
{
    TKameNtuple Ntuple;

    Ntuple.ImportProperty(Trend);
    Ntuple.Property("Type") = "Trend";

    Ntuple.Property("StartTime") = Trend.StartTime();
    Ntuple.Property("TickInterval") = Trend.TickInterval();
    Ntuple.Property("WindowLength") = Trend.WindowLength();

    Ntuple.SetColumnNameType(0, "Time", "int");
    Ntuple.SetColumnNameType(1, "Counts", "float");
    Ntuple.SetColumnNameType(2, "Sum", "float");
    Ntuple.SetColumnNameType(3, "Mean", "float");
    Ntuple.SetColumnNameType(4, "Deviation", "float");
    Ntuple.SetColumnNameType(5, "MinValue", "float");
    Ntuple.SetColumnNameType(6, "MaxValue", "float");

    unsigned NumberOfPoints = Trend.NumberOfPoints();

    for (unsigned Index = 0; Index < NumberOfPoints; Index++) {
	Ntuple[Index][0] = Trend.PassedTimeOf(Index);
	Ntuple[Index][1] = Trend.CountsOf(Index);
	Ntuple[Index][2] = Trend.SumOf(Index);
	Ntuple[Index][3] = Trend.MeanOf(Index);
	Ntuple[Index][4] = Trend.DeviationOf(Index);
	Ntuple[Index][5] = Trend.MinValueOf(Index);
	Ntuple[Index][6] = Trend.MaxValueOf(Index);
    }

    this->SaveNtuple(Ntuple, Name);
}

void TKameRepository::LoadTrend(TKameTrend& Trend, const std::string& Name, int Revision) throw(TKameException)
{
    TKameNtuple Ntuple;
    this->LoadNtuple(Ntuple, Name, Revision);

    if (Ntuple.Property("Type") != "Trend") {
	throw TKameException(
	    "TKameRepository::LoadTrend()",
	    "Trend object is expected: " + Name +
	    " (" + string(Ntuple.Property("Type")) + ")"
	);
    }
    Trend.ImportProperty(Ntuple);

    long StartTime = Ntuple.Property("StartTime");
    long TickInterval = Ntuple.Property("TickInterval");
    long WindowLength = Ntuple.Property("WindowLength");

    Trend.SetScale(TickInterval, WindowLength);
    Trend.Start(StartTime);

    int NumberOfRows = Ntuple.NumberOfRows();
    for (int i = 0; i < NumberOfRows; i++) {
	try {
	    long Time = Ntuple[i][0];
	    double Counts = Ntuple[i][1];
	    double Mean = Ntuple[i][3];
	    double Deviation = Ntuple[i][4];
	    double MinValue = Ntuple[i][4];
	    double MaxValue = Ntuple[i][4];
	    Trend.Fill(
		StartTime + Time, 
		Mean, Counts, Deviation, 
		MinValue, MaxValue
	    );
	}
	catch (TKameException &e) {
	    cout << "[" << Ntuple[i][0] << "]" << endl;
	    throw;
	}
    }
}

void TKameRepository::SaveHistory(const TKameHistory& History, const std::string& Name) throw(TKameException)
{
    TKameNtuple Ntuple;

    Ntuple.ImportProperty(History);
    Ntuple.Property("Type") = "History";

    Ntuple.Property("StartTime") = History.StartTime();
    Ntuple.Property("MaxNumberOfSamples") = History.MaxNumberOfSamples();

    Ntuple.SetColumnNameType(0, "Time", "int");
    Ntuple.SetColumnNameType(1, "Counts", "float");
    Ntuple.SetColumnNameType(2, "Sum", "float");
    Ntuple.SetColumnNameType(3, "Mean", "float");
    Ntuple.SetColumnNameType(4, "Deviation", "float");

    unsigned NumberOfPoints = History.NumberOfSamples();

    for (unsigned Index = 0; Index < NumberOfPoints; Index++) {
	Ntuple[Index][0] = History.PassedTimeOf(Index);
	Ntuple[Index][1] = History.CountsOf(Index);
	Ntuple[Index][2] = History.SumOf(Index);
	Ntuple[Index][3] = History.MeanOf(Index);
	Ntuple[Index][4] = History.DeviationOf(Index);
    }

    this->SaveNtuple(Ntuple, Name);
}

void TKameRepository::LoadHistory(TKameHistory& History, const std::string& Name, int Revision) throw(TKameException)
{
    TKameNtuple Ntuple;
    this->LoadNtuple(Ntuple, Name, Revision);

    if (Ntuple.Property("Type") != "History") {
	throw TKameException(
	    "TKameRepository::LoadHistory()",
	    "History object is expected: " + Name +
	    " (" + string(Ntuple.Property("Type")) + ")"
	);
    }
    History.ImportProperty(Ntuple);

    long MaxNumberOfSamples = Ntuple.Property("MaxNumberOfSamples");
    long StartTime = Ntuple.Property("StartTime");

    History.SetScale(MaxNumberOfSamples);
    History.Start(StartTime);

    int NumberOfRows = Ntuple.NumberOfRows();
    for (int i = 0; i < NumberOfRows; i++) {
	try {
	    long Time = Ntuple[i][0];
	    long Counts = Ntuple[i][1];
	    double Sum = Ntuple[i][2];
	    double Mean = Ntuple[i][3];
	    double Deviation = Ntuple[i][4];
	    History.InsertSample(
		StartTime + Time, Counts, Sum, Mean, Deviation
	    );
	}
	catch (TKameException &e) {
	    cout << "[" << Ntuple[i][0] << "]" << endl;
	    throw;
	}
    }
}
