/* KameTable.hh */
/* Created by Enomoto Sanshiro on 29 June 2009. */
/* Last updated by Enomoto Sanshiro on 29 June 2009. */


#ifndef __KameTable_hh__
#define __KameTable_hh__


#include <vector>
#include <map>
#include "KameVariant.hh"


namespace kame {


class TKameTable {
  public:
    class TRow {
      public:
	inline TRow(TKameTable& Table, unsigned RowIndex);
	inline TKameVariant& operator[](unsigned ColumnIndex);
	inline TKameVariant& operator[](const std::string& ColumnName) throw(TKameException);
      private:
	TKameTable& _Table;
	unsigned _RowIndex;
    };
    class TConstRow {
      public:
	inline TConstRow(const TKameTable& Table, unsigned RowIndex);
	inline const TKameVariant& operator[](unsigned ColumnIndex) const;
	inline const TKameVariant& operator[](const std::string& ColumnName) const throw(TKameException);
      private:
	const TKameTable& _Table;
	unsigned _RowIndex;
    };
  public:
    TKameTable(void);
    TKameTable(const TKameTable& Table);
    explicit TKameTable(unsigned NumberOfColumns);
    virtual ~TKameTable() {}
    virtual TKameTable& operator=(const TKameTable& Table);
    virtual void Clear(void);
    inline unsigned NumberOfColumns(void) const;
    inline unsigned NumberOfRows(void) const;
    inline std::vector<TKameVariant>& GetColumn(unsigned ColumnIndex);
    inline const std::vector<TKameVariant>& GetColumn(unsigned ColumnIndex) const;
    inline TRow operator[](unsigned RowIndex);
    inline TConstRow operator[](unsigned RowIndex) const;
  public:
    virtual void SetColumnName(unsigned ColumnIndex, const std::string& ColumnName);
    virtual const std::string& ColumnNameOf(unsigned ColumnIndex) const throw(TKameException);
    virtual unsigned ColumnIndexOf(const std::string& ColumnName) const throw(TKameException);
  private:
    mutable std::vector<std::vector<TKameVariant> > _Table;
    mutable unsigned _NumberOfRows;
    mutable std::map<unsigned, std::string> _ColumnIndexNameTable;
    mutable std::map<std::string, unsigned> _ColumnNameIndexTable;
};



inline unsigned TKameTable::NumberOfColumns(void) const
{
    return _Table.size();
}

inline unsigned TKameTable::NumberOfRows(void) const
{
    return _NumberOfRows;
}

inline std::vector<TKameVariant>& TKameTable::GetColumn(unsigned ColumnIndex)
{
    if (ColumnIndex >= _Table.size()) {
	_Table.insert(
	    _Table.end(), ColumnIndex-_Table.size()+1, 
	    std::vector<TKameVariant>(_NumberOfRows)
	);
    }

    return _Table[ColumnIndex];
}

inline const std::vector<TKameVariant>& TKameTable::GetColumn(unsigned ColumnIndex) const
{
    if (ColumnIndex >= _Table.size()) {
	_Table.insert(
	    _Table.end(), ColumnIndex-_Table.size()+1, 
	    std::vector<TKameVariant>(_NumberOfRows)
	);
    }

    return _Table[ColumnIndex];
}

inline TKameTable::TRow TKameTable::operator[](unsigned RowIndex)
{
    if (RowIndex >= _NumberOfRows) {
	_NumberOfRows = RowIndex + 1;
	for (unsigned i = 0; i < _Table.size(); i++) {
	    _Table[i].resize(_NumberOfRows);
	}
    }

    return TRow(*this, RowIndex);
}

inline TKameTable::TConstRow TKameTable::operator[](unsigned RowIndex) const
{
    if (RowIndex >= _NumberOfRows) {
	_NumberOfRows = RowIndex + 1;
	for (unsigned i = 0; i < _Table.size(); i++) {
	    _Table[i].resize(_NumberOfRows);
	}
    }

    return TConstRow(*this, RowIndex);
}

inline TKameTable::TRow::TRow(TKameTable& Table, unsigned RowIndex)
: _Table(Table), _RowIndex(RowIndex)
{
}

inline TKameVariant& TKameTable::TRow::operator[](unsigned ColumnIndex)
{
    return _Table.GetColumn(ColumnIndex)[_RowIndex];
}

inline TKameVariant& TKameTable::TRow::operator[](const std::string& ColumnName) throw(TKameException)
{
    return _Table.GetColumn(_Table.ColumnIndexOf(ColumnName))[_RowIndex];
}

inline TKameTable::TConstRow::TConstRow(const TKameTable& Table, unsigned RowIndex)
: _Table(Table), _RowIndex(RowIndex)
{
}

inline const TKameVariant& TKameTable::TConstRow::operator[](unsigned ColumnIndex) const
{
    return _Table.GetColumn(ColumnIndex)[_RowIndex];
}

inline const TKameVariant& TKameTable::TConstRow::operator[](const std::string& ColumnName) const throw(TKameException)
{
    return _Table.GetColumn(_Table.ColumnIndexOf(ColumnName))[_RowIndex];
}


}
#endif
