/* KameTrend.hh */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Updated by Enomoto Sanshiro on 29 June 2002. */
/* Updated by Enomoto Sanshiro on 13 February 2008. */
/* Updated by Enomoto Sanshiro on 9 July 2009. */


#ifndef __KameTrend_hh__
#define __KameTrend_hh__


#include <string>
#include <cmath>
#include <algorithm>
#include "KameObject.hh"


namespace kame {


class TKameTrend: public TKameObject {
  public:
    TKameTrend(void);
    TKameTrend(long TickInterval, long WindowLength);
    TKameTrend(const TKameTrend& Trend);
    virtual ~TKameTrend();
    virtual TKameTrend& operator=(const TKameTrend& Trend);
  public:
    void SetScale(long TickInterval, long WindowLength);
    void Clear(void);
    void Start(long StartTime);
    void UpdateRange(long MaxTimeStamp);
    inline void Fill(long TimeStamp, double Value);
    inline void Fill(long TimeStamp, double Mean, double Weight, double Deviation, double MinValue, double MaxValue);
  public:
    inline long StartTime(void) const;
    inline long TickInterval(void) const;
    inline long WindowLength(void) const;
    inline long NumberOfPoints(void) const;
    inline long TimeOf(long PointIndex) const;
    inline long PassedTimeOf(long PointIndex) const;
    inline double CountsOf(long PointIndex) const;
    inline double SumOf(long PointIndex) const;
    inline double MeanOf(long PointIndex) const;
    inline double DeviationOf(long PointIndex) const;
    inline double MinValueOf(long PointIndex) const;
    inline double MaxValueOf(long PointIndex) const;
  protected:
    inline long IndexOf(long TimeStamp) const;
  protected:
    long _Interval, _Length;
    long _Depth, _NumberOfPoints, _BaseIndex;
    long _StartTime, _MinTime, _MaxTime;
    double* _CountsStorage;
    double* _SumStorage;
    double* _SumOfSqrStorage;
    double* _MinValueStorage;
    double* _MaxValueStorage;
};


inline long TKameTrend::IndexOf(long TimeStamp) const
{
    return (((TimeStamp - _MinTime) / _Interval) + _BaseIndex) % _Depth;
}

inline void TKameTrend::Fill(long TimeStamp, double Value)
{
    if (_StartTime == 0) {
	Start(TimeStamp);
    }

    if (TimeStamp < _MinTime) {
	return;
    }
    if (TimeStamp >= _MaxTime) {
	UpdateRange(TimeStamp);
    }

    int Index = IndexOf(TimeStamp);

    if (_CountsStorage[Index] == 0) {
	_MinValueStorage[Index] = Value;
	_MaxValueStorage[Index] = Value;
    }
    else {
	_MinValueStorage[Index] = std::min(Value, _MinValueStorage[Index]);
	_MaxValueStorage[Index] = std::max(Value, _MaxValueStorage[Index]);
    }

    _CountsStorage[Index] += 1;
    _SumStorage[Index] += Value;
    _SumOfSqrStorage[Index] += Value * Value;
}

inline void TKameTrend::Fill(long TimeStamp, double Mean, double Weight, double Deviation, double MinValue, double MaxValue)
{
    if (_StartTime == 0) {
	Start(TimeStamp);
    }

    if (TimeStamp < _MinTime) {
	return;
    }
    if (TimeStamp >= _MaxTime) {
	UpdateRange(TimeStamp);
    }

    int Index = IndexOf(TimeStamp);

    if (_CountsStorage[Index] == 0) {
	_MinValueStorage[Index] = MinValue;
	_MaxValueStorage[Index] = MaxValue;
    }
    else {
	_MinValueStorage[Index] = std::min(MinValue, _MinValueStorage[Index]);
	_MaxValueStorage[Index] = std::max(MaxValue, _MaxValueStorage[Index]);
    }

    _CountsStorage[Index] += Weight;
    _SumStorage[Index] += Mean * Weight;
    _SumOfSqrStorage[Index] += (Deviation*Deviation + Mean*Mean) * Weight;
}

inline long TKameTrend::StartTime(void) const
{
    return _StartTime;
}

inline long TKameTrend::TickInterval(void) const
{
    return _Interval;
}

inline long TKameTrend::WindowLength(void) const
{
    return _Length;
}

inline long TKameTrend::NumberOfPoints(void) const
{
    return _NumberOfPoints;
}

inline long TKameTrend::TimeOf(long PointIndex) const
{
    return _MinTime + _Interval * PointIndex;
}

inline long TKameTrend::PassedTimeOf(long PointIndex) const
{
    return _MinTime + _Interval * PointIndex - _StartTime;
}

inline double TKameTrend::CountsOf(long PointIndex) const
{
    return _CountsStorage[(_BaseIndex + PointIndex) % _Depth];
}

inline double TKameTrend::SumOf(long PointIndex) const
{
    return _SumStorage[(_BaseIndex + PointIndex) % _Depth];
}

inline double TKameTrend::MeanOf(long PointIndex) const
{
    int Index = (_BaseIndex + PointIndex) % _Depth;
    if (_CountsStorage[Index] == 0) {
	return 0;
    }
    else {
	return _SumStorage[Index] / _CountsStorage[Index];
    }
}

inline double TKameTrend::DeviationOf(long PointIndex) const
{
    int Index = (_BaseIndex + PointIndex) % _Depth;
    if (_CountsStorage[Index] == 0) {
	return 0;
    }
    else {
	double MeanOfSqr = _SumOfSqrStorage[Index] / _CountsStorage[Index];
	double Mean = MeanOf(PointIndex);
	double SqrOfMean = Mean * Mean;
	
	if (MeanOfSqr > SqrOfMean) {
	    return sqrt(MeanOfSqr - SqrOfMean);
	}
	else {
	    return 0;
	}
    }
}

inline double TKameTrend::MinValueOf(long PointIndex) const
{
    return _MinValueStorage[(_BaseIndex + PointIndex) % _Depth];
}

inline double TKameTrend::MaxValueOf(long PointIndex) const
{
    return _MaxValueStorage[(_BaseIndex + PointIndex) % _Depth];
}


}
#endif
