/* Kasp2dHistogram.cc */
/* Created by Enomoto Sanshiro on 13 January 2001. */
/* Last updated by Enomoto Sanshiro on 29 June 2002. */


#include <string>
#include <deque>
#include <sstream>
#include "KaspHistogram.hh"
#include "KaspGraph.hh"
#include "Kasp2dHistogram.hh"


using namespace std;


TKasp2dHistogram::TKasp2dHistogram(const string& Title, const TKaspHistogramScale& XScale, const TKaspHistogramScale& YScale)
: TKaspObject(Title), _XScale(XScale), _YScale(YScale)
{
    _NumberOfXBins = _XScale.NumberOfBins();
    _NumberOfYBins = _YScale.NumberOfBins();

    _Storage = new double[_NumberOfXBins * _NumberOfYBins];

    Clear();
}

TKasp2dHistogram::~TKasp2dHistogram()
{
    delete[] _Storage;
}

void TKasp2dHistogram::Clear(void)
{
    _PeakCounts = 0;
    _NumberOfEntries = 0;

    for (int i = 0; i < _NumberOfXBins * _NumberOfYBins; i++) {
	_Storage[i] = 0;
    }
}

bool TKasp2dHistogram::HasData(void)
{
    return (_NumberOfEntries > 0);
}


TKaspGraph* TKasp2dHistogram::MakeContour(double Threshold, string Title)
{
    if (Title.empty()) {
	ostringstream os;
	os << " (z = " << Threshold << ")";
	Title = _Title + os.str();
    }

    TKaspGraph* Graph = new TKaspGraph(Title);

    deque<pair<pair<int, int>, pair<int, int> > > FragmentList;
    deque<pair<int, int> > StartPointList;

    int NumberOfXBins = _XScale.NumberOfBins();
    int NumberOfYBins = _YScale.NumberOfBins();

    // collect fragments //

    for (int YBin = 0; YBin < NumberOfYBins; YBin++) {
	for (int XBin = 0; XBin < NumberOfXBins; XBin++) {
	    double Z = CountsOf(XBin, YBin) - Threshold;

	    if (XBin > 0) {
		double LeftZ = CountsOf(XBin - 1, YBin) - Threshold;
		if ((LeftZ > 0) && (Z <= 0)) {
		    FragmentList.push_back(make_pair(
		        make_pair(XBin, YBin), make_pair(XBin, YBin + 1)
		    ));
		    if (YBin == 0) {
			StartPointList.push_back(FragmentList.back().first);
		    }
		}
		else if ((LeftZ <= 0) && (Z > 0)) {
		    FragmentList.push_back(make_pair(
		        make_pair(XBin, YBin + 1), make_pair(XBin, YBin)
		    ));
		    if (YBin == NumberOfYBins - 1) {
			StartPointList.push_back(FragmentList.back().first);
		    }
		}
	    }

	    if (YBin > 0) {
		double LowerZ = CountsOf(XBin, YBin - 1) - Threshold;
		if ((LowerZ <= 0) && (Z > 0)) {
		    FragmentList.push_back(make_pair(
			make_pair(XBin, YBin), make_pair(XBin + 1, YBin)
		    ));
		    if (XBin == 0) {
			StartPointList.push_back(FragmentList.back().first);
		    }
		}
		else if ((LowerZ > 0) && (Z <= 0)) {
		    FragmentList.push_back(make_pair(
			make_pair(XBin + 1, YBin), make_pair(XBin, YBin)
		    ));
		    if (XBin == NumberOfXBins - 1) {
			StartPointList.push_back(FragmentList.back().first);
		    }
		}
	    }
	}
    }

    if (FragmentList.empty()) {
	return Graph;
    }

    // build up segments //

    int SegmentCount = 1;
    deque<pair<pair<int, int>, pair<int, int> > >::iterator Fragment;
    pair<int, int> StartPoint, CurrentPoint, LastPoint;
    pair<double, double> StartPointXY;

    if (! StartPointList.empty()) {
	// segment which starts from an edge (open path)
	CurrentPoint = StartPointList.front();
	StartPointList.pop_front();
    }
    else {
	// segment which does not have start point (closed path)
	CurrentPoint = FragmentList.front().first;
    }
    StartPoint = CurrentPoint;
    LastPoint = CurrentPoint;

    while (! FragmentList.empty()) {
	Fragment = FragmentList.begin();
	while (Fragment != FragmentList.end()) {
	    if (Fragment->first == CurrentPoint) {
		CurrentPoint = Fragment->second;
		break;
	    }
	    else {
		Fragment++;
	    }
	}

	if (Fragment != FragmentList.end()) {
	    double X, Y;
	    if (CurrentPoint.first != LastPoint.first) {
		// right bound or left bound
		double X0 = _XScale.BinValueOf(LastPoint.first);
		double X1 = _XScale.BinValueOf(CurrentPoint.first);
		int XBin = min(LastPoint.first, CurrentPoint.first);
		double Y0 = _YScale.BinCenterOf(LastPoint.second - 1);
		double Y1 = _YScale.BinCenterOf(LastPoint.second);
		double Z0 = CountsOf(XBin, LastPoint.second - 1);
		double Z1 = CountsOf(XBin, LastPoint.second);
		X = (X0 + X1) / 2;
		Y = (Threshold - Z0) / (Z1 - Z0) * (Y1 - Y0) + Y0;
	    }
	    else {
		// upper bound or lower bound
		double Y0 = _YScale.BinValueOf(LastPoint.second);
		double Y1 = _YScale.BinValueOf(CurrentPoint.second);
		int YBin = min(LastPoint.second, CurrentPoint.second);
		double X0 = _XScale.BinCenterOf(LastPoint.first - 1);
		double X1 = _XScale.BinCenterOf(LastPoint.first);
		double Z0 = CountsOf(LastPoint.first - 1, YBin);
		double Z1 = CountsOf(LastPoint.first, YBin);
		Y = (Y0 + Y1) / 2;
		X = (Threshold - Z0) / (Z1 - Z0) * (X1 - X0) + X0;
	    }

	    if (LastPoint == StartPoint) {
		StartPointXY = make_pair(X, Y);
	    }
	    Graph->Fill(X, Y);

	    LastPoint = CurrentPoint;
	    FragmentList.erase(Fragment);
	}
	else {
	    if (CurrentPoint == StartPoint) {
		Graph->Fill(StartPointXY.first, StartPointXY.second);
	    }
	    Graph->BreakSegment();
	    SegmentCount++;

	    if (! StartPointList.empty()) {
		CurrentPoint = StartPointList.front();
		StartPointList.pop_front();
	    }
	    else {
		CurrentPoint = FragmentList.front().first;
	    }
	    StartPoint = CurrentPoint;
	    LastPoint = CurrentPoint;
	}
    }

    if (CurrentPoint == StartPoint) {
	Graph->Fill(StartPointXY.first, StartPointXY.second);
    }
    Graph->BreakSegment();

    return Graph;
}
