/* KaspHistogram.cc */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Last updated by Enomoto Sanshiro on 29 June 2002. */


#include <string>
#include <cmath>
#include "KaspHistogram.hh"

using namespace std;

#define sqr(x) ((x) * (x))


TKaspHistogram::TKaspHistogram(const string& Title, const TKaspHistogramScale& Scale)
: TKaspObject(Title), _Scale(Scale)
{
    _CountStorage = new double[_Scale.NumberOfBins()];
    _ErrorStorage = 0;

    Clear();
}

TKaspHistogram::TKaspHistogram(const string& Title, int NumberOfBins, double Min, double Max)
 : TKaspObject(Title), 
   _Scale(TKaspHistogramScale(Title, NumberOfBins, Min, Max))
{
    _CountStorage = new double[_Scale.NumberOfBins()];
    _ErrorStorage = 0;

    Clear();
}

TKaspHistogram::~TKaspHistogram()
{
    delete[] _CountStorage;
    delete[] _ErrorStorage;
}

void TKaspHistogram::Clear(void)
{
    _OverflowCounts = 0;
    _UnderflowCounts = 0;
    _NumberOfEntries = 0;

    _Sum = 0;
    _SumOfSquared = 0;

    _PeakCounts = 0;
    _PeakBinIndex = 0;

    _AccumulationTime = -1;

    for (int i = 0; i < _Scale.NumberOfBins(); i++) {
	_CountStorage[i] = 0;
    }
}

bool TKaspHistogram::HasData(void)
{
    return (_NumberOfEntries > 0);
}

void TKaspHistogram::SetAccumulationTime(long AccumulationTime)
{
    _AccumulationTime = AccumulationTime;
}

void TKaspHistogram::Scale(double Factor)
{
    int NumberOfBins = _Scale.NumberOfBins();

    if (_ErrorStorage == 0) {
	_ErrorStorage = new double[NumberOfBins];
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    _ErrorStorage[BinIndex] = sqrt(_CountStorage[BinIndex]);
	}
    }

    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	_CountStorage[BinIndex] *= Factor;
	_ErrorStorage[BinIndex] *= fabs(Factor);
    }
    _PeakCounts *= Factor;

    _UnderflowCounts *= Factor;
    _OverflowCounts *= Factor;

    _NumberOfEntries *= Factor;
    _Sum *= Factor;
    _SumOfSquared *= Factor;
}

void TKaspHistogram::AddHistogram(TKaspHistogram& Histogram, double Weight) throw(TKaspException)
{
    int NumberOfBins = _Scale.NumberOfBins();
    if (Histogram.Scale().NumberOfBins() != NumberOfBins) {
	throw TKaspException(
	    "TKaspHistogram::AddHistogram()", "inconsistent histogram size"
	);
    }

    if (_ErrorStorage == 0) {
	_ErrorStorage = new double[NumberOfBins];
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    _ErrorStorage[BinIndex] = sqrt(_CountStorage[BinIndex]);
	}
    }

    _PeakCounts = 0;
    _PeakBinIndex = 0;

    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	double Counts = (
	    CountsOf(BinIndex) + Weight * Histogram.CountsOf(BinIndex)
	);
	double Error = sqrt(
	    sqr(ErrorOf(BinIndex)) + sqr(Weight * Histogram.ErrorOf(BinIndex))
	);
	    
	_CountStorage[BinIndex] = Counts;
	_ErrorStorage[BinIndex] = Error;

	if (Counts > _PeakCounts) {
	     _PeakCounts = Counts;
	     _PeakBinIndex = BinIndex;
	}
    }

    _UnderflowCounts += Weight * Histogram._UnderflowCounts;
    _OverflowCounts += Weight * Histogram._OverflowCounts;

    _NumberOfEntries += Weight * Histogram._NumberOfEntries;
    _Sum += Weight * Histogram._Sum;
    _SumOfSquared += Weight * Histogram._SumOfSquared;
}

void TKaspHistogram::MultiplyHistogram(TKaspHistogram& Histogram, double Power) throw(TKaspException)
{
    int NumberOfBins = _Scale.NumberOfBins();
    if (Histogram.Scale().NumberOfBins() != NumberOfBins) {
	throw TKaspException(
	    "TKaspHistogram::MultiplyHistogram()", "inconsistent histogram size"
	);
    }

    if (_ErrorStorage == 0) {
	_ErrorStorage = new double[NumberOfBins];
	for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	    _ErrorStorage[BinIndex] = sqrt(_CountStorage[BinIndex]);
	}
    }

    _PeakCounts = 0;
    _PeakBinIndex = 0;

    _NumberOfEntries = 0;
    _Sum = 0;
    _SumOfSquared = 0;
    for (int BinIndex = 0; BinIndex < NumberOfBins; BinIndex++) {
	double Counts = (
	    CountsOf(BinIndex) * pow(Histogram.CountsOf(BinIndex), Power)
	);
	double Error = 0; //...
	    
	_CountStorage[BinIndex] = Counts;
	_ErrorStorage[BinIndex] = Error;

	if (Counts > _PeakCounts) {
	     _PeakCounts = Counts;
	     _PeakBinIndex = BinIndex;
	}
	_NumberOfEntries += Counts;
	_Sum += Counts * _Scale.BinCenterOf(BinIndex);
	_SumOfSquared += sqr(Counts * _Scale.BinCenterOf(BinIndex));
    }

    _UnderflowCounts *= pow(Histogram._UnderflowCounts, Power);
    _OverflowCounts *= pow(Histogram._OverflowCounts, Power);
}



TKaspHistogramScale::TKaspHistogramScale(int NumberOfBins, double Min, double Max)
{
    _NumberOfBins = NumberOfBins;
    _Min = Min;
    _Max = Max;

    _BinWidth = (_Max - _Min) / _NumberOfBins;
}

TKaspHistogramScale::TKaspHistogramScale(const string& Title, int NumberOfBins, double Min, double Max)
{
    _Title = Title;

    _NumberOfBins = NumberOfBins;
    _Min = Min;
    _Max = Max;

    _BinWidth = (_Max - _Min) / _NumberOfBins;
}

TKaspHistogramScale::~TKaspHistogramScale()
{
}

string TKaspHistogramScale::Title(void) const
{
    return _Title;
}

void TKaspHistogramScale::SetTitle(const string& Title)
{
    _Title = Title;
}
