/* KaspHistory.cc */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Last updated by Enomoto Sanshiro on 29 June 2002. */


#include <cmath>
#include <string>
#include "KaspHistory.hh"

using namespace std;

#define sqr(x) ((x) * (x))



TKaspHistory::TKaspHistory(const string& Title, long MaxNumberOfSamples)
: TKaspObject(Title)
{
    _MaxNumberOfSamples = MaxNumberOfSamples;

    _TimeStorage = new long[_MaxNumberOfSamples];
    _CountStorage = new long[_MaxNumberOfSamples];
    _SumStorage = new double[_MaxNumberOfSamples];
    _MeanStorage = new double[_MaxNumberOfSamples];
    _DeviationStorage = new double[_MaxNumberOfSamples];

    Clear();
}

TKaspHistory::~TKaspHistory()
{
    delete[] _TimeStorage;
    delete[] _CountStorage;
    delete[] _SumStorage;
    delete[] _MeanStorage;
    delete[] _DeviationStorage;
}

void TKaspHistory::Clear(void)
{
    _BaseIndex = 0;
    _NumberOfSamples = 0;
    _StartTime = 0;

    _CurrentSum = 0;
    _CurrentSumOfSquared = 0;
    _CurrentCounts = 0;
}

bool TKaspHistory::HasData(void)
{
    return (_NumberOfSamples > 0);
}

void TKaspHistory::Start(long CurrentTime)
{
    _StartTime = CurrentTime;
}

void TKaspHistory::HoldSample(long CurrentTime)
{
    double Mean = 0;
    double Deviation = 0;
    if (_CurrentCounts > 0) {
	Mean = _CurrentSum / _CurrentCounts;
	double MeanOfSquared = _CurrentSumOfSquared / _CurrentCounts;
	Deviation = sqrt(MeanOfSquared - sqr(Mean));
    }
    
    long Time = (CurrentTime > 0) ? CurrentTime : _LastDataTime;

    InsertSample(Time, _CurrentCounts, _CurrentSum, Mean, Deviation);
 
    _CurrentSum = 0;
    _CurrentSumOfSquared = 0;
    _CurrentCounts = 0;

    if (_StartTime == 0) {
	_StartTime = Time;
    }
}

void TKaspHistory::InsertSample(long Time, long Counts, double Sum, double Mean, double Deviation)
{
    long Index = (_BaseIndex + _NumberOfSamples) % _MaxNumberOfSamples;
    _TimeStorage[Index] = Time;
    _CountStorage[Index] = Counts;
    _SumStorage[Index] = Sum;
    _MeanStorage[Index] = Mean;
    _DeviationStorage[Index] = Deviation;
 
    if (_NumberOfSamples < _MaxNumberOfSamples) {
	_NumberOfSamples++;
    }
    else {
	_BaseIndex++;
	if (_BaseIndex == _MaxNumberOfSamples) {
	    _BaseIndex = 0;
	}
    }
}
