/* KaspTrend.hh */
/* Created by Enomoto Sanshiro on 29 October 1999. */
/* Updated by Enomoto Sanshiro on 29 June 2002. */
/* Last updated by Enomoto Sanshiro on 13 February 2008. */


#ifndef __KaspTrend_hh__
#define __KaspTrend_hh__


#include <string>
#include <cmath>
#include "KaspObject.hh"


class TKaspTrend: public TKaspObject {
  public:
    TKaspTrend(const std::string& Title, long TickInterval, long WindowLength);
    ~TKaspTrend();
    void Start(long StartTime);
    void Clear(void);
    void UpdateRange(long MaxTimeStamp);
    inline void Fill(long TimeStamp, double Value);
    inline void Fill(long TimeStamp, double Mean, double Weight, double Deviation);
  public:
    inline long StartTime(void);
    inline long TickInterval(void);
    inline long WindowLength(void);
    inline long NumberOfPoints(void);
    inline long TimeOf(long PointIndex);
    inline long PassedTimeOf(long PointIndex);
    inline double CountsOf(long PointIndex);
    inline double SumOf(long PointIndex);
    inline double MeanOf(long PointIndex);
    inline double DeviationOf(long PointIndex);
  protected:
    inline long IndexOf(long TimeStamp);
  protected:
    long _Interval, _Length;
    long _Depth, _NumberOfPoints, _BaseIndex;
    long _StartTime, _MinTime, _MaxTime;
    double* _CountsStorage;
    double* _SumStorage;
    double* _SumOfSqrStorage;
};


inline long TKaspTrend::IndexOf(long TimeStamp)
{
    return (((TimeStamp - _MinTime) / _Interval) + _BaseIndex) % _Depth;
}

inline void TKaspTrend::Fill(long TimeStamp, double Value)
{
    if (_StartTime == 0) {
	Start(TimeStamp);
    }

    if (TimeStamp < _MinTime) {
	return;
    }
    if (TimeStamp >= _MaxTime) {
	UpdateRange(TimeStamp);
    }

    int Index = IndexOf(TimeStamp);
    _CountsStorage[Index] += 1;
    _SumStorage[Index] += Value;
    _SumOfSqrStorage[Index] += Value * Value;
}

inline void TKaspTrend::Fill(long TimeStamp, double Mean, double Weight, double Deviation)
{
    if (_StartTime == 0) {
	Start(TimeStamp);
    }

    if (TimeStamp < _MinTime) {
	return;
    }
    if (TimeStamp >= _MaxTime) {
	UpdateRange(TimeStamp);
    }

    int Index = IndexOf(TimeStamp);
    _CountsStorage[Index] += Weight;
    _SumStorage[Index] += Mean * Weight;
    _SumOfSqrStorage[Index] += (Deviation*Deviation + Mean*Mean) * Weight;
}

inline long TKaspTrend::StartTime(void)
{
    return _StartTime;
}

inline long TKaspTrend::TickInterval(void)
{
    return _Interval;
}

inline long TKaspTrend::WindowLength(void)
{
    return _Length;
}

inline long TKaspTrend::NumberOfPoints(void)
{
    return _NumberOfPoints;
}

inline long TKaspTrend::TimeOf(long PointIndex)
{
    return _MinTime + _Interval * PointIndex;
}

inline long TKaspTrend::PassedTimeOf(long PointIndex)
{
    return _MinTime + _Interval * PointIndex - _StartTime;
}

inline double TKaspTrend::CountsOf(long PointIndex)
{
    return _CountsStorage[(_BaseIndex + PointIndex) % _Depth];
}

inline double TKaspTrend::SumOf(long PointIndex)
{
    return _SumStorage[(_BaseIndex + PointIndex) % _Depth];
}

inline double TKaspTrend::MeanOf(long PointIndex)
{
    int Index = (_BaseIndex + PointIndex) % _Depth;
    if (_CountsStorage[Index] == 0) {
	return 0;
    }
    else {
	return _SumStorage[Index] / _CountsStorage[Index];
    }
}

inline double TKaspTrend::DeviationOf(long PointIndex)
{
    int Index = (_BaseIndex + PointIndex) % _Depth;
    if (_CountsStorage[Index] == 0) {
	return 0;
    }
    else {
	double MeanOfSqr = _SumOfSqrStorage[Index] / _CountsStorage[Index];
	double Mean = MeanOf(PointIndex);
	double SqrOfMean = Mean * Mean;
	
	if (MeanOfSqr > SqrOfMean) {
	    return sqrt(MeanOfSqr - SqrOfMean);
	}
	else {
	    return 0;
	}
    }
}

#endif
