/* KcomEventProcessor.cc */
/* Created by Enomoto Sanshiro on 27 March 2000. */
/* Updated by Enomoto Sanshiro on 13 April 2002. */
/* Last updated by Enomoto Sanshiro on 19 January 2010. */


#include <string>
#include <vector>
#include <map>
#include <deque>
#include "KorbOrb.hh"
#include "KorbObjectReference.hh"
#include "KorbNamingContext.hh"
#include "KcomDefs.hh"
#include "KcomEvent.hh"
#include "KcomComponentOrb.hh"
#include "KcomEventProcessor.hh"

using namespace std;

static const int DefaultTimeout_sec = 10;
static const unsigned MaxNumberOfAsynchronousRequests = 10;


TKcomEventProcessor::TKcomEventProcessor(TKorbOrb* Orb)
{
    _Orb = Orb;

    _Timeout_sec = DefaultTimeout_sec;
    _IsTimeoutEnabled = true;
    _IsInAsynchronousMode = false;
    _IsInOnewayMode = false;
    _IsAsynchronousModeDisabled = false;
}

TKcomEventProcessor::~TKcomEventProcessor()
{
    map <string, TKcomComponentProxy*>::iterator Component;

    for (
	Component = _ComponentTable.begin();
	Component != _ComponentTable.end();
	Component++
    ){
	delete (*Component).second;
    }

    for (
	Component = _DismissedComponentTable.begin();
	Component != _DismissedComponentTable.end();
	Component++
    ){
	delete (*Component).second;
    }
}

int TKcomEventProcessor::RegisterComponent(const string& ComponentName)
{
    TKorbObjectReference ObjRef;
    try {
	if (_Orb->NamingContext()->LookupObject(ComponentName, ObjRef)) {
	    _ComponentTable[ComponentName] = (
		new TKcomComponentProxy(ObjRef, _Orb)
	    );
	}
	else {
	    return 0;
	}
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomEventProcessor::RegisterComponent()",
	    "korb exception: " + e.Message()
	);
    }

    return 1;
}

void TKcomEventProcessor::EnterAsynchronousMode(void)
{
    _IsInAsynchronousMode = true;
}

void TKcomEventProcessor::ExitAsynchronousMode(void)
{
    _IsInAsynchronousMode = false;
}

void TKcomEventProcessor::DisableAsynchronousMode(void)
{
    _IsAsynchronousModeDisabled = true;
}

void TKcomEventProcessor::EnterOnewayMode(void)
{
    _IsInOnewayMode = true;
}

void TKcomEventProcessor::ExitOnewayMode(void)
{
    _IsInOnewayMode = false;
}

void TKcomEventProcessor::SetTimeout(int Timeout_sec)
{
    _Timeout_sec = Timeout_sec;
    _IsTimeoutEnabled = true;
}

void TKcomEventProcessor::EnableTimeout(void)
{
    _IsTimeoutEnabled = true;
}

void TKcomEventProcessor::DisableTimeout(void)
{
    _IsTimeoutEnabled = false;
}

int TKcomEventProcessor::EmitEvent(const string& TargetName, TKcomEvent& Event, TKcomEventResponse& EventResponse) throw(TKcomException)
{
    if (_ComponentTable.count(TargetName) == 0) {
	if (RegisterComponent(TargetName) == 0) {
	    EventResponse.IsError() = true;
	    EventResponse.ReturnValue() = "no such component: " + TargetName;
	    cerr << "ERROR: " << EventResponse.ReturnValue() << endl;
	    return 0;
	}
    }

    if (_ComponentTable[TargetName] == 0) {
	// component was dismissed //
	EventResponse.IsError() = true;
	EventResponse.ReturnValue() = "message to dismissed component ignored: " + TargetName;

	return 0;
    }

    TKcomComponentProxy* TargetComponent = _ComponentTable[TargetName];
    int RequestId = EmitEventTo(TargetComponent, Event, _IsInOnewayMode);

    if (! _IsInOnewayMode) {
	if (_IsInAsynchronousMode && ! _IsAsynchronousModeDisabled) {
	    _AsynchronousEventQueue.push_back(make_pair(TargetComponent, RequestId));

	    // If the message queue becomes full, asynchronous transactions
	    // go into dead-lock state, bacause both of replys and new requests
            // cannot be sent. Note that reply packets are taken from the
            // message queue in WaitForNextReply(), which is called after 
            // all request packets are sent. The following is to avoid
            // this dead-lock state.
	    if (_AsynchronousEventQueue.size() >= MaxNumberOfAsynchronousRequests) {
		WaitForNextAsynchronousReply(EventResponse);
	    }
	}
	else {
	    WaitForReplyFrom(TargetComponent, RequestId, EventResponse);
	}
    }

    return 1;
}

int TKcomEventProcessor::WaitForNextAsynchronousReply(TKcomEventResponse& EventResponse) throw(TKcomException)
{
    if (_AsynchronousEventQueue.empty()) {
	return 0;
    }

    TKcomComponentProxy* Component = _AsynchronousEventQueue.front().first;
    int RequestId = _AsynchronousEventQueue.front().second;
    _AsynchronousEventQueue.pop_front();

    WaitForReplyFrom(Component, RequestId, EventResponse);

    return 1;
}

int TKcomEventProcessor::EmitEventTo(TKcomComponentProxy* ComponentProxy, TKcomEvent& Event, bool IsOneWay) throw(TKcomException)
{
    int RequestId = 0;
    try {
	RequestId = ComponentProxy->EmitEvent(Event, IsOneWay);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomEventProcessor::EmitEventTo()",
	    "korb exception: " + e.Message()
	);
    }

    return RequestId;
}

int TKcomEventProcessor::WaitForReplyFrom(TKcomComponentProxy* ComponentProxy, int RequestId, TKcomEventResponse& EventResponse) throw(TKcomException)
{
    int Result = 0;
    try {
	Result = ComponentProxy->WaitEventResponse(
	    RequestId, EventResponse, (_IsTimeoutEnabled ? _Timeout_sec : 0)
	);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomEventProcessor::WaitReplyFor()",
	    "korb exception: " + e.Message()
	);
    }

    if (Result == 0) {
	DismissComponent(ComponentProxy);
    }

    return Result;
}

void TKcomEventProcessor::DismissComponent(TKcomComponentProxy* ComponentProxy) throw(TKcomException)
{
    if (ComponentProxy == 0) {
	return;
    }

    map <string, TKcomComponentProxy*>::iterator ComponentIterator;
    for (
	ComponentIterator = _ComponentTable.begin();
	ComponentIterator != _ComponentTable.end();
	ComponentIterator++
    ){
	if ((*ComponentIterator).second == ComponentProxy) {
	    break;
	}
    }

    if (ComponentIterator != _ComponentTable.end()) {
	string ComponentName = (*ComponentIterator).first;
	cerr << "ERROR: no response from component: " << ComponentName << endl;
	cerr << "component " << ComponentName << " is dismissed..." << endl;

	_DismissedComponentTable[ComponentName] = ComponentProxy;
	_ComponentTable[ComponentName] = 0;
    }
}
