/* KcomObjectAssembler.cc */
/* Created by Enomoto Sanshiro on 17 August 2000. */
/* Last updated by Enomoto Sanshiro on 17 August 2000. */


#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include "MushTimer.hh"
#include "KorbOrb.hh"
#include "KorbNamingContext.hh"
#include "KcomObjectAssembler.hh"

using namespace std;


#define VERBOSE



TKcomAssignment::TKcomAssignment(const string& ProviderName, const string& ProviderObjectName, const string& UserName, const string& UserObjectName)
{
    _ProviderName = ProviderName;
    _ProviderObjectName = ProviderObjectName;
    _UserName = UserName;
    _UserObjectName = UserObjectName;
}

TKcomAssignment::~TKcomAssignment()
{
}

const string& TKcomAssignment::ProviderName(void) const
{
    return _ProviderName;
}

const string& TKcomAssignment::ProviderObjectName(void) const
{
    return _ProviderObjectName;
}

const string& TKcomAssignment::UserName(void) const
{
    return _UserName;
}

const string& TKcomAssignment::UserObjectName(void) const
{
    return _UserObjectName;
}

string TKcomAssignment::ExportName(void) const
{
    return _ProviderName + "." + _ProviderObjectName;
}

string TKcomAssignment::ImportName(void) const
{
    return _UserName + "." + _UserObjectName;
}

void TKcomAssignment::Dump(ostream& os, const string& Indent) const
{
    os << Indent << ImportName() << " := " << ExportName() << endl;
}



TKcomObjectAssembler::TKcomObjectAssembler(TKorbOrb* Orb)
{
    _Orb = Orb;
}

TKcomObjectAssembler::~TKcomObjectAssembler()
{
}

void TKcomObjectAssembler::AddAssignment(const TKcomAssignment& Assignment)
{
    _AssignmentList.push_back(Assignment);
    _AssignmentTable[Assignment.ImportName()] = Assignment.ExportName();
}

string TKcomObjectAssembler::AssignedObjectOf(const string& ImportName)
{
    return _AssignmentTable[ImportName];
}

int TKcomObjectAssembler::LookupObject(const string& ImportName, TKorbObjectReference& ObjectReference) throw(TKcomException)
{
    string ExportName = _AssignmentTable[ImportName];
    if (ExportName.empty()) {
	return 0;
    }

    if (_ExportObjectTable.count(ExportName) > 0) {
	ObjectReference = _ExportObjectTable[ExportName];
	return ObjectReference.IsNull() ? 0 : 1;
    }
    _ExportObjectTable[ExportName] = TKorbObjectReference(); // null object

    int Result = 0;
    const int MaxNumberOfTrys = 100;
    for (int i = 0; i < MaxNumberOfTrys; i++) {
	if (_Orb->NamingContext()->LookupObject(ExportName, ObjectReference)) {
	    _ExportObjectTable[ExportName] = ObjectReference;
	    Result = 1;
	    break;
	}
	TMushRealTimeTimer(0, 100000).Suspend();
    }

#ifdef VERBOSE
    if (Result == 0) {
	cerr << "TKcomObjectAssembler::LookupObject(): ";
	cerr << "unable to find export object for " << ImportName << endl;
    }
#endif

    return Result;
}

int TKcomObjectAssembler::LookupService(const string& ClassName, TKorbObjectReference& ObjectReference) throw(TKcomException)
{
    if (_ServiceObjectTable.count(ClassName) > 0) {
	ObjectReference = _ServiceObjectTable[ClassName];
	return ObjectReference.IsNull() ? 0 : 1;
    }
    _ServiceObjectTable[ClassName] = TKorbObjectReference(); // null object

    int Result = 0;
    const int MaxNumberOfTrys = 100;
    for (int i = 0; i < MaxNumberOfTrys; i++) {
	if (_Orb->NamingContext()->LookupService(ClassName, ObjectReference)) {
	    _ServiceObjectTable[ClassName] = ObjectReference;
	    Result = 1;
	    break;
	}
	TMushRealTimeTimer(0, 100000).Suspend();
    }

#ifdef VERBOSE
    if (Result == 0) {
	cerr << "TKcomObjectAssembler::LookupService(): ";
	cerr << "unable to find service for " << ClassName << endl;
    }
#endif

    return Result;
}

void TKcomObjectAssembler::Dump(ostream& os, const string& Indent) const
{
    for (unsigned int i = 0; i < _AssignmentList.size(); i++) {
	_AssignmentList[i].Dump(os, Indent);
    }
}
