/* KcomRegistryOrb.cc */
/* Created by Enomoto Sanshiro on 28 May 2000. */
/* Last updated by Enomoto Sanshiro on 19 May 2003. */


#include "KorbOrb.hh"
#include "KorbObjectProxy.hh"
#include "KorbObjectMessenger.hh"
#include "KorbObjectAdapter.hh"
#include "KcomDefs.hh"
#include "KcomRegistry.hh"
#include "KcomRegistryRepository.hh"
#include "KcomRegistryOrb.hh"

using namespace std;


static const char* RegistryClassName = "KcomRegistry";
static const char* RegistryObjectName = "KcomRegistry";


TKcomRegistryProxy::TKcomRegistryProxy(TKorbOrb* Orb)
: TKorbObjectProxy(RegistryClassName, Orb), TKcomRegistry()
{
}

TKcomRegistryProxy::~TKcomRegistryProxy()
{
}

void TKcomRegistryProxy::SetValue(const string& RegistryPath, const string& Value)
{
    int MethodId = TKcomRegistryMessenger::MethodId_SetValue;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);
    OrbPacket.IsOneWay() = 1;

    OrbPacket.ArgumentSerializer().PutString(RegistryPath);
    OrbPacket.ArgumentSerializer().PutString(Value);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException& e) {
	throw TKcomException(
	    "TKcomRegistryProxy::SetValue()", "korb exception: " + e.Message()
	);
    }
}

bool TKcomRegistryProxy::GetValue(const string& RegistryPath, string& Value)
{
    int MethodId = TKcomRegistryMessenger::MethodId_GetValue;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    OrbPacket.ArgumentSerializer().PutString(RegistryPath);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException& e) {
	throw TKcomException(
	    "TKcomRegistryProxy::GetValue()", "korb exception: " + e.Message()
	);
    }

    int Result;
    OrbPacket.ArgumentSerializer().GetInt(Result);
    OrbPacket.ArgumentSerializer().GetString(Value);

    return (bool) Result;
}

vector<string> TKcomRegistryProxy::ChildEntryNameListOf(const string& RegistryPath)
{
    int MethodId = TKcomRegistryMessenger::MethodId_ChildEntryNameListOf;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    OrbPacket.ArgumentSerializer().PutString(RegistryPath);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException& e) {
	throw TKcomException(
	    "TKcomRegistryProxy::ChildEntryNameListO()",
	    "korb exception: " + e.Message()
	);
    }

    int NumberOfChildEntries;
    vector<string> ChildEntryNameList;
    string ChildEntryName;
    OrbPacket.ArgumentSerializer().GetInt(NumberOfChildEntries);
    for (int i = 0; i < NumberOfChildEntries; i++) {
	OrbPacket.ArgumentSerializer().GetString(ChildEntryName);
	ChildEntryNameList.push_back(ChildEntryName);
    }

    return ChildEntryNameList;
}

int TKcomRegistryProxy::GetSequenceValue(const string& Category)
{
    int MethodId = TKcomRegistryMessenger::MethodId_GetSequenceValue;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);

    OrbPacket.ArgumentSerializer().PutString(Category);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException& e) {
	throw TKcomException(
	    "TKcomRegistryProxy::GetSequenceValue()",
	    "korb exception: " + e.Message()
	);
    }

    int NewValue;
    OrbPacket.ArgumentSerializer().GetInt(NewValue);

    return NewValue;
}

void TKcomRegistryProxy::SaveTo(const std::string& FileName, const std::string& BasePath) throw(TKcomException)
{
    int MethodId = TKcomRegistryMessenger::MethodId_SaveTo;
    TKorbOrbPacket OrbPacket = CreateOrbPacket(MethodId);
    OrbPacket.IsOneWay() = 1;

    OrbPacket.ArgumentSerializer().PutString(FileName);
    OrbPacket.ArgumentSerializer().PutString(BasePath);

    try {
	InvokeMethod(OrbPacket);
    }
    catch (TKorbException& e) {
	throw TKcomException(
	    "TKcomRegistryProxy::SaveTo()",
	    "korb exception: " + e.Message()
	);
    }
}

TKcomRegistryEntry* TKcomRegistryProxy::GetEntry(std::string RegistryPath, TKcomRegistryEntry* BaseEntry)
{
    throw TKcomException(
	"TKcomRegistryProxy::GetEntry()", "function not available for KORB"
    );

    return 0;
}



TKcomRegistryMessenger::TKcomRegistryMessenger(TKcomRegistry* Registry)
: TKorbObjectMessenger(RegistryClassName)
{
    _Registry = Registry;
}

TKcomRegistryMessenger::~TKcomRegistryMessenger()
{
}

int TKcomRegistryMessenger::DispatchMethodInvocation(TKorbOrbPacket& OrbPacket) throw(TKorbException)
{
    int Result;

    switch (OrbPacket.MethodId()) {
      case MethodId_SetValue:
	Result = SetValue(OrbPacket);
	break;
      case MethodId_GetValue:
	Result = GetValue(OrbPacket);
	break;
      case MethodId_ChildEntryNameListOf:
	Result = ChildEntryNameListOf(OrbPacket);
	break;
      case MethodId_GetSequenceValue:
	Result = GetSequenceValue(OrbPacket);
	break;
      case MethodId_SaveTo:
	Result = SaveTo(OrbPacket);
	break;
      default:
	Result = 0;
    }

    return Result;
}

int TKcomRegistryMessenger::SetValue(TKorbOrbPacket& OrbPacket)
{
    string RegistryPath; string Value;
    OrbPacket.ArgumentSerializer().GetString(RegistryPath);
    OrbPacket.ArgumentSerializer().GetString(Value);

    _Registry->SetValue(RegistryPath, Value);

    return 1;
}

int TKcomRegistryMessenger::GetValue(TKorbOrbPacket& OrbPacket)
{
    string RegistryPath; string Value;
    OrbPacket.ArgumentSerializer().GetString(RegistryPath);

    int Result = (int) _Registry->GetValue(RegistryPath, Value);
    
    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutInt(Result);
    OrbPacket.ArgumentSerializer().PutString(Value);
    
    return 1;
}

int TKcomRegistryMessenger::ChildEntryNameListOf(TKorbOrbPacket& OrbPacket)
{
    string RegistryPath; vector<string> ChildEntryNameList;
    OrbPacket.ArgumentSerializer().GetString(RegistryPath);

    ChildEntryNameList = _Registry->ChildEntryNameListOf(RegistryPath);

    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutInt(ChildEntryNameList.size());
    for (unsigned i = 0; i < ChildEntryNameList.size(); i++) {
	OrbPacket.ArgumentSerializer().PutString(ChildEntryNameList[i]);
    }
    
    return 1;
}

int TKcomRegistryMessenger::GetSequenceValue(TKorbOrbPacket& OrbPacket)
{
    string SequenceName; int SequenceValue;
    OrbPacket.ArgumentSerializer().GetString(SequenceName);

    SequenceValue = _Registry->GetSequenceValue(SequenceName);
    
    OrbPacket.ArgumentSerializer().Rewind();
    OrbPacket.ArgumentSerializer().PutInt(SequenceValue);
    
    return 1;
}

int TKcomRegistryMessenger::SaveTo(TKorbOrbPacket& OrbPacket)
{
    string FileName, BasePath;
    OrbPacket.ArgumentSerializer().GetString(FileName);
    OrbPacket.ArgumentSerializer().GetString(BasePath);

    try {
	_Registry->SaveTo(FileName, BasePath);
    }
    catch (TKcomException &e) {
	cerr << "ERROR: " << e << endl; //...
    }
    
    return 1;
}



TKcomRegistryServer::TKcomRegistryServer(void)
{
    _Orb = new TKorbOrb();
    _ObjectAdapter = _Orb->ObjectAdapter();

    _Registry = 0;
    _RegistryMessenger = 0;
}

TKcomRegistryServer::~TKcomRegistryServer()
{
    delete _RegistryMessenger;
    delete _Registry;
    delete _Orb;
}

void TKcomRegistryServer::Start(void) throw(TKcomException)
{
    _Registry = new TKcomRegistry();
    _RegistryMessenger = new TKcomRegistryMessenger(_Registry);

    try {
	_ObjectAdapter->AddMessenger(_RegistryMessenger, RegistryObjectName);
	_ObjectAdapter->Start();
	_ObjectAdapter->RemoveMessenger(RegistryObjectName);
    }
    catch (TKorbException &e) {
	throw TKcomException(
	    "TKcomRegistryServer::Start()",
	    "korb exception: " + e.Message()
	);
    }
}

TKcomRegistry* TKcomRegistryServer::Registry(void)
{
    return _Registry;
}
