/* KcomRegistryRepository.cc */
/* Created by Enomoto Sanshiro on 27 May 2000. */
/* Last updated by Enomoto Sanshiro on 29 October 2005. */


#include <iostream>
#include <fstream>
#include <string>
#include <deque>
#include "MushXml.hh"
#include "KcomRegistry.hh"
#include "KcomRegistryRepository.hh"

using namespace std;
using namespace mush;


class TKcomRegistryRepositoryHandler: public sax::DocumentHandler {
  public:
    TKcomRegistryRepositoryHandler(TKcomRegistryEntry* RootEntry);
    virtual ~TKcomRegistryRepositoryHandler();
  public:
    virtual void startElement(const std::string& name, const sax::AttributeList& attribute_list);
    virtual void endElement(const std::string& name);
    virtual void characters(const std::string& text);
  protected:
    std::deque<TKcomRegistryEntry*> _EntryStack;
};



TKcomRegistryRepositoryHandler::TKcomRegistryRepositoryHandler(TKcomRegistryEntry* RootEntry)
{
    _EntryStack.push_back(RootEntry);
}

TKcomRegistryRepositoryHandler::~TKcomRegistryRepositoryHandler()
{
}

void TKcomRegistryRepositoryHandler::startElement(const std::string& name, const sax::AttributeList& attribute_list)
{
    TKcomRegistryEntry* ChildEntry = new TKcomRegistryEntry(name, "");

    _EntryStack.back()->AddChildEntry(ChildEntry);
    _EntryStack.push_back(ChildEntry);
}

void TKcomRegistryRepositoryHandler::endElement(const std::string& name)
{
    _EntryStack.pop_back();
}

void TKcomRegistryRepositoryHandler::characters(const std::string& text)
{
    _EntryStack.back()->SetValue(_EntryStack.back()->Value() + text);
}



TKcomRegistryRepository::TKcomRegistryRepository(const string& FileName)
{
    _FileName = FileName;
}

TKcomRegistryRepository::~TKcomRegistryRepository()
{
}

void TKcomRegistryRepository::Save(TKcomRegistry* Registry, const string& BasePath) throw(TKcomException)
{
    ofstream File(_FileName.c_str());
    if (! File) {
	throw TKcomException(
	    "TKcomRegistryRepository::Save()", 
	    "unable to open file: " + _FileName
	);
    }

    File << "<?xml version=\"1.0\"?>" << endl;
    File << endl;
    File << "<kcom-registry>" << endl;

    TKcomRegistryEntry* BaseEntry = Registry->GetEntry(BasePath);
    WriteChildEntries(BaseEntry, File, "  ");

    File << "</kcom-registry>" << endl;
}

TKcomRegistry* TKcomRegistryRepository::Load(void) throw(TKcomException)
{
    ifstream File(_FileName.c_str());
    if (! File) {
	throw TKcomException(
	    "TKcomRegistryRepository::Load()", 
	    "unable to open file: " + _FileName
	);
    }

    TKcomRegistry* Registry = new TKcomRegistry();
    TKcomRegistryEntry* RootEntry = Registry->GetEntry("/");

    try {
	TKcomRegistryRepositoryHandler Handler(RootEntry);
	sax::Parser Parser;
	Parser.setDocumentHandler(&Handler);
	Parser.parse(_FileName);
    }
    catch (TSystemCallException &e) {
	delete Registry;
	throw TKcomException(
 	    "TKcomRegistryRepository::Load()",
	    "script exception: " + e.Message()
	);
    }

    return Registry;
}

void TKcomRegistryRepository::WriteChildEntries(TKcomRegistryEntry* Entry, ostream& os, string Indent)
{
    const vector<TKcomRegistryEntry*>& ChildEntryList = Entry->ChildEntryList();
    for (unsigned i = 0; i < ChildEntryList.size(); i++) {
	TKcomRegistryEntry* ChildEntry = ChildEntryList[i];

	string Value, RawValue = ChildEntry->Value();
	for (
	    string::iterator i = RawValue.begin(); 
	    i != RawValue.end(); i++
	){
	    switch (*i) {
	      case '&':
		Value += "&amp;";
		break;
	      case '<':
		Value += "&lt;";
		break;
	      case '>':
		Value += "&gt;";
		break;
	      case '"':
		Value += "&quot;";
		break;
	      case '\'':
		Value += "&apos;";
		break;
	      default:
		Value += *i;
		break;
	    }
	}

	os << Indent << "<" << ChildEntry->Name() << ">";
	if (ChildEntry->ChildEntryList().empty()) {
	    if (! Value.empty()) {
		os << Value;
	    }
	    os << "</" << ChildEntry->Name() << ">" << endl;
	}
	else {
	    os << endl;
	    if (! Value.empty()) {
		os << Indent << "  " << Value << endl;
	    }
	    WriteChildEntries(ChildEntry, os, Indent + "  ");	    
	    os << Indent << "</" << ChildEntry->Name() << ">" << endl;
	}
    }    
}
