/* KcomScript.cc */
/* Created by Enomoto Sanshiro on 8 April 2000. */
/* Last updated by Enomoto Sanshiro on 25 September 2005. */


#include <iostream>
#include <fstream>
#include <string>
#include <algorithm>
#include "MushMisc.hh"
#include "KiscScript.hh"
#include "KcomManager.hh"
#include "KcomEvent.hh"
#include "KcomRegistry.hh"
#include "KcomRegistryRepository.hh"
#include "KcomEventProcessor.hh"
#include "KcomComponentAssembler.hh"
#include "KcomScript.hh"

using namespace std;


//#define DEBUG
#ifdef DEBUG
static ofstream dumpfile("Dump-KcomScript", ios::app);
#endif


TKcomScript::TKcomScript(TKcomManager* Manager, int argc, char** argv)
: TKiscScript(argc, argv)
{
    _Manager = Manager;

    _ComponentAssembler = _Manager->GetComponentAssembler();
    _ObjectAssembler = _Manager->GetObjectAssembler();
    _EventProcessor = _Manager->GetEventProcessor();
    _Registry = _Manager->GetRegistry();
}

TKcomScript::~TKcomScript()
{
}

TParaTokenTable* TKcomScript::CreateTokenTable(void)
{
    TParaTokenTable* TokenTable = TKiscScript::CreateTokenTable();
    
    TokenTable->AddKeyword("import");
    TokenTable->AddKeyword("component");
    TokenTable->AddKeyword("assign");
    TokenTable->AddKeyword("asynchronously");
    TokenTable->AddKeyword("oneway");
    TokenTable->AddKeyword("invoke");
    TokenTable->AddKeyword("exit");
    TokenTable->AddKeyword("on");
    TokenTable->AddOperator(":=");
    TokenTable->AddOperator("=>");

    return TokenTable;
}

TParaOperatorTable* TKcomScript::CreateOperatorTable(void)
{
    return TKiscScript::CreateOperatorTable();
}

TParaStatementTable* TKcomScript::CreateStatementTable(void)
{
    TParaStatementTable* StatementTable;
    StatementTable = TKiscScript::CreateStatementTable();

    StatementTable->AddStatement(
	new TKcomImportStatement(
	    _ComponentAssembler, _EventProcessor, _Registry
	)
    );
    StatementTable->AddStatement(
	new TKcomAsynchronousStatement(_EventProcessor)
    );
    StatementTable->AddStatement(
	new TKcomOnewayStatement(_EventProcessor)
    );
    StatementTable->AddStatement(new TKcomAssignStatement(_ObjectAssembler));
    StatementTable->AddStatement(new TKcomExitStatement(_Manager));
    StatementTable->AddStatement(new TKcomInvokeStatement(this));
    StatementTable->AddStatement(new TKcomAfterStatement(this));

    //... for backward compatibility ...//
    StatementTable->AddStatement(new TKcomComponentStatement());

    return StatementTable;
}

TParaPackage* TKcomScript::CreatePackage(void)
{
    TParaPackage* Package = TKiscScript::CreatePackage();

    Package->AddEntry(new TKcomEventHandlerEntry());
    Package->AddEntry(new TKcomPeriodicTaskEntry(this));

    return Package;
}

TParaObjectPrototypeTable* TKcomScript::CreateObjectPrototypeTable(void)
{
    TParaObjectPrototypeTable* ObjectPrototypeTable;
    ObjectPrototypeTable = TKiscScript::CreateObjectPrototypeTable();

    return ObjectPrototypeTable;
}

TParaBuiltinFunctionTable* TKcomScript::CreateBuiltinFunctionTable(void)
{
    TParaBuiltinFunctionTable* BuiltinFunctionTable;
    BuiltinFunctionTable = TKiscScript::CreateBuiltinFunctionTable();

    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKcomSystemScriptMessenger(_Manager)
    );
    BuiltinFunctionTable->RegisterAnonymousClass(
	new TKcomRegistryScriptMessenger(_Registry)
    );

    return BuiltinFunctionTable;
}

int TKcomScript::DispatchEvent(TKcomEvent& Event) throw(TKcomException)
{
    string HandlerName = TKcomEventHandler::HandlerNameOf(
	Event.SourceName(), Event.Name()
    );
    if (_Package->GetEntry(HandlerName) == 0) {
	HandlerName = TKcomEventHandler::HandlerNameOf(
	    TKcomEventHandler::AnonymousObjectName(), Event.Name()
	);

	if (_Package->GetEntry(HandlerName) == 0) {
	    return 0;
	}
    }

    int NumberOfArguments = Event.ArgumentList().size();
    TParaValue* ArgumentValueList = 0;
    vector<TParaValue*> ArgumentList;

    if (NumberOfArguments > 0) {
	ArgumentValueList = new TParaValue[NumberOfArguments];
	for (int i = 0; i < NumberOfArguments; i++) {
	    ArgumentValueList[i] = TParaValue(Event.ArgumentList()[i]);
	    ArgumentList.push_back(&ArgumentValueList[i]);
	}
    }

    try {
	_Package->Execute(HandlerName, ArgumentList, _SymbolTable);
    }
    catch (TScriptException &e) {
	throw TKcomException(
	    "TKcomScript::DispatchEvent()",
	    "script exception: " + e.Message()
	);
    }

    delete[] ArgumentValueList;

    return 1;
}

void TKcomScript::AddPeriodicTask(long Interval, TParaPackageEntry* Task)
{
    _PeriodicTaskTable.push_back(make_pair(Interval, Task));
    _NextExecutionTimeTable.push_back(0);
}

void TKcomScript::AddDelayedTask(long ExecutionTime, TParaFunctionCallNode* Task)
{
    _DelayedTaskList.push_back(make_pair(ExecutionTime, Task));
}

int TKcomScript::ExecuteEventTask(const string& Name) throw(TKcomException)
{
    string HandlerName = TKcomEventHandler::HandlerNameOf(
	TKcomEventHandler::ManagerObjectName(), Name
    );

    if (_Package->GetEntry(HandlerName) == 0) {
	return 0;
    }

    vector<TParaValue*> ArgumentList;
    try {
	_Package->Execute(HandlerName, ArgumentList, _SymbolTable);
    }
    catch (TScriptException &e) {
	throw TKcomException(
	    "TKcomScript::ExecuteHook()",
	    "script exception: " + e.Message()
	);
    }

    return 1;    
}

int TKcomScript::ExecuteScheduledTask(long PresentTime) throw(TKcomException)
{
    int Result = 0;

    for (unsigned i = 0; i < _PeriodicTaskTable.size(); i++) {
	long NextTime = _NextExecutionTimeTable[i];
	if (NextTime == 0) {
	    NextTime = PresentTime;
	}

	if (NextTime <= PresentTime) {
	    long Interval = _PeriodicTaskTable[i].first;
	    TParaPackageEntry* Task = _PeriodicTaskTable[i].second;
	    vector<TParaValue*> ArgumentList;
	    try {
		Task->Execute(ArgumentList, _SymbolTable);
	    }
	    catch (TScriptException &e) {
		_NextExecutionTimeTable[i] = (
		    PresentTime + max(Interval, 86400L)
		);
		throw TKcomException(
		    "TKcomScript::ExecuteScheduledTask()", 
		    "Script Exception: " + e.Message()
		);
	    }

	    NextTime += Interval * ((PresentTime - NextTime) / Interval + 1);
	    _NextExecutionTimeTable[i] = NextTime;

	    Result++;
	}
    }

    list<pair<long, TParaFunctionCallNode*> >::iterator ThisTask, DoneTask;
    ThisTask = _DelayedTaskList.begin(); 
    while (ThisTask != _DelayedTaskList.end()) {
	if (ThisTask->first <= PresentTime) {
	    try {
		ThisTask->second->EvaluateFunction(_SymbolTable);
	    }
	    catch (TScriptException &e) {
		_DelayedTaskList.erase(ThisTask);
		throw TKcomException(
		    "TKcomScript::ExecuteScheduledTask()", 
		    "Script Exception: " + e.Message()
		);
	    }
	    DoneTask = ThisTask;
	    ThisTask++;
	    Result++;
	    _DelayedTaskList.erase(DoneTask);
	}
	else {
	    ThisTask++;
	}
    }

    return Result;
}



TKcomEventHandler::TKcomEventHandler(void)
{
}

TKcomEventHandler::~TKcomEventHandler()
{
}

void TKcomEventHandler::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("on");
    SetReturnValue(new TParaValue());

    TParaToken Token = Tokenizer->Next();
    if (Token.IsIdentifier()) {
	if (Tokenizer->LookAhead().Is(".")) {
	    _ObjectName = Token.AsString();
	    Tokenizer->Next();
	}
	else {
	    _ObjectName = ManagerObjectName();
	    Tokenizer->Unget(Token);
	}
    }
    else if (Token.Is(".")) {
	_ObjectName = AnonymousObjectName();
    }
    else {
        throw TScriptException("event name is expected.");
    }

    Token = Tokenizer->Next();
    if (! Token.IsIdentifier()) {
        throw TScriptException("event name is expected.");
    }
    _EventName = Token.AsString();

    string FunctionName = HandlerNameOf(_ObjectName, _EventName);
    SetName(FunctionName);

    ParseArgumentDeclaration(Tokenizer, StatementParser, SymbolTable);
    
    TParaStatement* Statement = StatementParser->Parse(Tokenizer, SymbolTable);
    SetStatement(Statement);
}

string TKcomEventHandler::HandlerNameOf(const string& ObjectName, const string& EventName)
{
    return ObjectName + "." + EventName;
}

string TKcomEventHandler::AnonymousObjectName(void)
{
    return "[ANONYMOUS]";
}

string TKcomEventHandler::ManagerObjectName(void)
{
    return "[Manager]";
}



TKcomEventHandlerEntry::TKcomEventHandlerEntry(void)
: TParaPackageEntry("event handler")
{
    _EventHandler = 0;
}

TKcomEventHandlerEntry::~TKcomEventHandlerEntry()
{
    delete _EventHandler;
}

TParaPackageEntry* TKcomEventHandlerEntry::Clone(void)
{
    return new TKcomEventHandlerEntry();
}

bool TKcomEventHandlerEntry::HasEntryWordsOf(TParaTokenizer* Tokenizer)
{
    return (
	Tokenizer->LookAhead(1).Is("on") &&
	Tokenizer->LookAhead(2).IsNot("every")
    );
}

void TKcomEventHandlerEntry::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _EventHandler = new TKcomEventHandler();
    _EventHandler->Parse(Tokenizer, StatementParser, SymbolTable);

    string EntryName = _EventHandler->Name();

    SetEntryName(EntryName);
}

TParaValue TKcomEventHandlerEntry::Execute(const vector<TParaValue*>& ArgumentList, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _EventHandler->Execute(ArgumentList, SymbolTable);
}



TKcomPeriodicTaskEntry::TKcomPeriodicTaskEntry(TKcomScript* Script)
: TParaPackageEntry("task")
{
    _Script = Script;
    _Statement = 0;
}

TKcomPeriodicTaskEntry::~TKcomPeriodicTaskEntry()
{
    delete _Statement;
}

TParaPackageEntry* TKcomPeriodicTaskEntry::Clone(void)
{
    return new TKcomPeriodicTaskEntry(_Script);
}

bool TKcomPeriodicTaskEntry::HasEntryWordsOf(TParaTokenizer* Tokenizer)
{
    return (
	Tokenizer->LookAhead(1).Is("on") &&
	Tokenizer->LookAhead(2).Is("every")
    );
}

void TKcomPeriodicTaskEntry::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("on");
    Tokenizer->Next().MustBe("every");
    Tokenizer->Next().MustBe("(");

    long Interval = Tokenizer->Next().AsLong();

    TParaToken Token = Tokenizer->Next();
    if (Token.Is("sec")) {
	Interval *= 1;
    }
    else if (Token.Is("min")) {
	Interval *= 60;
    }
    else if (Token.Is("hour")) {
	Interval *= 60 * 60;
    }
    else if (Token.Is("day")) {
	Interval *= 60 * 60 * 24;
    }
    else {
	Token.ThrowUnexpected("sec, min, hour or day");
    }

    Tokenizer->Next().MustBe(")");

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);

    _Script->AddPeriodicTask(Interval, this);
}

TParaValue TKcomPeriodicTaskEntry::Execute(const vector<TParaValue*>& ArgumentList, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _Statement->Execute(SymbolTable).ReturnValue;
}



TKcomImportStatement::TKcomImportStatement(TKcomComponentAssembler* ComponentAssembler, TKcomEventProcessor* EventProcessor, TKcomRegistry* Registry)
{
    _ComponentAssembler = ComponentAssembler;
    _EventProcessor = EventProcessor;
    _Registry = Registry;
}

TKcomImportStatement::~TKcomImportStatement()
{
}

TParaStatement* TKcomImportStatement::Clone(void)
{
    return new TKcomImportStatement(
	_ComponentAssembler, _EventProcessor, _Registry
    );
}

string TKcomImportStatement::FirstToken(void) const
{
    return string("import");
}

void TKcomImportStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("import");
    TParaToken ComponentTypeToken = Tokenizer->Next();
    Tokenizer->Next().MustBe(";");

    ComponentTypeToken.MustBe(TParaToken::ttIdentifier);

    SymbolTable->ObjectPrototypeTable()->RegisterClass(
	ComponentTypeToken.AsString(), 
	new TKcomComponentScriptMessenger(
	    _ComponentAssembler, _EventProcessor, _Registry
	)
    );
}

TParaStatement::TExecResult TKcomImportStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return TParaStatement::TExecResult();
}



TKcomAssignStatement::TKcomAssignStatement(TKcomObjectAssembler* ObjectAssembler)
{
    _ObjectAssembler = ObjectAssembler;
}

TKcomAssignStatement::~TKcomAssignStatement()
{
}

TParaStatement* TKcomAssignStatement::Clone(void)
{
    return new TKcomAssignStatement(_ObjectAssembler);
}

string TKcomAssignStatement::FirstToken(void) const
{
    return string("assign");
}

void TKcomAssignStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("assign");

    _UserName = Tokenizer->Next().AsString();
    Tokenizer->Next().MustBe(".");
    _UserObjectName = Tokenizer->Next().AsString();
    
    TParaToken Token = Tokenizer->Next();
    if (Token.IsNot(":=") && Token.IsNot("=>")) {
	Token.MustBe("=");
    }
    
    _ProviderName = Tokenizer->Next().AsString();
    Tokenizer->Next().MustBe(".");
    _ProviderObjectName = Tokenizer->Next().AsString();
    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKcomAssignStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _ObjectAssembler->AddAssignment(
	TKcomAssignment(
	    _ProviderName, _ProviderObjectName, _UserName, _UserObjectName
	)
    );

    return TParaStatement::TExecResult();
}




TKcomAsynchronousStatement::TKcomAsynchronousStatement(TKcomEventProcessor* EventProcessor)
{
    _EventProcessor = EventProcessor;
}

TKcomAsynchronousStatement::~TKcomAsynchronousStatement()
{
}

TParaStatement* TKcomAsynchronousStatement::Clone(void)
{
    return new TKcomAsynchronousStatement(_EventProcessor);
}

string TKcomAsynchronousStatement::FirstToken(void) const
{
    return string("asynchronous");
}

void TKcomAsynchronousStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKcomAsynchronousStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    try {
        _EventProcessor->EnterAsynchronousMode();
        TExecResult Result = _Statement->Execute(SymbolTable);
        _EventProcessor->ExitAsynchronousMode();

	TKcomEventResponse EventResponse;
	while (_EventProcessor->WaitForNextAsynchronousReply(EventResponse)) {
	    if (EventResponse.IsError()) {
#if 0
	        throw TScriptException(EventResponse.ReturnValue());
#else
		cerr << "ERROR: " << EventResponse.ReturnValue();
		cerr << " (continue processing)" << endl;
#endif
	    }
	}

	return Result;
    }
    catch (TKcomException &e) {
        throw TScriptException("KCOM Exception: " + e.Message());
    }
    catch (TScriptException &e) {
        throw;
    }
}



TKcomOnewayStatement::TKcomOnewayStatement(TKcomEventProcessor* EventProcessor)
{
    _EventProcessor = EventProcessor;
}

TKcomOnewayStatement::~TKcomOnewayStatement()
{
}

TParaStatement* TKcomOnewayStatement::Clone(void)
{
    return new TKcomOnewayStatement(_EventProcessor);
}

string TKcomOnewayStatement::FirstToken(void) const
{
    return string("oneway");
}

void TKcomOnewayStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKcomOnewayStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _EventProcessor->EnterOnewayMode();
    TExecResult Result = _Statement->Execute(SymbolTable);
    _EventProcessor->ExitOnewayMode();

    return Result;
}



TKcomInvokeStatement::TKcomInvokeStatement(TKcomScript* Script)
{
    _Script = Script;
}

TKcomInvokeStatement::~TKcomInvokeStatement()
{
    for (unsigned i = 0; i < _EventArgumentExpressionList.size(); i++) {
	delete _EventArgumentExpressionList[i];
    }
}

TParaStatement* TKcomInvokeStatement::Clone(void)
{
    return new TKcomInvokeStatement(_Script);
}

string TKcomInvokeStatement::FirstToken(void) const
{
    return string("invoke");
}

void TKcomInvokeStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());

    _EventName = Tokenizer->Next().AsString();
    if (Tokenizer->LookAhead().Is("(")) {
	_EventArgumentExpressionList = ExpressionParser->ParseExpressionList(
	    Tokenizer, SymbolTable, "(", ")", ","
	);
    }

    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKcomInvokeStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TKcomEvent Event;

    Event.SourceName() = "kcom-manager";
    Event.Name() = _EventName;
    for (unsigned i = 0; i < _EventArgumentExpressionList.size(); i++) {
	Event.ArgumentList().push_back(
	    _EventArgumentExpressionList[i]->Evaluate(SymbolTable).AsString()
	);
    }

    long ReturnValue = _Script->DispatchEvent(Event);
    
    TExecResult Result;
    Result.ReturnValue = TParaValue(ReturnValue);

    return Result;
}



TKcomExitStatement::TKcomExitStatement(TKcomManager* Manager)
{
    _Manager = Manager;
}

TKcomExitStatement::~TKcomExitStatement()
{
}

TParaStatement* TKcomExitStatement::Clone(void)
{
    return new TKcomExitStatement(_Manager);
}

string TKcomExitStatement::FirstToken(void) const
{
    return "exit";
}

void TKcomExitStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("exit");
    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKcomExitStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _Manager->Stop();

    TExecResult Result;
    Result.ExecStatus = TParaStatement::esReturn;

    return Result;
}



TKcomAfterStatement::TKcomAfterStatement(TKcomScript* Script)
{
    _Script = Script;
    _TimeToExecutionExpression = 0;
    _FunctionCallNode = 0;
}

TKcomAfterStatement::~TKcomAfterStatement()
{
    delete _FunctionCallNode;
    delete _TimeToExecutionExpression;
}

TParaStatement* TKcomAfterStatement::Clone(void)
{
    return new TKcomAfterStatement(_Script);
}

string TKcomAfterStatement::FirstToken(void) const
{
    return string("after");
}

void TKcomAfterStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe("after");
    Tokenizer->Next().MustBe("(");
    
    _TimeToExecutionExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);
    if (Tokenizer->LookAhead().Is("sec")) {
	Tokenizer->Next();
    }

    Tokenizer->Next().MustBe(")");

    Tokenizer->Next().MustBe("call");
    _FunctionCallNode = ExpressionParser->ParseFunctionCall(Tokenizer, SymbolTable);

    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKcomAfterStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long TimeToExecution = _TimeToExecutionExpression->Evaluate(SymbolTable).AsLong();
    
    _FunctionCallNode->EvaluateArguments(SymbolTable);
    _Script->AddDelayedTask(
	TMushDateTime::SecSinceEpoch() + TimeToExecution,
	_FunctionCallNode
    );

    return TExecResult();
}



TKcomComponentStatement::TKcomComponentStatement(void)
{
}

TKcomComponentStatement::~TKcomComponentStatement()
{
}

TParaStatement* TKcomComponentStatement::Clone(void)
{
    return new TKcomComponentStatement();
}

string TKcomComponentStatement::FirstToken(void) const
{
    return "component";
}

void TKcomComponentStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("component");
}

TParaStatement::TExecResult TKcomComponentStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return TExecResult();
}



TKcomSystemScriptMessenger::TKcomSystemScriptMessenger(TKcomManager* Manager)
: TParaObjectPrototype("kcom")
{
    _Manager = Manager;
}

TKcomSystemScriptMessenger::~TKcomSystemScriptMessenger()
{
}

TParaObjectPrototype* TKcomSystemScriptMessenger::Clone(void)
{
    return new TKcomSystemScriptMessenger(_Manager);
}

int TKcomSystemScriptMessenger::MethodIdOf(const string& MethodName)
{
    if (MethodName == "setTimeout") {
        return MethodId_SetTimeout;
    }
    else if (MethodName == "enableTimeout") {
        return MethodId_EnableTimeout;
    }
    else if (MethodName == "disableTimeout") {
        return MethodId_DisableTimeout;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKcomSystemScriptMessenger::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_SetTimeout:
        Result = SetTimeout(ArgumentList, ReturnValue);
	break;
      case MethodId_EnableTimeout:
        Result = EnableTimeout(ArgumentList, ReturnValue);
	break;
      case MethodId_DisableTimeout:
        Result = DisableTimeout(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TKcomSystemScriptMessenger::SetTimeout(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::setTimeout(int)", "too few argument[s]"
	);
    }

    if (! ArgumentList[0]->IsLong() || (ArgumentList[0]->AsLong() <= 0)) {
	throw TScriptException(
	    _InternalClassName + "::setTimeout(int)", "invalid argument[s]"
	);
    }

    int Timeout = ArgumentList[0]->AsLong();
    _Manager->GetEventProcessor()->SetTimeout(Timeout);

    return 1;
}

int TKcomSystemScriptMessenger::EnableTimeout(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _Manager->GetEventProcessor()->EnableTimeout();

    return 1;
}

int TKcomSystemScriptMessenger::DisableTimeout(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _Manager->GetEventProcessor()->DisableTimeout();

    return 1;
}



TKcomComponentScriptMessenger::TKcomComponentScriptMessenger(TKcomComponentAssembler* ComponentAssembler, TKcomEventProcessor* EventProcessor, TKcomRegistry* Registry)
: TParaObjectPrototype("component")
{
    _ComponentAssembler = ComponentAssembler;
    _EventProcessor = EventProcessor;
    _Registry = Registry; 
}

TKcomComponentScriptMessenger::~TKcomComponentScriptMessenger()
{
}

TParaObjectPrototype* TKcomComponentScriptMessenger::Clone(void)
{
    return new TKcomComponentScriptMessenger(
	_ComponentAssembler, _EventProcessor, _Registry
    );
}

void TKcomComponentScriptMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    _ComponentTypeName = ClassName;
    _ComponentName = this->ObjectName();

    if (ArgumentList.size() > 0) {
	_Host = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	_Control = ArgumentList[1]->AsString();
    }
    else {
	_Control = "null";
    }

    _ComponentAssembler->AddComponent(
	TKcomDeployment(
	    _ComponentName, _ComponentTypeName, _Host, _Control
	)
    );
}

int TKcomComponentScriptMessenger::InvokeMethodByName(const string& MethodName, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    TKcomEvent Event(MethodName);
    Event.SourceName() = "KcomScript";

    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	string Argument = ArgumentList[i]->AsString();
	if (ArgumentList[i]->IsObject()) {
	    Argument = ArgumentList[i]->AsObject()->ObjectName();
	}

	Event.ArgumentList().push_back(Argument);
    }

#ifdef DEBUG
    dumpfile << _ComponentName << ": send event: " << MethodName << endl;
#endif

    TKcomEventResponse EventResponse;
    try {
	_EventProcessor->EmitEvent(_ComponentName, Event, EventResponse);
    }
    catch (TKcomException &e) {
	throw TScriptException(
	    string("TKcomComponentScriptMessenger::InvokeMethodByName") +
	    "(\"" + MethodName + "\")",
	    "kcom exception: " + e.Message()
	);
    }

    if (EventResponse.IsError()) {
#if 0
	throw TScriptException(
	    _ComponentName + "." + Event.Name(), EventResponse.ReturnValue()
	);
#else
	cerr << "ERROR: " << _ComponentName + "." + Event.Name() << "(): ";
	cerr << EventResponse.ReturnValue();
	cerr << " (continue processing)" << endl;
#endif
    }
    else {
	ReturnValue = TParaValue(EventResponse.ReturnValue());
    }

    return 1;
}

int TKcomComponentScriptMessenger::GetPropertyByName(const std::string& PropertyName, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = TParaObjectPrototype::GetPropertyByName(PropertyName, ReturnValue);

    if (Result == 0) {
	string Path = TKcomComponent::PropertyRegistryPathOf(
	    _ComponentName, PropertyName
	);
	string Value;
	if (_Registry->GetValue(Path, Value)) {
	    ReturnValue = TParaValue(Value);
	    Result = 1;
	}
    }

    return Result;
}



TKcomRegistryScriptMessenger::TKcomRegistryScriptMessenger(TKcomRegistry* Registry)
: TParaObjectPrototype("Registry")
{
    _Registry = Registry;
}

TKcomRegistryScriptMessenger::~TKcomRegistryScriptMessenger()
{
}

TParaObjectPrototype* TKcomRegistryScriptMessenger::Clone(void)
{
    return new TKcomRegistryScriptMessenger(_Registry);
}

int TKcomRegistryScriptMessenger::MethodIdOf(const string& MethodName)
{
    if (MethodName == "getRegistry") {
        return MethodId_GetRegistry;
    }
    else if (MethodName == "setRegistry") {
	return MethodId_SetRegistry;
    }
    else if (MethodName == "saveRegistry") {
	return MethodId_SaveRegistry;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKcomRegistryScriptMessenger::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_GetRegistry:
        Result = GetRegistry(ArgumentList, ReturnValue);
	break;
      case MethodId_SetRegistry:
        Result = SetRegistry(ArgumentList, ReturnValue);
	break;
      case MethodId_SaveRegistry:
        Result = SaveRegistry(ArgumentList, ReturnValue);
	break;
      default:
	Result = 0;
    }
    
    return Result;
}

int TKcomRegistryScriptMessenger::GetRegistry(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::getRegistry()", "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::getRegistry()", "invalid argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value;
    try {
	_Registry->GetValue(Name, Value);
    }
    catch (TKcomException &e) {
	cerr << "ERROR: KCOM: unable to access registry: ";
	cerr << "empty value returned to '" << Name << "'" << endl;
	Value = "";
    }

    ReturnValue = TParaValue(Value);

    return 1;
}

int TKcomRegistryScriptMessenger::SetRegistry(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    _InternalClassName + "::setRegistry()", "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    _InternalClassName + "::setRegistry()", "invalid argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();

    try {
	_Registry->SetValue(Name, Value);
    }
    catch (TKcomException &e) {
	cerr << "ERROR: KCOM: unable to access registry: ";
	cerr << "writing skipped for value '" << Name << "'" << endl;
    }
    
    return 1;
}

int TKcomRegistryScriptMessenger::SaveRegistry(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _InternalClassName + "::" +
            "saveRegistry(string file_name, string base_path=\"/\")", 
	    "too few argument[s]"
	);
    }

    string FileName = ArgumentList[0]->AsString();
    string BasePath = "/";

    if (ArgumentList.size() > 1) {
	BasePath = ArgumentList[1]->AsString();
    }

    try {
        _Registry->SaveTo(FileName, BasePath);
    }
    catch (TKcomException &e) {
        throw TScriptException("Kcom Exception: " + e.Message());
    }
    
    return 1;
}
