/* dbms-MySQL.cc */
/* Created by Enomoto Sanshiro on 19 June 2002. */
/* Last updated by Enomoto Sanshiro on 28 September 2002. */


#include <sstream>
#include <string>
#include <map>
#include <mysql.h>
#include "KdbcManager.hh"
#include "KdbcDriver.hh"
#include "KdbcResult.hh"
#include "dbms-MySQL.hh"

using namespace std;


static TKdbcDriverCreator Creator("MySQL", new TKdbcDriver_MySQL());


MYSQL* TKdbcDriver_MySQL::_RootConnection = 0;

TKdbcDriver_MySQL::TKdbcDriver_MySQL(void)
{
}

TKdbcDriver_MySQL::~TKdbcDriver_MySQL()
{
}

TKdbcDriver* TKdbcDriver_MySQL::Clone(void)
{
    return new TKdbcDriver_MySQL();
}

TKdbcConnection* TKdbcDriver_MySQL::CreateConnection(const string& DatabaseName) throw(TKdbcException)
{
    if (_RootConnection == 0) {
	_RootConnection = new MYSQL;
	mysql_init(_RootConnection);
    }

    char* Host = "";
    char* UserName = "";
    char* Password = "";
    unsigned Port = 0;
    char* SocketName = 0; 
    unsigned Options = 0;

    MYSQL* Connection = mysql_real_connect(
	_RootConnection,
	Host, UserName, Password, 
	DatabaseName.c_str(), 
	Port, SocketName, Options
    );

    if (Connection == NULL) {
	string ErrorMessage = mysql_error(_RootConnection);
	if (*(ErrorMessage.end() - 1) == '\n') {
	    ErrorMessage.erase(ErrorMessage.end() - 1);
	}

	throw TKdbcException(
	    "TKdbcDriver_MySQL::CreateConnection()",
	    ErrorMessage + ": " + DatabaseName
	);
    }

    return new TKdbcConnection_MySQL(Connection);
}

TKdbcConnection* TKdbcDriver_MySQL::CreateConnection(const string& DatabaseName, const std::string& Host, const std::string& Port, const string& UserName, const string& Password) throw(TKdbcException)
{
    if (_RootConnection == 0) {
	_RootConnection = new MYSQL;
	mysql_init(_RootConnection);
    }

    unsigned PortNumber = 0;
    char* SocketName = 0; 
    unsigned Options = 0;

    istringstream PortNumberStream(Port);
    PortNumberStream >> PortNumber;

    MYSQL* Connection = mysql_real_connect(
	_RootConnection,
	Host.c_str(), UserName.c_str(), Password.c_str(), 
	DatabaseName.c_str(), 
	PortNumber, SocketName, Options
    );

    if (Connection == NULL) {
	string ErrorMessage = mysql_error(_RootConnection);
	if (*(ErrorMessage.end() - 1) == '\n') {
	    ErrorMessage.erase(ErrorMessage.end() - 1);
	}

	throw TKdbcException(
	    "TKdbcDriver_MySQL::CreateConnection()",
	    ErrorMessage + ": " + DatabaseName
	);
    }

    return new TKdbcConnection_MySQL(Connection);
}



TKdbcConnection_MySQL::TKdbcConnection_MySQL(MYSQL* Connection)
{
    _Connection = Connection;
}

TKdbcConnection_MySQL::~TKdbcConnection_MySQL()
{
    if (_Connection != 0) {
	mysql_close(_Connection);
    }
}

void TKdbcConnection_MySQL::Disconnect(void)
{
    if (_Connection != 0) {
	mysql_close(_Connection);
	_Connection = 0;
    }
}

TKdbcResult* TKdbcConnection_MySQL::ExecuteSql(const string& Statement) throw(TKdbcException)
{
    int QueryState = mysql_query(_Connection, Statement.c_str());
    if (QueryState != 0) {
	string ErrorMessage = mysql_error(_Connection);
	if (*(ErrorMessage.end() - 1) == '\n') {
	    ErrorMessage.erase(ErrorMessage.end() - 1);
	}
	throw TKdbcException(
	    "TKdbcConnection_MySQL::ExecuteSql()",
	    ErrorMessage + ": " + Statement
	);
    }

    MYSQL_RES* Result = mysql_store_result(_Connection);
    long NumberOfAffectedRows = mysql_affected_rows(_Connection);

    return new TKdbcResult_MySQL(Result, NumberOfAffectedRows);
}



TKdbcResult_MySQL::TKdbcResult_MySQL(MYSQL_RES* Result, long NumberOfAffectedRows)
{
    _Result = Result;
    _NumberOfAffectedRows = NumberOfAffectedRows;

    _IsQueryResult = (_Result != NULL);

    if (_IsQueryResult) {
	_NumberOfRows = mysql_num_rows(_Result);
	_NumberOfColumns = mysql_num_fields(_Result);
    }

    _CurrentRowDataRow = -1;
    _FieldIndexTable = 0;
}

TKdbcResult_MySQL::~TKdbcResult_MySQL()
{
    if (_Result != 0) {
	mysql_free_result(_Result);
    }

    delete _FieldIndexTable;
}

void TKdbcResult_MySQL::Destroy(void)
{
    if (_Result != 0) {
	mysql_free_result(_Result);
	_Result = 0;
    }
}

bool TKdbcResult_MySQL::Next(void)
{
    bool Result = TKdbcResult::Next();

    if (Result) {
	_CurrentRowData = mysql_fetch_row(_Result);
	_CurrentRowDataRow = _CurrentRow;
    }

    return Result;
}

string TKdbcResult_MySQL::FieldNameOf(long Column) throw(TKdbcException)
{
    if (Column >= _NumberOfColumns) {
	throw TKdbcException(
	    "TKdbcResult_MySQL::FieldNameOf()",
	    "invalid column number"
	);
    }

    return string(mysql_fetch_fields(_Result)[Column].name);
}

string TKdbcResult_MySQL::GetValueOf(long Row, long Column) throw(TKdbcException)
{
    if ((Row >= _NumberOfRows) || (Column >= _NumberOfColumns)) {
	throw TKdbcException(
	    "TKdbcResult_MySQL::GetValueOf()",
	    "invalid raw or column number"
	);
    }

    if (Row != _CurrentRowDataRow) {
	MYSQL_ROWS* Position = mysql_row_tell(_Result);
	_CurrentRowData = mysql_fetch_row(_Result);
	_CurrentRowDataRow = Row;
	mysql_row_seek(_Result, Position);
    }

    return _CurrentRowData[Column];
}

string TKdbcResult_MySQL::GetValueOf(long Row, const string& FieldName) throw(TKdbcException)
{
    if (_FieldIndexTable == 0) {
	_FieldIndexTable = new map<string, long>;
	MYSQL_FIELD* FieldList = mysql_fetch_fields(_Result);
	for (long Index = 0; Index < _NumberOfColumns; Index++) {
	    (*_FieldIndexTable)[FieldList[Index].name] = Index;
	}
    }

    if (Row >= _NumberOfRows) {
	throw TKdbcException(
	    "TKdbcResult_MySQL::GetValueOf()",
	    "invalid raw number"
	);
    }

    if (_FieldIndexTable->count(FieldName) == 0) {
	throw TKdbcException(
	    "TKdbcResult_MySQL::GetValueOf()",
	    "invalid field name: " + FieldName
	);
    }

    return GetValueOf(Row, (*_FieldIndexTable)[FieldName]);
}
