/* dbms-PostgreSQL.cc */
/* Created by Enomoto Sanshiro on 3 November 2000. */
/* Last updated by Enomoto Sanshiro on 3 November 2000. */


#include <sstream>
#include <libpq-fe.h>
#include "KdbcManager.hh"
#include "KdbcDriver.hh"
#include "KdbcResult.hh"
#include "dbms-PostgreSQL.hh"

using namespace std;


static TKdbcDriverCreator Creator("PostgreSQL", new TKdbcDriver_PostgreSQL());


TKdbcDriver_PostgreSQL::TKdbcDriver_PostgreSQL(void)
{
}

TKdbcDriver_PostgreSQL::~TKdbcDriver_PostgreSQL()
{
}

TKdbcDriver* TKdbcDriver_PostgreSQL::Clone(void)
{
    return new TKdbcDriver_PostgreSQL();
}

TKdbcConnection* TKdbcDriver_PostgreSQL::CreateConnection(const string& DatabaseName) throw(TKdbcException)
{
    char *Host = NULL;
    char *Port = NULL;
    char *Options = NULL;
    char *Tty = NULL;

    PGconn* PgConnection = PQsetdb(
	Host, Port, Options, Tty, DatabaseName.c_str()
    );

    if (PQstatus(PgConnection) == CONNECTION_BAD) {
	string ErrorMessage = PQerrorMessage(PgConnection);
	if (*(ErrorMessage.end() - 1) == '\n') {
	    ErrorMessage.erase(ErrorMessage.end() - 1);
	}

	throw TKdbcException(
	    "TKdbcDriver_PostgreSQL::CreateConnection()",
	    ErrorMessage + ": " + DatabaseName
	);
    }

    return new TKdbcConnection_PostgreSQL(PgConnection);
}

TKdbcConnection* TKdbcDriver_PostgreSQL::CreateConnection(const string& DatabaseName, const std::string& Host, const std::string& Port, const string& UserName, const string& Password) throw(TKdbcException)
{
    char *Options = NULL;
    char *Tty = NULL;

    PGconn* PgConnection = PQsetdbLogin(
	Host.c_str(), Port.c_str(), Options, Tty, DatabaseName.c_str(), 
	UserName.c_str(), Password.c_str()
    );

    if (PQstatus(PgConnection) == CONNECTION_BAD) {
	string ErrorMessage = PQerrorMessage(PgConnection);
	if (*(ErrorMessage.end() - 1) == '\n') {
	    ErrorMessage.erase(ErrorMessage.end() - 1);
	}

	throw TKdbcException(
	    "TKdbcDriver_PostgreSQL::CreateConnection()",
	    ErrorMessage + ": " + DatabaseName
	);
    }

    return new TKdbcConnection_PostgreSQL(PgConnection);
}



TKdbcConnection_PostgreSQL::TKdbcConnection_PostgreSQL(PGconn* PgConnection)
{
    _PgConnection = PgConnection;
}

TKdbcConnection_PostgreSQL::~TKdbcConnection_PostgreSQL()
{
    if (_PgConnection != 0) {
	PQfinish(_PgConnection);
    }
}

void TKdbcConnection_PostgreSQL::Disconnect(void)
{
    if (_PgConnection != 0) {
	PQfinish(_PgConnection);
	_PgConnection = 0;
    }
}

TKdbcResult* TKdbcConnection_PostgreSQL::ExecuteSql(const string& Statement) throw(TKdbcException)
{
    PGresult* Result = PQexec(_PgConnection, Statement.c_str());

    if (
	(PQresultStatus(Result) != PGRES_EMPTY_QUERY) &&
	(PQresultStatus(Result) != PGRES_COMMAND_OK) &&
	(PQresultStatus(Result) != PGRES_TUPLES_OK) &&
	(PQresultStatus(Result) != PGRES_COPY_IN) &&
	(PQresultStatus(Result) != PGRES_COPY_OUT)
    ){
	string ErrorMessage = PQerrorMessage(_PgConnection);
	if (*(ErrorMessage.end() - 1) == '\n') {
	    ErrorMessage.erase(ErrorMessage.end() - 1);
	}

	throw TKdbcException(
	    "TKdbcConnection_PostgreSQL::ExecuteSql()",
	    ErrorMessage + ": " + Statement
	);
    }

    return new TKdbcResult_PostgreSQL(Result);
}



TKdbcResult_PostgreSQL::TKdbcResult_PostgreSQL(PGresult* PgResult)
{
    _PgResult = PgResult;

    _IsQueryResult = (PQresultStatus(_PgResult) == PGRES_TUPLES_OK);
    if (_IsQueryResult) {
	_NumberOfRows = PQntuples(_PgResult);
	_NumberOfColumns = PQnfields(_PgResult);
    }
    else {
	istringstream ResultStream(PQcmdTuples(_PgResult));
	ResultStream >> _NumberOfAffectedRows;
    }
}

TKdbcResult_PostgreSQL::~TKdbcResult_PostgreSQL()
{
    if (_PgResult != 0) {
	PQclear(_PgResult);
    }
}

void TKdbcResult_PostgreSQL::Destroy(void)
{
    if (_PgResult != 0) {
	PQclear(_PgResult);
	_PgResult = 0;
    }
}

string TKdbcResult_PostgreSQL::FieldNameOf(long Column) throw(TKdbcException)
{
    if (Column >= _NumberOfColumns) {
	throw TKdbcException(
	    "TKdbcResult_PostgreSQL::FieldNameOf()",
	    "invalid column number"
	);
    }

    return PQfname(_PgResult, Column);
}

string TKdbcResult_PostgreSQL::GetValueOf(long Row, long Column) throw(TKdbcException)
{
    if ((Row >= _NumberOfRows) || (Column >= _NumberOfColumns)) {
	throw TKdbcException(
	    "TKdbcResult_PostgreSQL::GetValueOf()",
	    "invalid raw or column number"
	);
    }

    return PQgetvalue(_PgResult, Row, Column);
}

string TKdbcResult_PostgreSQL::GetValueOf(long Row, const string& FieldName) throw(TKdbcException)
{
    if (Row >= _NumberOfRows) {
	throw TKdbcException(
	    "TKdbcResult_PostgreSQL::GetValueOf()",
	    "invalid raw number"
	);
    }

    long Column = PQfnumber(_PgResult, FieldName.c_str());
    if (Column < 0) {
	throw TKdbcException(
	    "TKdbcResult_PostgreSQL::GetValueOf()",
	    "invalid field name: " + FieldName
	);
    }

    return PQgetvalue(_PgResult, Row, Column);    
}
