/* KiscAnalysisScript.cc */
/* Created by Enomoto Sanshiro on 17 May 2002. */
/* Last updated by Enomoto Sanshiro on 10 July 2009. */


#include <iostream>
#include <typeinfo>
#include "MushFileSystem.hh"
#include "ParaParser.hh"
#include "KiscNtuple.hh"
#include "KiscAnalysisScript.hh"


using namespace std;
using namespace kame;


TKiscAnalysisScript::TKiscAnalysisScript(void)
{
}

TKiscAnalysisScript::~TKiscAnalysisScript()
{
}

TParaObjectPrototypeTable* TKiscAnalysisScript::CreateObjectPrototypeTable(void)
{
    TParaObjectPrototypeTable* ObjectPrototypeTable;
    ObjectPrototypeTable = TParaParser::CreateObjectPrototypeTable();

    ObjectPrototypeTable->RegisterClass(new TKiscNtupleObject());

    return ObjectPrototypeTable;
}



TKiscAnalysisObject::TKiscAnalysisObject(const std::string& ClassName)
: TParaObjectPrototype(ClassName)
{
}

TKiscAnalysisObject::~TKiscAnalysisObject()
{
}

int TKiscAnalysisObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "title") {
        return MethodId_Title;
    }
    else if (MethodName == "setDelimiter") {
        return MethodId_SetDelimiter;
    }
    else if (MethodName == "setQuote") {
        return MethodId_SetQuote;
    }
    else if (MethodName == "load") {
        return MethodId_Load;
    }
    else if (MethodName == "save") {
        return MethodId_Save;
    }
    else if (MethodName == "propertyList") {
        return MethodId_PropertyList;
    }
    else if (MethodName == "addProperty") {
        return MethodId_AddProperty;
    }
    else if (MethodName == "getPropertyValueOf") {
        return MethodId_GetPropertyValueOf;
    }
    else if (MethodName == "getSpiltPropertyValueOf") {
        return MethodId_GetSpiltPropertyValueOf;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscAnalysisObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_Title) {
        Title(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_SetDelimiter) {
        SetDelimiter(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_SetQuote) {
        SetQuote(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_Load) {
        Load(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_Save) {
        Save(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_PropertyList) {
        PropertyList(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_AddProperty) {
        AddProperty(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_GetPropertyValueOf) {
        GetPropertyValueOf(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_GetSpiltPropertyValueOf) {
        GetSpiltPropertyValueOf(ArgumentList, ReturnValue);
    }
    else {
	return TParaObjectPrototype::InvokeMethod(
	    MethodId, ArgumentList, ReturnValue
	);
    }

    return 1;
}

void TKiscAnalysisObject::Title(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue(KameObject().Title());
}

void TKiscAnalysisObject::SetDelimiter(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    InternalClassName() + "::setDelimiter(string delimiter)", 
	    "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    InternalClassName() + "::setDelimiter(string delimiter)", 
	    "invalid argument[s]"
	);
    }

    KiscKameObject().SetDelimiter(ArgumentList[0]->AsString());
}

void TKiscAnalysisObject::SetQuote(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    InternalClassName() + "::setQuote(string delimiter)", 
	    "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    InternalClassName() + "::setQuote(string delimiter)", 
	    "invalid argument[s]"
	);
    }

    KiscKameObject().SetQuote(ArgumentList[0]->AsString());
}

void TKiscAnalysisObject::Load(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    InternalClassName() + "::Load(string file_name)", 
	    "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    InternalClassName() + "::Load(string file_name)", 
	    "invalid argument[s]"
	);
    }
    string FileName = ArgumentList[0]->AsString();

    string ObjectName;
    if (ArgumentList.size() > 1) {
	if (! ArgumentList[1]->IsString()) {
	    throw TScriptException(
		InternalClassName() + 
		"::Load(string repository_name, string object_name)", 
		"invalid argument[s]"
	    );
	}
	ObjectName = ArgumentList[1]->AsString();
    }

    string RepositoryName;
    if (! ObjectName.empty()) {
	RepositoryName = FileName;
    }
    else if (TMushFileAttribute(FileName).Extension() == "root") {
	throw TScriptException(
	    InternalClassName() + 
	    "::Load(string repository_name, string object_name)", 
	    "object name required for ROOT file"
	);
    }
    else {
	RepositoryName = "";
	ObjectName = FileName;
    }

    KiscKameObject().Load(RepositoryName, ObjectName);
}

void TKiscAnalysisObject::Save(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    InternalClassName() + "::Save(string file_name)", 
	    "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    InternalClassName() + "::Save(string file_name)", 
	    "invalid argument[s]"
	);
    }
    string FileName = ArgumentList[0]->AsString();

    string RepositoryName, ObjectName;
    if (ArgumentList.size() > 1) {
	if (! ArgumentList[1]->IsString()) {
	    throw TScriptException(
		InternalClassName() + 
		"::Save(string repository_name, string object_name)", 
		"invalid argument[s]"
	    );
	}
	RepositoryName = FileName;
	ObjectName = ArgumentList[1]->AsString();
    }
    else {
	RepositoryName = "";
	ObjectName = FileName;
    }

    KiscKameObject().Save(RepositoryName, ObjectName);
}

void TKiscAnalysisObject::PropertyList(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue(TParaListValue());
    TParaListValue& ListValue = ReturnValue.AsList();

    const vector<string>& PropertyNameList = KameObject().PropertyNameList();
    for (unsigned i = 0; i < PropertyNameList.size(); i++) {
	string Name = PropertyNameList[i];
	string Value = KameObject().Property(Name);
	ListValue.ValueOf(Name) = TParaValue(Value);
    }
}

void TKiscAnalysisObject::AddProperty(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    InternalClassName() + "addProperty(string name, string value)",
	    "too few argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();

    KameObject().Property(Name) = Value;
}

void TKiscAnalysisObject::GetPropertyValueOf(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    InternalClassName() + "getPropertyValueOf(string name)",
	    "too few argument[s]"
	);
    }

    string Name = ArgumentList[0]->AsString();
    string Value = KameObject().Property(Name);

    ReturnValue = TParaValue(Value);
}

void TKiscAnalysisObject::GetSpiltPropertyValueOf(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    InternalClassName() + "getPropertyValueOf(string name)",
	    "too few argument[s]"
	);
    }

    string Value = KameObject().Property(ArgumentList[0]->AsString());
    vector<string> SpiltValueList;
    
    KiscKameObject().TokenizeLine(Value, SpiltValueList);

    ReturnValue = TParaValue(TParaListValue());
    TParaListValue& ListValue = ReturnValue.AsList();

    for (unsigned i = 0; i < SpiltValueList.size(); i++) {
	ListValue.AppendValue(TParaValue(SpiltValueList[i]));
    }
}



TKiscNtupleObject::TKiscNtupleObject(void)
: TKiscAnalysisObject("Ntuple")
{
}

TKiscNtupleObject::~TKiscNtupleObject()
{
}

TParaObjectPrototype* TKiscNtupleObject::Clone(void)
{
    return new TKiscNtupleObject();
}

TKameObject& TKiscNtupleObject::KameObject(void)
{
    return _Ntuple;
}

TKiscKameObject& TKiscNtupleObject::KiscKameObject(void)
{
    return _Ntuple;
}

int TKiscNtupleObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "fill") {
        return MethodId_Fill;
    }
    else if (MethodName == "clear") {
        return MethodId_Clear;
    }
    else if (MethodName == "getColumnOf") {
        return MethodId_GetColumnOf;
    }
    else if (MethodName == "getRowOf") {
        return MethodId_GetRowOf;
    }
    else if (MethodName == "numberOfColumns") {
        return MethodId_NumberOfColumns;
    }
    else if (MethodName == "numberOfRows") {
        return MethodId_NumberOfRows;
    }
    else if (MethodName == "columnNameOf") {
        return MethodId_ColumnNameOf;
    }
    else if (MethodName == "columnIndexOf") {
        return MethodId_ColumnIndexOf;
    }
    else if (MethodName == "setColumnName") {
        return MethodId_SetColumnName;
    }

    return TKiscAnalysisObject::MethodIdOf(MethodName);
}

int TKiscNtupleObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_Fill) {
        Fill(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_Clear) {
        Clear(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_GetColumnOf) {
        GetColumnOf(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_GetRowOf) {
        GetRowOf(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_NumberOfColumns) {
        NumberOfColumns(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_NumberOfRows) {
        NumberOfRows(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_ColumnNameOf) {
        ColumnNameOf(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_ColumnIndexOf) {
        ColumnIndexOf(ArgumentList, ReturnValue);
    }
    else if (MethodId == MethodId_SetColumnName) {
        SetColumnName(ArgumentList, ReturnValue);
    }
    else {
	return TKiscAnalysisObject::InvokeMethod(
	    MethodId, ArgumentList, ReturnValue
	);
    }

    return 1;
}

TParaValue& TKiscNtupleObject::EvaluateOperator(TParaOperator* Operator, TParaValue& LeftValue, TParaValue& RightValue, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    if (Operator->Name() == "ArrayReference") {
	TParaExpression* IndexExpression = Operator->InternalExpression();
	TParaValue& IndexValue = IndexExpression->Evaluate(SymbolTable);
	vector<TParaValue*> ArgumentList;
	ArgumentList.push_back(&IndexValue);
	this->GetRowOf(ArgumentList, Result);
	return Result;
    }

    if (Operator->Name() == "TableReference") {
	TParaExpression* IndexExpression = Operator->InternalExpression();
	TParaValue& IndexValue = IndexExpression->Evaluate(SymbolTable);
	vector<TParaValue*> ArgumentList;
	ArgumentList.push_back(&IndexValue);
	this->GetColumnOf(ArgumentList, Result);
	return Result;
    }
    
    return TParaObjectPrototype::EvaluateOperator(
	Operator, LeftValue, RightValue, SymbolTable, Result
    );
}

void TKiscNtupleObject::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() > 0) {
	TParaValue ReturnValue;
	Load(ArgumentList, ReturnValue);
    }
}

void TKiscNtupleObject::GetColumnOf(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Ntuple::GetColumnOf(...)", "too few argument[s]"
	);
    }

    if ((ArgumentList.size() == 1) && ArgumentList[0]->IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();

	TParaValue Result;
	vector<TParaValue*> InputArgumentList;
	InputArgumentList.push_back(&InputList[0]);

	for (unsigned i = 0; i < InputList.size(); i++) {
	    InputArgumentList[0] = &InputList[i];
	    this->GetColumnOf(InputArgumentList, Result);
	    ResultList.push_back(Result);
	}
	return;
    }

    unsigned ColumnIndex;
    if (ArgumentList[0]->IsLong()) {
	ColumnIndex = ArgumentList[0]->AsLong();
    }
    else {
	string ColumnName = ArgumentList[0]->AsString();
	try {
	    ColumnIndex = _Ntuple.ColumnIndexOf(ColumnName);
	}
	catch (TKameException &e) {
	    throw TScriptException(
		"Ntuple::GetColumnOf(string)", 
		"no such column: " + ColumnName
	    );
	}
    }

    if ((ColumnIndex < 0) || (ColumnIndex >= _Ntuple.NumberOfColumns())) {
	throw TScriptException(
	    "Ntuple::GetColumnOf(int)", "invalid column index"
	);
    }

    const vector<TKameVariant>& Source = _Ntuple.GetColumn(ColumnIndex);
    vector<TParaValue> Destination(_Ntuple.NumberOfRows());
    for (unsigned RowIndex = 0; RowIndex < Source.size(); RowIndex++) {
	if (Source[RowIndex].IsLong()) {
	    Destination[RowIndex] = TParaValue((long) Source[RowIndex]);
	}
	else if (Source[RowIndex].IsDouble()) {
	    Destination[RowIndex] = TParaValue((double) Source[RowIndex]);
	}
	else if (Source[RowIndex].IsString()) {
	    Destination[RowIndex] = TParaValue((string) Source[RowIndex]);
	}
	else if (Source[RowIndex].IsUnknown()) {
	    const TKameUnknown& KameUnknown = Source[RowIndex].AsUnknown();
	    if (typeid(KameUnknown) == typeid(TKameUnknownParaValue)) {
		Destination[RowIndex] = (
		    ((TKameUnknownParaValue&) KameUnknown).ParaValue()
		);
	    }
	    else {
		throw TScriptException(
		    "Ntuple::GetColumnOf(int)", 
		    "unknown unknown object: " + (string) Source[RowIndex]
		);
	    }
	}
    }

    ReturnValue = TParaValue(TParaListValue(Destination));
}

void TKiscNtupleObject::Fill(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Row = _Ntuple.NumberOfRows();
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	if (ArgumentList[i]->IsLong()) {
	    _Ntuple[Row][i] = ArgumentList[i]->AsLong();
	}
	else if (ArgumentList[i]->IsDouble()) {
	    _Ntuple[Row][i] = ArgumentList[i]->AsDouble();
	}
	else {
	    _Ntuple[Row][i] = ArgumentList[i]->AsString();
	}
    }
}

void TKiscNtupleObject::Clear(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _Ntuple.Clear();
}

void TKiscNtupleObject::GetRowOf(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Ntuple::GetRowOf(int/list)", "too few argument[s]"
	);
    }

    if ((ArgumentList.size() == 1) && ArgumentList[0]->IsList()) {
	ReturnValue = TParaValue(TParaListValue());
	vector<TParaValue>& InputList = ArgumentList[0]->AsValueList();
	vector<TParaValue>& ResultList = ReturnValue.AsValueList();

	TParaValue Result;
	vector<TParaValue*> InputArgumentList;
	InputArgumentList.push_back(&InputList[0]);

	for (unsigned i = 0; i < InputList.size(); i++) {
	    InputArgumentList[0] = &InputList[i];
	    this->GetRowOf(InputArgumentList, Result);
	    ResultList.push_back(Result);
	}
	return;
    }

    if (! ArgumentList[0]->IsLong()) {
	throw TScriptException(
	    "Ntuple::GetRowOf(int)", "invalid argument[s]"
	);
    }
    unsigned RowIndex = ArgumentList[0]->AsLong();
    if ((RowIndex < 0) || (RowIndex >= _Ntuple.NumberOfRows())) {
	throw TScriptException(
	    "Ntuple::GetRowOf(int)", "invalid row index"
	);
    }

    unsigned NumberOfColumns = _Ntuple.NumberOfColumns();
    vector<TParaValue> Row(NumberOfColumns);
    for (unsigned Index = 0; Index < NumberOfColumns; Index++) {
	TKameVariant Value = _Ntuple[RowIndex][Index];
	if (Value.IsLong()) {
	    Row[Index] = TParaValue((long) Value);
	}
	else if (Value.IsDouble()) {
	    Row[Index] = TParaValue((double) Value);
	}
	else if (Value.IsString()) {
	    Row[Index] = TParaValue((string) Value);
	}
	else if (Value.IsUnknown()) {
	    const TKameUnknown& KameUnknown = Value.AsUnknown();
	    if (typeid(KameUnknown) == typeid(TKameUnknownParaValue)) {
		Row[Index] = (
		    ((TKameUnknownParaValue&) KameUnknown).ParaValue()
		);
	    }
	    else {
		throw TScriptException(
		    "Ntuple::GetRowOf(int)", 
		    "unknown unknown object: " + (string) Value
		);
	    }
	}
    }

    ReturnValue = TParaValue(TParaListValue(Row));
}

void TKiscNtupleObject::NumberOfColumns(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue((long) _Ntuple.NumberOfColumns());
}

void TKiscNtupleObject::NumberOfRows(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue((long) _Ntuple.NumberOfRows());
}

void TKiscNtupleObject::ColumnNameOf(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Ntuple::columnNameOf(int column_index)", "too few argument[s]"
	);
    }

    unsigned ColumnIndex = ArgumentList[0]->AsLong();
    if ((ColumnIndex < 0) || (ColumnIndex >= _Ntuple.NumberOfColumns())) {
	throw TScriptException(
	    "Ntuple::columnNameOf(int column_index)", "invalid column index"
	);
    }

    ReturnValue = TParaValue(_Ntuple.ColumnNameOf(ColumnIndex));
}

void TKiscNtupleObject::ColumnIndexOf(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Ntuple::ColumnIndexOf(string column_name)", "too few argument[s]"
	);
    }

    string ColumnName = ArgumentList[0]->AsString();
    long ColumnIndex;
    try {
	ColumnIndex = _Ntuple.ColumnIndexOf(ColumnName);
    }
    catch (TKameException &e) {
	throw TScriptException(
	    "Ntuple::columnIndexOf()", "no such column: " + ColumnName
	);
    }

    ReturnValue = TParaValue(ColumnIndex);
}

void TKiscNtupleObject::SetColumnName(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	_Ntuple.SetColumnName(i, ArgumentList[i]->AsString());
    }
}
