/* KiscConfigFile.cc */
/* Created by Enomoto Sanshiro on 18 July 2001 */
/* Last updated by Enomoto Sanshiro on 18 July 2001. */



#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include "ParaTokenizer.hh"
#include "ParaValue.hh"
#include "KiscConfigFile.hh"

using namespace std;


TKiscConfigFile::TKiscConfigFile(const string& FileName)
{
    _FileName = FileName;
    _IsConstructed = false;
}

TKiscConfigFile::~TKiscConfigFile()
{
}

void TKiscConfigFile::SetTargetSection(const string& SectionName, const string& Qualifier)
{
    if (! _IsConstructed) {
	_IsConstructed = true;
	Construct();
    }

    _SectionName = SectionName;
    _Qualifier = Qualifier;
}

TParaValue TKiscConfigFile::GetValueOf(const string& EntryName) throw(TScriptException)
{
    const vector<TParaValue>& ValueList = GetValueListOf(EntryName);

    if (ValueList.empty()) {
	throw TScriptException(
	    "unable to find entry '" + _SectionName + "/" + EntryName + 
	    "' in file '" + _FileName + "'"
	);
    }

    return ValueList[0];
}

TParaValue TKiscConfigFile::GetValueOfWithDefault(const string& EntryName, const string& DefaultValue) throw(TScriptException)
{
    const vector<TParaValue>& ValueList = GetValueListOf(EntryName);
    if (! ValueList.empty()) {
	return ValueList[0];
    }
    else {
	return TParaValue(DefaultValue);
    }
}

const vector<TParaValue>& TKiscConfigFile::GetValueListOf(const string& EntryName) throw(TScriptException)
{
    if (! _IsConstructed) {
	_IsConstructed = true;
	Construct();
    }

    string EntryPath = _SectionName + "/" + EntryName;
    string QualifiedPath = EntryPath + "[" + _Qualifier + "]";
    if (_EntryTable.count(QualifiedPath) > 0) {
	return _EntryTable[QualifiedPath];
    }
    else if (_EntryTable.count(EntryPath) > 0) {
	return _EntryTable[EntryPath];
    }
    else {
	return _EntryTable[""];
    }
}

void TKiscConfigFile::DumpTo(ostream& os, const string& Prefix)
{
    if (! _IsConstructed) {
	_IsConstructed = true;
	Construct();
    }

    map<string, vector<TParaValue> >::iterator EntryIterator;
    for (
	EntryIterator = _EntryTable.begin();
	EntryIterator != _EntryTable.end();
	EntryIterator++
    ){
	string EntryName = EntryIterator->first;
	if ((! EntryName.empty()) && EntryName[0] == '/') {
	    EntryName.erase(0, 1);
	}

	os << Prefix;
	os << EntryName << ":";
	vector<TParaValue>& ValueList = EntryIterator->second;
	for (unsigned i = 0; i < ValueList.size(); i++) {
	    os << " " << ValueList[i].AsString();
	}
	os << endl;
    }
}



TKiscBasicConfigFile::TKiscBasicConfigFile(const string& FileName)
: TKiscConfigFile(FileName)
{
}

TKiscBasicConfigFile::~TKiscBasicConfigFile()
{
}

void TKiscBasicConfigFile::Construct(void) throw(TScriptException)
{
    ifstream ScriptFile(_FileName.c_str());
    if (! ScriptFile) {
	throw TScriptException(
	    "TKiscBasicConfigFile::Construct()",
	    "unable to open file: " + _FileName
	);
    }

    string Line;
    int LineNumber = 0;
    TParaCxxTokenTable TokenTable;
    while (getline(ScriptFile, Line, '\n')) {
	istringstream LineStream(Line);
	LineNumber++;

	TParaTokenizer Tokenizer(LineStream, &TokenTable);
	try {
	    ParseLine(Tokenizer);
	}
	catch (TScriptException &e) {
	    ostringstream Message;
	    Message << "line " << LineNumber << ": " << e.Message();
	    throw TScriptException(Message.str());
	}
    }
}

void TKiscBasicConfigFile::ParseLine(TParaTokenizer& Tokenizer) throw(TScriptException)
{
    if (Tokenizer.LookAhead().IsEmpty()) {
	;
    }
    else if (Tokenizer.LookAhead().AsString()[0] == '#') {
	;
    }
    else if (Tokenizer.LookAhead().Is("[")) {
	Tokenizer.Next().MustBe("[");
	_CurrentSection = Tokenizer.Next().RemoveQuotation('"').AsString();
	Tokenizer.Next().MustBe("]");
	if (! Tokenizer.LookAhead().IsEmpty()) {
	    throw TScriptException(
		"unexpected token: " + Tokenizer.Next().AsString()
	    );
	}
    }
    else {
	string EntryName = Tokenizer.Next().AsString();
	string Qualifier;
	if (Tokenizer.LookAhead().Is("[")) {
	    Tokenizer.Next().MustBe("[");
	    Qualifier = Tokenizer.Next().RemoveQuotation('"').AsString();
	    Tokenizer.Next().MustBe("]");
	}
	Tokenizer.Next().MustBe("=");
	
	string Path = _CurrentSection + "/" + EntryName;
	if (! Qualifier.empty()) {
	    Path += "[" + Qualifier + "]";
	}	
	vector<TParaValue>& Entry = _EntryTable[Path];

	TParaToken Token;
	while (! ((Token = Tokenizer.Next()).IsEmpty() || Token.Is("#"))) {
	    string StringValue = Token.RemoveQuotation('"').AsString();
	    Entry.push_back(TParaValue(StringValue));
	}
    }
}
