/* KiscDatabaseScript.cc */
/* Created by Enomoto Sanshiro on 5 December 2001. */
/* Last updated by Enomoto Sanshiro on 5 December 2001. */


#include <string>
#include <deque>
#include "ParaParser.hh"
#include "KdbcManager.hh"
#include "KiscDatabaseScript.hh"

using namespace std;


TKiscDatabaseScript::TKiscDatabaseScript(void)
{
    _ResultStack = new TKiscDbQueryResultStack();
}

TKiscDatabaseScript::~TKiscDatabaseScript()
{
    delete _ResultStack;
}

TParaTokenTable* TKiscDatabaseScript::CreateTokenTable(void)
{
    TParaTokenTable* TokenTable;
    TokenTable = TParaParser::CreateTokenTable();

    TokenTable->AddKeyword("sql");

    return TokenTable;
}

TParaObjectPrototypeTable* TKiscDatabaseScript::CreateObjectPrototypeTable(void)
{
    TParaObjectPrototypeTable* ObjectPrototypeTable;
    ObjectPrototypeTable = TParaParser::CreateObjectPrototypeTable();

    ObjectPrototypeTable->RegisterClass(new TKiscDatabaseObject());
    ObjectPrototypeTable->RegisterClass(new TKiscDbQueryResultObject());

    return ObjectPrototypeTable;
}

TParaOperatorTable* TKiscDatabaseScript::CreateOperatorTable(void)
{
    TParaOperatorTable* OperatorTable;
    OperatorTable = TParaParser::CreateOperatorTable();

    OperatorTable->AddPrepositionalOperator(
	new TKiscDbFieldAccessOperator(_ResultStack)
    );

    return OperatorTable;
}

TParaStatementTable* TKiscDatabaseScript::CreateStatementTable(void)
{
    TParaStatementTable* StatementTable;
    StatementTable = TParaParser::CreateStatementTable();

    StatementTable->AddStatement(new TKiscDbSqlStatement(_ResultStack));

    return StatementTable;
}



TKiscDbQueryResultStack::TKiscDbQueryResultStack(void)
{
}

TKiscDbQueryResultStack::~TKiscDbQueryResultStack()
{
}
    
void TKiscDbQueryResultStack::PushResult(TKdbcResult* QueryResult)
{
    _ResultStack.push_back(QueryResult);
}

void TKiscDbQueryResultStack::PopResult(void)
{
    _ResultStack.pop_back();
}

TKdbcResult* TKiscDbQueryResultStack::GetLastResult(void)
{
    if (_ResultStack.empty()) {
	return 0;
    }

    return _ResultStack.back();
}



TKiscDatabaseObject::TKiscDatabaseObject(void)
: TParaObjectPrototype("Database")
{
    _Driver = 0;
    _Connection = 0;
}

TKiscDatabaseObject::~TKiscDatabaseObject()
{
}

TParaObjectPrototype* TKiscDatabaseObject::Clone(void)
{
    return new TKiscDatabaseObject();
}

void TKiscDatabaseObject::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "Database::Database(string DriverName, string DatabaseName)",
	    "too few argument[s]"
	);
    }

    _DriverName = ArgumentList[0]->AsString();
    _DatabaseName = ArgumentList[1]->AsString();

    if (ArgumentList.size() >= 3) {
	_HostName = ArgumentList[2]->AsString();
    }
    if (ArgumentList.size() >= 4) {
	_PortNumber = ArgumentList[3]->AsString();
    }

    try {
	_Driver = TKdbcManager::GetInstance()->CreateDriver(_DriverName);

	if (_HostName.empty()) {
	    _Connection = _Driver->CreateConnection(_DatabaseName);
	}
    }
    catch (TKdbcException &e) {
	throw TScriptException(
	    "Database::Database(string DriverName, string DatabaseName)",
	    "unable to connect to the database: " + e.Message()
	);
    }
}

void TKiscDatabaseObject::Destruct(void) throw(TScriptException)
{
    delete _Connection;
    delete _Driver;
}

int TKiscDatabaseObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "login") {
        return MethodId_Login;
    }
    else if (MethodName == "executeSql") {
        return MethodId_ExecuteSql;
    }
    else if (MethodName == "getValueOf") {
        return MethodId_GetValueOf;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscDatabaseObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_Login) {
        ReturnValue = Login(ArgumentList);
    }
    else if (MethodId == MethodId_ExecuteSql) {
        ReturnValue = ExecuteSql(ArgumentList);
    }
    else if (MethodId == MethodId_GetValueOf) {
        ReturnValue = GetValueOf(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKiscDatabaseObject::Login(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_HostName.empty()) {
	// already connected
	return TParaValue((long) 0);
    }

    if (_Connection != 0) {
	throw TScriptException(
	    "Database::login(string userName, string password)",
	    "connection already established"
	);
    }

    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "Database::login(string user_name, string password)",
	    "too few argument[s]"
	);
    }

    string UserName = ArgumentList[0]->AsString();
    string Password = ArgumentList[1]->AsString();

    try {
	_Connection = _Driver->CreateConnection(
	    _DatabaseName, _HostName, _PortNumber, UserName, Password
	);
    }
    catch (TKdbcException &e) {
	throw TScriptException(
	    "Database::login()", 
	    "unable to create connection to the database: " + e.Message()
	);
    }

    return TParaValue((long) 0);
}

TParaValue TKiscDatabaseObject::ExecuteSql(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Database::executeSql(string query_text)", "too few argument[s]"
	);
    }

    string QueryText = ArgumentList[0]->AsString();
    TKdbcResult* QueryResult = ExecuteKdbcSql(QueryText);
    
    TParaObjectPrototype* Result = new TKiscDbQueryResultObject(QueryResult);
    TParaValue* ResultValue = new TParaValue(Result);

    return TParaValue(ResultValue);
}

TParaValue TKiscDatabaseObject::GetValueOf(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Database::getValueOf(string query_text)", "too few argument[s]"
	);
    }

    string QueryText = ArgumentList[0]->AsString();
    TKdbcResult* QueryResult = ExecuteKdbcSql(QueryText);

    if (QueryResult->NumberOfRows() != 1) {
	string Message;
	if (QueryResult->NumberOfRows() == 0) {
	    Message = "no matching result: " + QueryText;
	}
	else {
	    Message = "too many query results: " + QueryText;
	}
	delete QueryResult;
	throw TScriptException(
	    "Database::getValueOf(string query_text)", Message
	);
    }

    if (QueryResult->NumberOfColumns() != 1) {
	delete QueryResult;
	throw TScriptException(
	    "Database::getValueOf(string query_text)", 
	    "bad number of result columns"
	);
    }

    QueryResult->Next();
    string Result = QueryResult->GetString(0);

    return TParaValue(Result);
}

TKdbcResult* TKiscDatabaseObject::ExecuteKdbcSql(const string& Query) throw(TScriptException)
{
    if (_Connection == 0) {
	throw TScriptException("database is not connected");
    }

    TKdbcResult* QueryResult = 0;
    try {
	QueryResult = _Connection->ExecuteSql(Query);
    }
    catch (TKdbcException &e) {
	throw TScriptException(e.Message());
    }

    return QueryResult;
}



TKiscDbQueryResultObject::TKiscDbQueryResultObject(TKdbcResult* QueryResult)
: TParaObjectPrototype("QueryResult")
{
    _QueryResult = QueryResult;
}

TKiscDbQueryResultObject::~TKiscDbQueryResultObject()
{
}

TParaObjectPrototype* TKiscDbQueryResultObject::Clone(void)
{
    return new TKiscDbQueryResultObject(0);
}

void TKiscDbQueryResultObject::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    throw TScriptException(
	"QueryResult::QueryResult()",
	"direct object creation is prohibited"
    );
}

void TKiscDbQueryResultObject::Destruct(void) throw(TScriptException)
{
    delete _QueryResult;
}

int TKiscDbQueryResultObject::MethodIdOf(const string& MethodName)
{
    if (MethodName == "numberOfRows") {
	return MethodId_NumberOfRows;
    }
    else if (MethodName == "numberOfColumns") {
	return MethodId_NumberOfColumns;
    }
    else if (MethodName == "numberOfAffectedRows") {
	return MethodId_NumberOfAffectedRows;
    }
    else if (MethodName == "fieldNameOf") {
	return MethodId_FieldNameOf;
    }
    else if (MethodName == "next") {
	return MethodId_Next;
    }
    else if (MethodName == "getAt") {
	return MethodId_GetAt;
    }

    return TParaObjectPrototype::MethodIdOf(MethodName);
}

int TKiscDbQueryResultObject::InvokeMethod(int MethodId, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (MethodId == MethodId_NumberOfRows) {
        ReturnValue = NumberOfRows(ArgumentList);
    }
    else if (MethodId == MethodId_NumberOfColumns) {
        ReturnValue = NumberOfColumns(ArgumentList);
    }
    else if (MethodId == MethodId_NumberOfAffectedRows) {
        ReturnValue = NumberOfAffectedRows(ArgumentList);
    }
    else if (MethodId == MethodId_FieldNameOf) {
        ReturnValue = FieldNameOf(ArgumentList);
    }
    else if (MethodId == MethodId_Next) {
        ReturnValue = Next(ArgumentList);
    }
    else if (MethodId == MethodId_GetAt) {
        ReturnValue = GetAt(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKiscDbQueryResultObject::NumberOfRows(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_QueryResult == 0) {
	throw TScriptException(
	    "QueryResult::numberOfRows()", "empty result"
	);
    }

    long Result = _QueryResult->NumberOfRows();

    return TParaValue(Result);
}

TParaValue TKiscDbQueryResultObject::NumberOfColumns(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_QueryResult == 0) {
	throw TScriptException(
	    "QueryResult::NumberOfColumns()", "empty result"
	);
    }

    long Result = _QueryResult->NumberOfColumns();

    return TParaValue(Result);
}

TParaValue TKiscDbQueryResultObject::NumberOfAffectedRows(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_QueryResult == 0) {
	throw TScriptException(
	    "QueryResult::numberOfAffectedRows()", "empty result"
	);
    }

    long Result = _QueryResult->NumberOfAffectedRows();

    return TParaValue(Result);
}

TParaValue TKiscDbQueryResultObject::FieldNameOf(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_QueryResult == 0) {
	throw TScriptException(
	    "QueryResult::FieldNameOf(int ColumnIndex)", "empty result"
	);
    }

    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "QueryResult::FieldNameOf(int ColumnIndex)",
	    "too few argument[s]"
	);
    }
    if (! ArgumentList[0]->IsLong()) {
	throw TScriptException(
	    "QueryResult::FieldNameOf(int ColumnIndex)",
	    "invalid argument[s]"
	);
    }

    int ColumnIndex = ArgumentList[0]->AsLong();
    string Result = _QueryResult->FieldNameOf(ColumnIndex);

    return TParaValue(Result);
}

TParaValue TKiscDbQueryResultObject::Next(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_QueryResult == 0) {
	throw TScriptException(
	    "QueryResult::next()", "empty result"
	);
    }

    long Result = _QueryResult->Next();

    return TParaValue(Result);
}

TParaValue TKiscDbQueryResultObject::GetAt(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_QueryResult == 0) {
	throw TScriptException(
	    "QueryResult::getAt()", "empty result"
	);
    }

    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "QueryResult::getAt(string FieldName)",
	    "too few argument[s]"
	);
    }

    string Value;
    if (ArgumentList[0]->IsLong()) {
	int ColumnIndex = ArgumentList[0]->AsLong();
	Value = _QueryResult->GetString(ColumnIndex);
    }
    else if (ArgumentList[0]->IsString()) {
	string FieldName = ArgumentList[0]->AsString();
	Value = _QueryResult->GetString(FieldName);
    }
    else {
	throw TScriptException(
	    "QueryResult::getAt(string FieldName)",
	    "invalid argument[s]"
	);
    }

    return TParaValue(Value);
}



TKiscDbSqlStatement::TKiscDbSqlStatement(TKiscDbQueryResultStack* ResultStack)
{
    _ResultStack = ResultStack;

    _DbExpression = 0;
    _QueryExpression = 0;

    _Statement = 0;
    _ElseStatement = 0;
}

TKiscDbSqlStatement::~TKiscDbSqlStatement()
{
    delete _ElseStatement;
    delete _Statement;
    delete _QueryExpression;
    delete _DbExpression;
}

TParaStatement* TKiscDbSqlStatement::Clone(void)
{
    return new TKiscDbSqlStatement(_ResultStack);
}

string TKiscDbSqlStatement::FirstToken(void) const
{
    return string("sql");
}

void TKiscDbSqlStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe("sql");

    Tokenizer->Next().MustBe("[");
    _DbExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);
    Tokenizer->Next().MustBe("]");

    _QueryExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);

    if (Tokenizer->LookAhead().IsNot(";")) {
	_Statement = StatementParser->Parse(Tokenizer, SymbolTable);
    }
    else {
	Tokenizer->Next();
    }

    if (Tokenizer->LookAhead().Is("else")) {
	Tokenizer->Next();
	_ElseStatement = StatementParser->Parse(Tokenizer, SymbolTable);
    }
}

TParaStatement::TExecResult TKiscDbSqlStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& DbValue = _DbExpression->Evaluate(SymbolTable);
    if (! DbValue.IsObject("Database")) {
	throw TScriptException("sql: 'Database' object is expexted");
    }
    TKiscDatabaseObject* DbObject = (TKiscDatabaseObject*) DbValue.AsObject();

    TParaValue& QueryTextValue = _QueryExpression->Evaluate(SymbolTable);
    if (! QueryTextValue.IsString()) {
	throw TScriptException("sql: query string is expected");
    }
    string QueryText = _QueryExpression->Evaluate(SymbolTable).AsString();

    TKdbcResult* QueryResult = 0;
    try {
	QueryResult = DbObject->ExecuteKdbcSql(QueryText);
    }
    catch (TScriptException) {
	throw;
    }

    TParaStatement::TExecResult StatementResult;

    if ((QueryResult->NumberOfRows() > 0) && (_Statement == 0)) {
	delete QueryResult;
	return StatementResult;
    }
    if (QueryResult->NumberOfRows() == 0) {
	if (_ElseStatement != 0) {
	    StatementResult = _ElseStatement->Execute(SymbolTable);
	}

	delete QueryResult;
	return StatementResult;
    }

    _ResultStack->PushResult(QueryResult);
    try {
	while (QueryResult->Next()) {
	    StatementResult = _Statement->Execute(SymbolTable);
	    
	    if (StatementResult.ExecStatus == TParaStatement::esBreak) {
		StatementResult.ExecStatus = esNormal;
		break;
	    }
	    if (StatementResult.ExecStatus == TParaStatement::esContinue) {
		StatementResult.ExecStatus = esNormal;
		continue;
	    }
	    if (StatementResult.ExecStatus == TParaStatement::esReturn) {
		break;
	    }
	}
    }
    catch (TScriptException &e) {
	_ResultStack->PopResult();
	delete QueryResult;
	throw;
    }
    _ResultStack->PopResult();

    delete QueryResult;

    return StatementResult;
}



TKiscDbFieldAccessOperator::TKiscDbFieldAccessOperator(TKiscDbQueryResultStack* ResultStack)
{
    _ResultStack = ResultStack;
}

TKiscDbFieldAccessOperator::~TKiscDbFieldAccessOperator()
{
}

TParaOperator* TKiscDbFieldAccessOperator::Clone(void) const
{
    return new TKiscDbFieldAccessOperator(_ResultStack);
}

string TKiscDbFieldAccessOperator::Symbol(void) const
{
    return string("@");
}

string TKiscDbFieldAccessOperator::Name(void) const
{
    return string("DbFieldAccess");
}

void TKiscDbFieldAccessOperator::Parse(TParaTokenizer* Tokenizer, TParaExpressionParser* ExpressionParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next();

    string FieldName = Tokenizer->Next().RemoveQuotation('"').AsString();
    TParaToken FieldNameToken(
	'"' + FieldName + '"', TParaToken::ttQuote, 0, 
	Tokenizer->LineNumber()
    );

    Tokenizer->Unget(FieldNameToken);
}

TParaValue& TKiscDbFieldAccessOperator::Evaluate(TParaValue& Left, TParaValue& Right, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    string FieldName = Right.AsString();
    string FieldValue;
    try {
	TKdbcResult* Result = _ResultStack->GetLastResult();
	if (Result == 0) {
	    throw TScriptException(
		"TKiscDbFieldAccessOperator::Evaluate()",
		"empty query result"
	    );
	}
	FieldValue = Result->GetString(FieldName);
    }
    catch (TKdbcException &e) {
	throw TScriptException(
	    "TKiscDbFieldAccessOperator::Evaluate()",
	    "kdbc exception: " + e.Message()
	);
    }

    return Result = TParaValue(FieldValue);
}
